<?php
// admin/modules/study_programs/process_study_program.php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__, 3));
}
require_once PROJECT_ROOT . '/includes/db_connect.php';
require_once PROJECT_ROOT . '/admin/includes/functions.php'; // Contains generate_slug, validate_csrf_token, set_flash_message, esc_html, upload_file etc.

// Function to check if the request is AJAX
function is_ajax_request_sp() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
}

$is_ajax = is_ajax_request_sp();
if ($is_ajax) {
    header('Content-Type: application/json');
}

$module_redirect_url_base = 'index.php'; // Base URL for redirection within this module

// Security checks: User authentication and role
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    if ($is_ajax) {
        http_response_code(403); // Forbidden
        echo json_encode(['success' => false, 'message' => 'Akses ditolak. Anda tidak memiliki izin untuk aksi ini.']);
        exit;
    }
    set_flash_message('danger', 'Anda tidak memiliki izin untuk aksi ini.');
    header('Location: ../../login.php'); // Redirect to login if not admin
    exit;
}

// Database connection check
if (!isset($pdo)) {
    if ($is_ajax) {
        http_response_code(503); // Service Unavailable
        echo json_encode(['success' => false, 'message' => 'Koneksi database gagal.']);
        exit;
    }
    set_flash_message('danger', 'Koneksi database gagal.');
    header('Location: ' . $module_redirect_url_base);
    exit;
}

// --- Validation Function ---
function validate_study_program_input($data, $is_edit = false, $sp_id_to_edit = null, $pdo_conn = null) {
    $errors = [];

    // Validate Faculty ID (optional, but must be numeric if provided)
    if (!empty($data['faculty_id']) && !is_numeric($data['faculty_id'])) {
        $errors['faculty_id'] = 'Fakultas tidak valid.';
    }

    // Validate Name (required, max length, unique)
    if (empty($data['name'])) {
        $errors['name'] = 'Nama Program Studi wajib diisi.';
    } elseif (strlen($data['name']) > 150) {
        $errors['name'] = 'Nama Program Studi maksimal 150 karakter.';
    } else {
        // Check for uniqueness
        $sql_check_name = "SELECT id FROM portal_study_programs WHERE name = :name";
        $params_check = [':name' => $data['name']];
        if ($is_edit && $sp_id_to_edit) {
            $sql_check_name .= " AND id != :id";
            $params_check[':id'] = $sp_id_to_edit;
        }
        $stmt_check = $pdo_conn->prepare($sql_check_name);
        $stmt_check->execute($params_check);
        if ($stmt_check->fetch()) {
            $errors['name'] = 'Nama Program Studi "' . esc_html($data['name']) . '" sudah ada.';
        }
    }

    // Validate Slug (optional, specific pattern, max length, unique)
    if (!empty($data['slug'])) {
        if (!preg_match('/^[a-z0-9]+(?:-[a-z0-9]+)*$/', $data['slug'])) {
            $errors['slug'] = 'Slug hanya boleh berisi huruf kecil, angka, dan tanda hubung (-).';
        } elseif (strlen($data['slug']) > 170) {
            $errors['slug'] = 'Slug maksimal 170 karakter.';
        } else {
            // Check for uniqueness
            $sql_check_slug = "SELECT id FROM portal_study_programs WHERE slug = :slug";
            $params_check_slug = [':slug' => $data['slug']];
            if ($is_edit && $sp_id_to_edit) {
                $sql_check_slug .= " AND id != :id";
                $params_check_slug[':id'] = $sp_id_to_edit;
            }
            $stmt_check_slug = $pdo_conn->prepare($sql_check_slug);
            $stmt_check_slug->execute($params_check_slug);
            if ($stmt_check_slug->fetch()) {
                $errors['slug'] = 'Slug "' . esc_html($data['slug']) . '" sudah digunakan.';
            }
        }
    } elseif ($is_edit && empty($data['slug']) && !empty($data['name'])) { // If editing and slug is cleared, it's an error (or auto-generate)
         // If you want to auto-generate slug when cleared on edit, handle it before validation or remove this error.
         // For now, let's assume it's an error if explicitly emptied during edit.
         // If auto-generation on empty slug during edit is desired, this logic needs adjustment.
         // $errors['slug'] = 'Slug wajib diisi saat edit jika dikosongkan, atau akan digenerate otomatis dari nama.';
    }


    // Validate Degree (max length)
    if (!empty($data['degree']) && strlen($data['degree']) > 50) {
        $errors['degree'] = 'Jenjang Pendidikan maksimal 50 karakter.';
    }

    // Validate Accreditation Status (max length)
    if (!empty($data['accreditation_status']) && strlen($data['accreditation_status']) > 50) {
        $errors['accreditation_status'] = 'Status Akreditasi maksimal 50 karakter.';
    }

    // Validate Accreditation Number (max length)
    if (!empty($data['accreditation_number']) && strlen($data['accreditation_number']) > 100) {
        $errors['accreditation_number'] = 'Nomor SK Akreditasi maksimal 100 karakter.';
    }

    // Validate Accreditation Expiry Date (valid date format)
    if (!empty($data['accreditation_expiry_date'])) {
        try {
            new DateTime($data['accreditation_expiry_date']);
        } catch (Exception $e) {
            $errors['accreditation_expiry_date'] = 'Format tanggal kadaluarsa akreditasi tidak valid.';
        }
    }

    // Validate Head of Program Name (max length)
    if (!empty($data['head_of_program_name']) && strlen($data['head_of_program_name']) > 100) {
        $errors['head_of_program_name'] = 'Nama Ketua Program Studi maksimal 100 karakter.';
    }

    // Validate Website URL (valid URL format, max length)
    if (!empty($data['website_url'])) {
        if (!filter_var($data['website_url'], FILTER_VALIDATE_URL)) {
            $errors['website_url'] = 'Format URL website prodi tidak valid.';
        } elseif (strlen($data['website_url']) > 255) {
            $errors['website_url'] = 'URL Website prodi maksimal 255 karakter.';
        }
    }

    // Validate Program Order (numeric, non-negative)
    if (!isset($data['program_order']) || !is_numeric($data['program_order']) || (int)$data['program_order'] < 0) {
        $errors['program_order'] = 'Urutan Program Studi harus berupa angka non-negatif.';
    }

    // Validate Is Active (must be 0 or 1)
    if (!isset($data['is_active']) || !in_array($data['is_active'], [0, 1, '0', '1'], true)) {
        $errors['is_active'] = 'Status Program Studi tidak valid.';
    }
    // Add other field validations as per your `portal_study_programs` table structure if needed

    return $errors;
}

// --- File Upload Handling Function ---
function handle_sp_file_upload($file_key_name, $old_file_path, $upload_subdir, $allowed_mimes, $max_size_mb = 2, &$form_data_ref, &$errors_ref) {
    // Use PROJECT_ROOT directly as it's defined globally in this script
    $uploaded_file_db_path = $old_file_path; // Default to old path

    if (isset($_FILES[$file_key_name]) && $_FILES[$file_key_name]['error'] == UPLOAD_ERR_OK) {
        $upload_dir_relative = 'uploads/' . $upload_subdir . '/'; // Relative to project root
        $upload_dir_absolute = PROJECT_ROOT . '/' . $upload_dir_relative;

        // Ensure the upload directory exists, create if not
        if (!is_dir($upload_dir_absolute)) {
            if (!mkdir($upload_dir_absolute, 0775, true)) { // 0775 for permissions, true for recursive
                $errors_ref[$file_key_name] = "Gagal membuat direktori upload: " . $upload_dir_absolute;
                // Update the corresponding field in $form_data_ref to the old path or null
                $target_key = '';
                if ($file_key_name === 'logo_file') $target_key = 'logo_url';
                elseif ($file_key_name === 'brochure_file') $target_key = 'brochure_url';
                elseif ($file_key_name === 'accreditation_certificate_file') $target_key = 'accreditation_certificate_url';
                if ($target_key) $form_data_ref[$target_key] = $old_file_path; // Keep old path if dir creation fails
                return; // Stop further processing for this file
            }
        }

        $upload_result = upload_file($_FILES[$file_key_name], $upload_dir_absolute, $allowed_mimes, ($max_size_mb * 1024 * 1024));

        if ($upload_result['success']) {
            // Delete old file if a new one is successfully uploaded and an old path exists
            if (!empty($old_file_path) && file_exists(PROJECT_ROOT . '/' . $old_file_path)) {
                @unlink(PROJECT_ROOT . '/' . $old_file_path);
            }
            $uploaded_file_db_path = $upload_dir_relative . $upload_result['filename']; // Store relative path
        } else {
            $errors_ref[$file_key_name] = $upload_result['message'];
            // Do not change $uploaded_file_db_path, it remains the $old_file_path
        }
    } elseif (isset($_FILES[$file_key_name]) && $_FILES[$file_key_name]['error'] != UPLOAD_ERR_NO_FILE) {
        // Handle other upload errors (e.g., file too large as per php.ini, partial upload)
        $errors_ref[$file_key_name] = "Error saat upload file " . esc_html($file_key_name) . ": Kode error " . $_FILES[$file_key_name]['error'];
        // Keep $uploaded_file_db_path as $old_file_path
    }

    // Update the relevant path in $form_data_ref (e.g., $form_data_ref['logo_url'])
    $target_key = '';
    if ($file_key_name === 'logo_file') $target_key = 'logo_url';
    elseif ($file_key_name === 'brochure_file') $target_key = 'brochure_url';
    elseif ($file_key_name === 'accreditation_certificate_file') $target_key = 'accreditation_certificate_url';
    // Add other file types here if necessary

    if ($target_key) {
        $form_data_ref[$target_key] = $uploaded_file_db_path;
    }
    // This function now directly modifies $form_data_ref, so returning the path is redundant if using by reference for $form_data_ref.
}


// --- Main Processing Logic (POST for Create/Edit, GET for Delete) ---

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF Token Validation
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        if ($is_ajax) {
            http_response_code(403); // Forbidden
            echo json_encode(['success' => false, 'message' => 'Kesalahan CSRF token. Mohon muat ulang halaman dan coba lagi.']);
            exit;
        }
        set_flash_message('danger', 'Kesalahan CSRF token. Mohon muat ulang halaman dan coba lagi.');
        // Determine redirect based on action to preserve form context if possible
        $action_redirect = $_POST['action'] ?? 'create';
        $id_redirect = $_POST['id'] ?? null;
        header('Location: ' . ($action_redirect === 'edit' && $id_redirect ? 'edit.php?id=' . $id_redirect : 'create.php'));
        exit;
    }

    $action = $_POST['action'] ?? null; // 'create' or 'edit'
    $sp_id = isset($_POST['id']) ? (int)$_POST['id'] : null; // Study Program ID for edit action

    // Determine redirect URL for errors to send user back to the correct form
    $redirect_on_error_url = ($action === 'edit' && $sp_id ? 'edit.php?id=' . $sp_id : 'create.php');

    // Sanitize and retrieve form data
    $form_data_input = [
        'faculty_id' => !empty($_POST['faculty_id']) ? (int)$_POST['faculty_id'] : null,
        'name' => trim($_POST['name'] ?? ''),
        'slug' => trim($_POST['slug'] ?? ''),
        'degree' => trim($_POST['degree'] ?? ''),
        'accreditation_status' => trim($_POST['accreditation_status'] ?? ''),
        'accreditation_number' => trim($_POST['accreditation_number'] ?? ''),
        'accreditation_expiry_date' => !empty($_POST['accreditation_expiry_date']) ? $_POST['accreditation_expiry_date'] : null,
        'head_of_program_name' => trim($_POST['head_of_program_name'] ?? ''),
        'description_html' => $_POST['description_html'] ?? null, // HTML content, ensure TinyMCE sanitizes or use a purifier on output if needed
        'vision_html' => $_POST['vision_html'] ?? null,
        'mission_html' => $_POST['mission_html'] ?? null,
        'website_url' => trim($_POST['website_url'] ?? ''),
        'program_order' => isset($_POST['program_order']) ? (int)$_POST['program_order'] : 0,
        'is_active' => isset($_POST['is_active']) ? (int)$_POST['is_active'] : 0,
        // Old file paths for reference when updating/deleting files
        'old_logo_url' => $_POST['old_logo_url'] ?? null,
        'old_brochure_url' => $_POST['old_brochure_url'] ?? null,
        'old_accreditation_certificate_url' => $_POST['old_accreditation_certificate_url'] ?? null,
        // These will be updated by handle_sp_file_upload
        'logo_url' => $_POST['old_logo_url'] ?? null, // Initialize with old, will be overwritten if new file uploaded
        'brochure_url' => $_POST['old_brochure_url'] ?? null,
        'accreditation_certificate_url' => $_POST['old_accreditation_certificate_url'] ?? null,
    ];

    // Auto-generate slug if empty during creation or if explicitly requested for edit
    if ($action === 'create' && empty($form_data_input['slug']) && !empty($form_data_input['name'])) {
        $form_data_input['slug'] = generate_slug($form_data_input['name']);
        // Fallback if slug generation fails (e.g., name is all symbols)
        if (empty($form_data_input['slug'])) {
            $form_data_input['slug'] = 'prodi-' . time(); // Unique fallback
        }
    } elseif ($action === 'edit' && empty($form_data_input['slug']) && !empty($form_data_input['name'])) {
        // If slug is intentionally emptied during edit, regenerate it from the name.
        // This behavior should be clarified: does user expect it to be empty or regenerated?
        // Assuming regeneration if name is present.
        $form_data_input['slug'] = generate_slug($form_data_input['name']);
         if (empty($form_data_input['slug'])) {
            $form_data_input['slug'] = 'prodi-' . $sp_id . '-' . time(); // Unique fallback for edit
        }
    }


    $errors = []; // Initialize array for all errors (validation + file upload)

    // Handle file uploads
    // Allowed MIME types for logo: JPEG, PNG, GIF, SVG. Max size: 2MB.
    handle_sp_file_upload('logo_file', $form_data_input['old_logo_url'], 'logos_prodi', ['image/jpeg', 'image/png', 'image/gif', 'image/svg+xml'], 2, $form_data_input, $errors);

    // Allowed MIME types for brochure: PDF. Max size: 5MB.
    handle_sp_file_upload('brochure_file', $form_data_input['old_brochure_url'], 'brosur_prodi', ['application/pdf'], 5, $form_data_input, $errors);

    // Allowed MIME types for accreditation certificate: PDF, JPEG, PNG. Max size: 5MB.
    handle_sp_file_upload('accreditation_certificate_file', $form_data_input['old_accreditation_certificate_url'], 'sertifikat_akreditasi_prodi', ['application/pdf', 'image/jpeg', 'image/png'], 5, $form_data_input, $errors);

    // Validate other form inputs
    $validation_errors_fields = validate_study_program_input($form_data_input, ($action === 'edit'), $sp_id, $pdo);
    $errors = array_merge($errors, $validation_errors_fields); // Combine file errors and field validation errors

    // If there are any errors, redirect back with errors and old input
    if (!empty($errors)) {
        if ($is_ajax) {
            http_response_code(422); // Unprocessable Entity (validation errors)
            echo json_encode(['success' => false, 'message' => 'Validasi gagal. Periksa kembali input Anda.', 'errors' => $errors]);
        } else {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $form_data_input; // Send back all data, including processed file paths
            set_flash_message('danger', 'Gagal menyimpan Program Studi. Terdapat kesalahan pada input.');
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    }

    // If validation passes and files handled, proceed with database operation
    try {
        $pdo->beginTransaction();

        // Prepare SQL parameters
        $sql_params = [
            ':faculty_id' => $form_data_input['faculty_id'],
            ':name' => $form_data_input['name'],
            ':slug' => $form_data_input['slug'],
            ':degree' => $form_data_input['degree'],
            ':accreditation_status' => $form_data_input['accreditation_status'],
            ':accreditation_number' => $form_data_input['accreditation_number'],
            ':accreditation_expiry_date' => $form_data_input['accreditation_expiry_date'],
            ':accreditation_certificate_url' => $form_data_input['accreditation_certificate_url'],
            ':head_of_program_name' => $form_data_input['head_of_program_name'],
            ':description_html' => $form_data_input['description_html'],
            ':vision_html' => $form_data_input['vision_html'],
            ':mission_html' => $form_data_input['mission_html'],
            ':website_url' => $form_data_input['website_url'],
            ':logo_url' => $form_data_input['logo_url'],
            ':brochure_url' => $form_data_input['brochure_url'],
            ':program_order' => $form_data_input['program_order'],
            ':is_active' => $form_data_input['is_active'],
            // Consider adding user IDs for tracking
            // ':created_by_user_id' => $_SESSION['user_id'], // For create
            // ':updated_by_user_id' => $_SESSION['user_id'], // For edit
        ];

        if ($action === 'create') {
            // Add created_at and potentially created_by_user_id
            // $sql_params[':created_by_user_id'] = $_SESSION['user_id'];
            $sql = "INSERT INTO portal_study_programs (
                        faculty_id, name, slug, degree, accreditation_status, accreditation_number,
                        accreditation_expiry_date, accreditation_certificate_url, head_of_program_name,
                        description_html, vision_html, mission_html, website_url, logo_url, brochure_url,
                        program_order, is_active, created_at, updated_at
                        -- created_by_user_id
                    ) VALUES (
                        :faculty_id, :name, :slug, :degree, :accreditation_status, :accreditation_number,
                        :accreditation_expiry_date, :accreditation_certificate_url, :head_of_program_name,
                        :description_html, :vision_html, :mission_html, :website_url, :logo_url, :brochure_url,
                        :program_order, :is_active, NOW(), NOW()
                        -- :created_by_user_id
                    )";
            $stmt = $pdo->prepare($sql);
        } elseif ($action === 'edit' && $sp_id) {
            $sql_params[':id'] = $sp_id;
            // $sql_params[':updated_by_user_id'] = $_SESSION['user_id'];
            $sql = "UPDATE portal_study_programs SET
                        faculty_id = :faculty_id, name = :name, slug = :slug, degree = :degree,
                        accreditation_status = :accreditation_status, accreditation_number = :accreditation_number,
                        accreditation_expiry_date = :accreditation_expiry_date,
                        accreditation_certificate_url = :accreditation_certificate_url,
                        head_of_program_name = :head_of_program_name, description_html = :description_html,
                        vision_html = :vision_html, mission_html = :mission_html, website_url = :website_url,
                        logo_url = :logo_url, brochure_url = :brochure_url, program_order = :program_order,
                        is_active = :is_active, updated_at = NOW()
                        -- updated_by_user_id = :updated_by_user_id
                    WHERE id = :id";
            $stmt = $pdo->prepare($sql);
        } else {
            throw new Exception("Aksi tidak valid atau ID Program Studi tidak tersedia.");
        }

        $stmt->execute($sql_params);
        $pdo->commit();

        $success_message = 'Program Studi berhasil ' . ($action === 'create' ? 'dibuat' : 'diperbarui') . '.';
        if ($is_ajax) {
            // For AJAX, include potentially updated file URLs if needed by the client-side to update previews
            $response_data = [
                'success' => true,
                'message' => $success_message,
                'redirect_url' => $module_redirect_url_base
            ];
            // If client needs to update specific fields (like new file URLs if names changed on server)
            // $response_data['updated_data'] = [
            //     'logo_url' => $form_data_input['logo_url'],
            //     'brochure_url' => $form_data_input['brochure_url'],
            //     'accreditation_certificate_url' => $form_data_input['accreditation_certificate_url'],
            //     'slug' => $form_data_input['slug'] // If slug was auto-generated
            // ];
            echo json_encode($response_data);
        } else {
            set_flash_message('success', $success_message);
            header('Location: ' . $module_redirect_url_base);
        }
        exit;

    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Database Error: " . $e->getMessage() . " for action: " . $action); // Log the detailed error
        $error_message_user = 'Terjadi kesalahan pada database saat mencoba menyimpan Program Studi. ';
        // if (strpos($e->getMessage(), 'Duplicate entry') !== false) {
        //     $error_message_user .= 'Kemungkinan ada data yang duplikat (misalnya Nama atau Slug).';
        // }

        if ($is_ajax) {
            http_response_code(500); // Internal Server Error
            echo json_encode(['success' => false, 'message' => $error_message_user . 'Silakan coba lagi.']);
        } else {
            $_SESSION['form_errors'] = $errors; // Resend original errors if any, or a generic one
            $_SESSION['form_data'] = $form_data_input;
            set_flash_message('danger', $error_message_user . 'Silakan coba lagi. Error: ' . $e->getCode());
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    } catch (Exception $e) {
        // General exception handling (e.g., "Aksi tidak valid")
        error_log("General Error: " . $e->getMessage());
        if ($is_ajax) {
            http_response_code(400); // Bad Request
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        } else {
            set_flash_message('danger', $e->getMessage());
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    }

} elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'delete') {
    // --- Delete Action ---
    if (!isset($_GET['csrf_token']) || !validate_csrf_token($_GET['csrf_token'])) {
        if ($is_ajax) { http_response_code(403); echo json_encode(['success' => false, 'message' => 'Kesalahan CSRF.']); exit; }
        set_flash_message('danger', 'Kesalahan CSRF token.');
        header('Location: ' . $module_redirect_url_base); exit;
    }

    $sp_id_to_delete = isset($_GET['id']) ? (int)$_GET['id'] : 0;

    if ($sp_id_to_delete <= 0) {
        if ($is_ajax) { http_response_code(400); echo json_encode(['success' => false, 'message' => 'ID Program Studi tidak valid.']); exit; }
        set_flash_message('danger', 'ID Program Studi tidak valid.');
        header('Location: ' . $module_redirect_url_base); exit;
    }

    try {
        $pdo->beginTransaction();

        // First, get file paths to delete them from the server
        $stmt_get_files = $pdo->prepare("SELECT logo_url, brochure_url, accreditation_certificate_url FROM portal_study_programs WHERE id = :id");
        $stmt_get_files->execute([':id' => $sp_id_to_delete]);
        $files_to_delete_paths = $stmt_get_files->fetch(PDO::FETCH_ASSOC);

        // Then, delete the record from the database
        $stmt_delete = $pdo->prepare("DELETE FROM portal_study_programs WHERE id = :id");
        $stmt_delete->execute([':id' => $sp_id_to_delete]);

        if ($stmt_delete->rowCount() > 0) {
            // If record deleted successfully, delete associated files
            if ($files_to_delete_paths) {
                $file_paths_to_check = [
                    $files_to_delete_paths['logo_url'],
                    $files_to_delete_paths['brochure_url'],
                    $files_to_delete_paths['accreditation_certificate_url']
                ];
                foreach ($file_paths_to_check as $relative_path) {
                    if (!empty($relative_path) && file_exists(PROJECT_ROOT . '/' . $relative_path)) {
                        @unlink(PROJECT_ROOT . '/' . $relative_path);
                    }
                }
            }
            $pdo->commit();
            if ($is_ajax) { echo json_encode(['success' => true, 'message' => 'Program Studi berhasil dihapus.']); }
            else { set_flash_message('success', 'Program Studi berhasil dihapus.'); header('Location: ' . $module_redirect_url_base); }
        } else {
            $pdo->rollBack(); // Nothing was deleted
            if ($is_ajax) { http_response_code(404); echo json_encode(['success' => false, 'message' => 'Program Studi tidak ditemukan atau sudah dihapus.']); }
            else { set_flash_message('warning', 'Program Studi tidak ditemukan atau sudah dihapus.'); header('Location: ' . $module_redirect_url_base); }
        }
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Database Error on Delete: " . $e->getMessage());
        // Check for foreign key constraint violation if study programs are linked elsewhere
        $user_message = 'Gagal menghapus Program Studi karena kesalahan database.';
        if ($e->getCode() == '23000') { // Integrity constraint violation
            $user_message = 'Gagal menghapus Program Studi. Kemungkinan masih terhubung dengan data lain (misalnya, mahasiswa, mata kuliah).';
        }
        if ($is_ajax) { http_response_code(500); echo json_encode(['success' => false, 'message' => $user_message]); }
        else { set_flash_message('danger', $user_message); header('Location: ' . $module_redirect_url_base); }
    }
    exit;

} else {
    // Handle invalid request method or missing action
    if ($is_ajax) {
        http_response_code(405); // Method Not Allowed
        echo json_encode(['success' => false, 'message' => 'Metode permintaan tidak valid.']);
    } else {
        set_flash_message('danger', 'Permintaan tidak valid.');
        header('Location: ' . $module_redirect_url_base);
    }
    exit;
}
?>