<?php
// public/contact.php
if (session_status() == PHP_SESSION_NONE) {
    // session_start(); // config.php will handle this
}

// Define PROJECT_ROOT if not already defined (e.g., in a central config.php)
if (!defined('PROJECT_ROOT')) {
    // This assumes contact.php is in the 'public' directory directly under the project root
    define('PROJECT_ROOT', dirname(__DIR__));
}

// Ensure critical files are loaded first
// 1. Database connection (which should load config.php)
$db_connect_path = PROJECT_ROOT . '/includes/db_connect.php';
if (file_exists($db_connect_path)) {
    require_once $db_connect_path; // This defines $pdo and loads config.php (PROJECT_ROOT, BASE_URL)
} else {
    error_log("FATAL ERROR: Main db_connect.php not found at: " . $db_connect_path . " from " . __FILE__);
    die("Terjadi kesalahan konfigurasi sistem. Harap hubungi administrator. (Error Code: DC01)");
}

// 2. Admin functions (for CSRF, helpers, etc.)
$functions_path = PROJECT_ROOT . '/admin/includes/functions.php';
if (file_exists($functions_path)) {
    require_once $functions_path; // This defines generate_csrf_token, esc_html, etc.
} else {
    error_log("FATAL ERROR: Admin functions file not found at: " . $functions_path . " from " . __FILE__);
    die("Terjadi kesalahan konfigurasi sistem. Harap hubungi administrator. (Error Code: AF01)");
}


$site_title = "Hubungi Kami";
$meta_description = "Kirimkan pesan, pertanyaan, atau saran Anda kepada Universitas Graha Karya Muara Bulian.";

// Initialize form data for repopulation
$form_data = $_SESSION['form_data'] ?? [
    'sender_name' => '', 'sender_email' => '', 'sender_phone' => '',
    'subject' => '', 'message_content' => ''
];
$form_errors = $_SESSION['form_errors'] ?? [];
clear_form_session_data(); // Clear after use


// Fetch settings for contact info and site name
$settings = []; // $settings will be populated by header_public.php if this is not done here
if (isset($pdo)) { // Check if $pdo is available
    try {
        $stmt_settings = $pdo->query("SELECT setting_key, setting_value FROM portal_settings WHERE is_public = 1");
        while ($row = $stmt_settings->fetch()) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
        if (!empty($settings['site_name'])) {
            $site_title = "Hubungi Kami - " . esc_html($settings['site_name']);
        }
        if (!empty($settings['contact_page_meta_description'])) {
            $meta_description = esc_html($settings['contact_page_meta_description']);
        }
    } catch (\PDOException $e) {
        error_log("Error fetching settings for contact page: " . $e->getMessage());
    }
}


if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        set_flash_message('danger', 'Kesalahan validasi keamanan (CSRF). Silakan coba lagi.');
        $_SESSION['form_data'] = $_POST; // Persist form data
        header('Location: contact.php');
        exit;
    }

    $form_data['sender_name'] = trim($_POST['sender_name'] ?? '');
    $form_data['sender_email'] = trim($_POST['sender_email'] ?? '');
    $form_data['sender_phone'] = trim($_POST['sender_phone'] ?? '');
    $form_data['subject'] = trim($_POST['subject'] ?? '');
    $form_data['message_content'] = trim($_POST['message_content'] ?? '');

    if (empty($form_data['sender_name'])) {
        $form_errors['sender_name'] = 'Nama lengkap wajib diisi.';
    } elseif (strlen($form_data['sender_name']) > 100) {
        $form_errors['sender_name'] = 'Nama lengkap maksimal 100 karakter.';
    }

    if (empty($form_data['sender_email'])) {
        $form_errors['sender_email'] = 'Email wajib diisi.';
    } elseif (!filter_var($form_data['sender_email'], FILTER_VALIDATE_EMAIL)) {
        $form_errors['sender_email'] = 'Format email tidak valid.';
    } elseif (strlen($form_data['sender_email']) > 100) {
        $form_errors['sender_email'] = 'Email maksimal 100 karakter.';
    }

    if (!empty($form_data['sender_phone']) && !preg_match('/^[0-9\-\+\s\(\)]{7,20}$/', $form_data['sender_phone'])) {
        $form_errors['sender_phone'] = 'Format nomor telepon tidak valid.';
    } elseif (!empty($form_data['sender_phone']) && strlen($form_data['sender_phone']) > 30) { // Max length check based on DB
        $form_errors['sender_phone'] = 'Nomor telepon maksimal 30 karakter.';
    }


    if (empty($form_data['subject'])) {
        $form_errors['subject'] = 'Subjek wajib diisi.';
    } elseif (strlen($form_data['subject']) > 200) {
        $form_errors['subject'] = 'Subjek maksimal 200 karakter.';
    }

    if (empty($form_data['message_content'])) {
        $form_errors['message_content'] = 'Isi pesan wajib diisi.';
    } elseif (strlen($form_data['message_content']) > 5000) { // Assuming TEXT allows more, but good to have a limit. Adjust if needed.
        $form_errors['message_content'] = 'Isi pesan maksimal 5000 karakter.';
    }

    if (empty($form_errors)) {
        if (isset($pdo)) {
            try {
                // MODIFIED INSERT STATEMENT:
                // Removed updated_at as it's not in the new schema.
                // created_at has a DEFAULT current_timestamp() in the DB,
                // so we don't need to explicitly set it here either, the DB will handle it.
                $stmt_insert = $pdo->prepare("INSERT INTO portal_contact_messages
                                            (sender_name, sender_email, sender_phone, subject, message_content, ip_address, user_agent)
                                            VALUES (:name, :email, :phone, :subject, :message, :ip, :ua)");

                $ip_address = $_SERVER['REMOTE_ADDR'] ?? null;
                $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? null;

                $stmt_insert->bindParam(':name', $form_data['sender_name']);
                $stmt_insert->bindParam(':email', $form_data['sender_email']);
                $stmt_insert->bindParam(':phone', $form_data['sender_phone']); // Bind NULL if empty
                $stmt_insert->bindParam(':subject', $form_data['subject']);
                $stmt_insert->bindParam(':message', $form_data['message_content']);
                $stmt_insert->bindParam(':ip', $ip_address);
                $stmt_insert->bindParam(':ua', $user_agent);

                $stmt_insert->execute();
                set_flash_message('success', 'Pesan Anda telah berhasil terkirim. Terima kasih!');

                $recipient_email = $settings['contact_form_recipient_email'] ?? null;
                $site_name_for_email = $settings['site_name'] ?? 'UGKMB Website';

                if ($recipient_email && filter_var($recipient_email, FILTER_VALIDATE_EMAIL)) {
                    $email_subject_to_admin = "Pesan Baru dari Form Kontak (" . $site_name_for_email . "): " . $form_data['subject'];
                    $email_body_to_admin = "Anda menerima pesan baru dari formulir kontak website:\n\n" .
                                 "Nama: " . $form_data['sender_name'] . "\n" .
                                 "Email: " . $form_data['sender_email'] . "\n" .
                                 "Telepon: " . ($form_data['sender_phone'] ?: '-') . "\n" .
                                 "Subjek: " . $form_data['subject'] . "\n\n" .
                                 "Pesan:\n" . $form_data['message_content'] . "\n\n" .
                                 "IP Address: " . $ip_address . "\n" .
                                 "User Agent: " . $user_agent;
                    $base_url_host = defined('BASE_URL') ? parse_url(BASE_URL, PHP_URL_HOST) : ($_SERVER['SERVER_NAME'] ?? 'localhost');
                    $from_email_domain = str_replace('www.', '', $base_url_host);
                    $headers = "From: noreply@" . $from_email_domain . "\r\n" .
                               "Reply-To: " . $form_data['sender_email'] . "\r\n" .
                               "X-Mailer: PHP/" . phpversion();

                    // Mail sending (ensure mail server is configured)
                    // if (!@mail($recipient_email, $email_subject_to_admin, $email_body_to_admin, $headers)) {
                    //     error_log("Contact form email FAILED to send to: " . $recipient_email);
                    // } else {
                    //     error_log("Contact form email notification prepared for: " . $recipient_email);
                    // }
                }

                $_SESSION['form_data'] = [ // Clear form data on success
                    'sender_name' => '', 'sender_email' => '', 'sender_phone' => '',
                    'subject' => '', 'message_content' => ''
                ];
                header('Location: contact.php');
                exit;

            } catch (\PDOException $e) {
                set_flash_message('danger', 'Terjadi kesalahan saat mengirim pesan. Silakan coba lagi nanti.');
                error_log("Contact form submission DB error: " . $e->getMessage());
                $_SESSION['form_data'] = $form_data; // Persist form data on DB error
                $_SESSION['form_errors'] = $form_errors;
                header('Location: contact.php');
                exit;
            }
        } else {
            set_flash_message('danger', 'Koneksi database tidak tersedia.');
            $_SESSION['form_data'] = $form_data; // Persist form data
            $_SESSION['form_errors'] = $form_errors;
            header('Location: contact.php');
            exit;
        }
    } else {
        set_flash_message('danger', 'Harap perbaiki kesalahan pada formulir.');
        $_SESSION['form_data'] = $form_data; // Persist form data with errors
        $_SESSION['form_errors'] = $form_errors;
        header('Location: contact.php');
        exit;
    }
}

$theme = $_COOKIE['theme'] ?? 'light-theme';

// Include header AFTER all header modifications and potential redirects
require_once PROJECT_ROOT . '/public/includes/header_public.php';
?>

<main class="container py-5">
    <div class="row">
        <div class="col-lg-10 mx-auto">
            <header class="mb-5 text-center">
                <h1 class="display-4 fw-bold">Hubungi Kami</h1>
                <p class="lead text-muted"><?php echo esc_html($meta_description); ?></p>
            </header>

            <?php echo display_flash_message(); ?>

            <div class="row">
                <div class="col-lg-7 mb-4 mb-lg-0">
                    <div class="card shadow-sm">
                        <div class="card-body p-4 p-md-5">
                            <h3 class="card-title mb-4"><i class="fas fa-paper-plane me-2"></i>Kirim Pesan</h3>
                            <form action="contact.php" method="POST" id="contactForm">
                                <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                                <div class="mb-3">
                                    <label for="sender_name" class="form-label">Nama Lengkap <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control <?php echo isset($form_errors['sender_name']) ? 'is-invalid' : ''; ?>" id="sender_name" name="sender_name" value="<?php echo old_form_data('sender_name', '', $form_data); ?>" required>
                                    <?php echo get_form_error('sender_name', $form_errors); ?>
                                </div>
                                <div class="mb-3">
                                    <label for="sender_email" class="form-label">Email <span class="text-danger">*</span></label>
                                    <input type="email" class="form-control <?php echo isset($form_errors['sender_email']) ? 'is-invalid' : ''; ?>" id="sender_email" name="sender_email" value="<?php echo old_form_data('sender_email', '', $form_data); ?>" required>
                                    <?php echo get_form_error('sender_email', $form_errors); ?>
                                </div>
                                <div class="mb-3">
                                    <label for="sender_phone" class="form-label">Nomor Telepon (Opsional)</label>
                                    <input type="tel" class="form-control <?php echo isset($form_errors['sender_phone']) ? 'is-invalid' : ''; ?>" id="sender_phone" name="sender_phone" value="<?php echo old_form_data('sender_phone', '', $form_data); ?>">
                                    <?php echo get_form_error('sender_phone', $form_errors); ?>
                                </div>
                                <div class="mb-3">
                                    <label for="subject" class="form-label">Subjek <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control <?php echo isset($form_errors['subject']) ? 'is-invalid' : ''; ?>" id="subject" name="subject" value="<?php echo old_form_data('subject', '', $form_data); ?>" required>
                                    <?php echo get_form_error('subject', $form_errors); ?>
                                </div>
                                <div class="mb-3">
                                    <label for="message_content" class="form-label">Pesan Anda <span class="text-danger">*</span></label>
                                    <textarea class="form-control <?php echo isset($form_errors['message_content']) ? 'is-invalid' : ''; ?>" id="message_content" name="message_content" rows="5" required><?php echo old_form_data('message_content', '', $form_data); ?></textarea>
                                    <?php echo get_form_error('message_content', $form_errors); ?>
                                </div>
                                <button type="submit" class="btn btn-primary btn-lg w-100"><i class="fas fa-envelope me-2"></i>Kirim Pesan</button>
                            </form>
                        </div>
                    </div>
                </div>
                <div class="col-lg-5">
                    <div class="card shadow-sm mb-4">
                         <div class="card-body p-4">
                            <h3 class="card-title mb-3"><i class="fas fa-info-circle me-2"></i>Informasi Kontak</h3>
                            <p><i class="fas fa-map-marker-alt fa-fw me-2 text-primary"></i>
                                <?php echo nl2br(esc_html($settings['address'] ?? 'Alamat belum diatur.')); ?>
                            </p>
                            <p><i class="fas fa-phone fa-fw me-2 text-primary"></i>
                                <?php echo esc_html($settings['contact_phone'] ?? 'Telepon belum diatur.'); ?>
                            </p>
                            <?php if (!empty($settings['contact_whatsapp_number'])): ?>
                            <p><i class="fab fa-whatsapp fa-fw me-2 text-success"></i>
                                <a href="https://wa.me/<?php echo preg_replace('/[^0-9]/', '', esc_html($settings['contact_whatsapp_number'])); ?>" target="_blank" rel="noopener noreferrer">
                                    <?php echo esc_html($settings['contact_whatsapp_number']); ?>
                                </a>
                            </p>
                            <?php endif; ?>
                            <p><i class="fas fa-envelope fa-fw me-2 text-primary"></i>
                                <a href="mailto:<?php echo esc_html($settings['contact_email'] ?? 'email@belum.diatur'); ?>">
                                    <?php echo esc_html($settings['contact_email'] ?? 'email@belum.diatur'); ?>
                                </a>
                            </p>
                        </div>
                    </div>
                                <?php if (!empty($settings['maps_embed_url'])): ?>
                    <div class="card shadow-sm">
                        <div class="card-body p-0">
                             <h3 class="card-title my-3 text-center"><i class="fas fa-map-marked-alt me-2"></i>Peta Lokasi</h3>
                            <div class="ratio ratio-16x9">
                                <?php
                                $maps_url_contact = $settings['maps_embed_url'];
                                $iframe_src_contact = '';
                                if (strpos(strtolower($maps_url_contact), '<iframe') !== false) {
                                    preg_match('/src="([^"]+)"/', $maps_url_contact, $matches_contact);
                                    $iframe_src_contact = $matches_contact[1] ?? '';
                                } elseif (filter_var($maps_url_contact, FILTER_VALIDATE_URL)) {
                                    $iframe_src_contact = $maps_url_contact; // Assume direct embeddable URL
                                }

                                if (!empty($iframe_src_contact) && filter_var($iframe_src_contact, FILTER_VALIDATE_URL) && 
                                    (strpos($iframe_src_contact, 'google.com/maps/embed') !== false || strpos($iframe_src_contact, 'google.com/maps?q=') !== false  || strpos($iframe_src_contact, 'google.com/maps/place') !== false || strpos($iframe_src_contact, 'googleusercontent.com/maps') !== false)) {
                                    // Ensure it's an embed URL for iframes
                                    if (strpos($iframe_src_contact, 'google.com/maps?q=') !== false || strpos($iframe_src_contact, 'google.com/maps/place') !== false) {
                                        // Convert view URL to embed URL if possible (basic attempt)
                                        // This is a simplified conversion; a proper one is more complex.
                                        // For now, we just ensure it has /embed/ if it's a direct Google Maps URL for iframe.
                                        // If it's already an embed URL, it's fine.
                                        // If it's a share URL like goo.gl/maps/, that might not work directly in iframe src without resolution.
                                        // Best practice is to get the actual "Embed a map" iframe code from Google Maps.
                                        // This example will just use it if it looks like an embed or a query.
                                    }
                                    echo '<iframe src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3988.085587424482!2d103.2775825!3d-1.6886614999999998!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x2e2f713613db00af%3A0x5454d5564ed940bf!2sUniversitas%20Graha%20Karya%20Muara%20Bulian!5e0!3m2!1sid!2sid!4v1747924807915!5m2!1sid!2sid" width="600" height="450" style="border:0;" allowfullscreen="" loading="lazy" referrerpolicy="no-referrer-when-downgrade"></iframe>';
                                } else {
                                    echo '<p class="p-3 text-danger text-center">Format URL Peta tidak valid atau tidak dapat di-embed.</p>';
                                }
                                ?>
                            </div>
                        </div>
                    </div>
                    <?php else: ?>
                        <p class="text-center text-muted mt-3">Peta lokasi belum diatur.</p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</main>

<?php
require_once PROJECT_ROOT . '/includes/footer_public.php';
?>