<?php
// public/category.php (Listing articles by category)
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
$project_root = dirname(__DIR__);
require_once $project_root . '/includes/db_connect.php';
require_once $project_root . '/admin/includes/functions.php'; // For resolve_url, esc_html

$category_slug = $_GET['slug'] ?? null;
$category_data = null;
$articles = [];
$total_articles = 0;
$limit = 9; // Articles per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

$site_title = "Kategori";
$meta_description = "Artikel berdasarkan kategori.";

if (empty($category_slug)) {
    set_flash_message('danger', 'Kategori tidak ditemukan (slug kosong).');
    header("Location: " . resolve_url('articles.php')); // Redirect to all articles
    exit;
}

if (isset($pdo)) {
    try {
        // Fetch category details
        $stmt_cat = $pdo->prepare("SELECT id, name, description, slug FROM portal_categories WHERE slug = :slug LIMIT 1");
        $stmt_cat->bindParam(':slug', $category_slug);
        $stmt_cat->execute();
        $category_data = $stmt_cat->fetch();

        if ($category_data) {
            $site_title = "Kategori: " . esc_html($category_data['name']);
            $meta_description = !empty($category_data['description']) ? esc_html($category_data['description']) : "Artikel dalam kategori " . esc_html($category_data['name']);

            // Count total articles in this category for pagination
            $stmt_count = $pdo->prepare("SELECT COUNT(*) FROM portal_articles 
                                         WHERE category_id = :category_id AND status = 'published' AND published_at <= NOW()");
            $stmt_count->bindParam(':category_id', $category_data['id'], PDO::PARAM_INT);
            $stmt_count->execute();
            $total_articles = $stmt_count->fetchColumn();

            // Fetch articles for the current page in this category
            $stmt_articles = $pdo->prepare("SELECT a.title, a.slug, a.excerpt, a.featured_image_url, a.published_at, c.name as category_name, c.slug as category_slug
                                           FROM portal_articles a
                                           LEFT JOIN portal_categories c ON a.category_id = c.id
                                           WHERE a.category_id = :category_id AND a.status = 'published' AND a.published_at <= NOW()
                                           ORDER BY a.published_at DESC
                                           LIMIT :limit OFFSET :offset");
            $stmt_articles->bindParam(':category_id', $category_data['id'], PDO::PARAM_INT);
            $stmt_articles->bindParam(':limit', $limit, PDO::PARAM_INT);
            $stmt_articles->bindParam(':offset', $offset, PDO::PARAM_INT);
            $stmt_articles->execute();
            $articles = $stmt_articles->fetchAll();
        } else {
            http_response_code(404);
            $site_title = "Kategori Tidak Ditemukan";
            $meta_description = "Kategori yang Anda cari tidak ditemukan.";
        }
    } catch (\PDOException $e) {
        error_log("Error fetching category page: " . $e->getMessage());
        // Set a user-friendly error message
    }
}

// Fetch global settings for header/footer
$settings = []; // Will be populated by header_public.php
$theme = $_COOKIE['theme'] ?? 'light-theme'; // For body class

// Include header
require_once $project_root . '/includes/header_public.php';
?>

<main class="container py-5">
    <div class="row">
        <div class="col-lg-12">
            <header class="mb-5 text-center">
                <?php if ($category_data): ?>
                    <h1 class="display-4 fw-bold">Kategori: <?php echo esc_html($category_data['name']); ?></h1>
                    <?php if (!empty($category_data['description'])): ?>
                        <p class="lead text-muted"><?php echo esc_html($category_data['description']); ?></p>
                    <?php endif; ?>
                <?php else: ?>
                    <h1 class="display-4 fw-bold"><?php echo $site_title; ?></h1>
                    <p class="lead text-muted"><?php echo $meta_description; ?></p>
                <?php endif; ?>
            </header>

            <?php if (isset($_SESSION['flash_message'])): ?>
                <div class="alert alert-<?php echo $_SESSION['flash_message']['type']; ?> alert-dismissible fade show" role="alert">
                    <?php echo $_SESSION['flash_message']['text']; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['flash_message']); ?>
            <?php endif; ?>

            <?php if (!empty($articles)): ?>
                <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
                    <?php foreach ($articles as $article): ?>
                    <div class="col">
                        <div class="card h-100 shadow-sm border-0 article-card-hover">
                            <?php if (!empty($article['featured_image_url'])): ?>
                            <a href="<?php echo resolve_url('article.php?slug=' . esc_html($article['slug'])); ?>">
                                <img src="<?php echo resolve_url(esc_html($article['featured_image_url'])); ?>" class="card-img-top" alt="<?php echo esc_html($article['title']); ?>" style="height: 200px; object-fit: cover;">
                            </a>
                            <?php endif; ?>
                            <div class="card-body d-flex flex-column">
                                <h5 class="card-title mb-2"><a href="<?php echo resolve_url('article.php?slug=' . esc_html($article['slug'])); ?>" class="text-decoration-none stretched-link article-title-link"><?php echo esc_html($article['title']); ?></a></h5>
                                <p class="card-text small text-muted mb-2"><i class="fas fa-calendar-alt me-1"></i><?php echo date('d F Y', strtotime($article['published_at'])); ?></p>
                                <p class="card-text flex-grow-1 mb-3 article-excerpt"><?php echo esc_html(substr(strip_tags($article['excerpt'] ?? ''), 0, 120)) . (strlen(strip_tags($article['excerpt'] ?? '')) > 120 ? '...' : ''); ?></p>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>

                <?php if ($total_articles > $limit): ?>
                <nav aria-label="Page navigation" class="mt-5">
                    <ul class="pagination justify-content-center">
                        <?php
                        $total_pages = ceil($total_articles / $limit);
                        $prev_disabled = ($page <= 1) ? "disabled" : "";
                        $next_disabled = ($page >= $total_pages) ? "disabled" : "";
                        ?>
                        <li class="page-item <?php echo $prev_disabled; ?>">
                            <a class="page-link" href="?slug=<?php echo esc_html($category_slug); ?>&page=<?php echo $page - 1; ?>" aria-label="Previous">
                                <span aria-hidden="true">&laquo;</span>
                            </a>
                        </li>
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?php echo ($page == $i) ? "active" : ""; ?>"><a class="page-link" href="?slug=<?php echo esc_html($category_slug); ?>&page=<?php echo $i; ?>"><?php echo $i; ?></a></li>
                        <?php endfor; ?>
                        <li class="page-item <?php echo $next_disabled; ?>">
                            <a class="page-link" href="?slug=<?php echo esc_html($category_slug); ?>&page=<?php echo $page + 1; ?>" aria-label="Next">
                                <span aria-hidden="true">&raquo;</span>
                            </a>
                        </li>
                    </ul>
                </nav>
                <?php endif; ?>

            <?php elseif ($category_data): // Category exists but no articles ?>
                <div class="alert alert-info text-center" role="alert">
                    Tidak ada artikel yang ditemukan dalam kategori "<?php echo esc_html($category_data['name']); ?>".
                </div>
            <?php endif; // No need for an else if category_data is false, as it's handled in the header section ?>
             <div class="mt-4 text-center">
                <a href="<?php echo resolve_url('articles.php'); ?>" class="btn btn-outline-secondary">Lihat Semua Artikel</a>
            </div>
        </div>
    </div>
</main>

<?php
require_once $project_root . '/includes/footer_public.php';
?>
