<?php
// public/articles.php (Listing all articles & WP Content)
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
$project_root = dirname(__DIR__);
require_once $project_root . '/includes/db_connect.php';
require_once $project_root . '/admin/includes/functions.php'; // For resolve_url, esc_html

$site_title = "Arsip & Konten Universitas";
$meta_description = "Kumpulan berita, informasi, halaman, dan konten lainnya dari UGKMB.";

// ======================================================================
// BAGIAN INI TIDAK DIUBAH - TETAP MENGAMBIL DATA DARI DATABASE LOKAL
// ======================================================================
$articles = [];
$total_articles = 0;
$limit = 9; // Articles per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

if (isset($pdo)) {
    try {
        // Count total articles for pagination
        $stmt_count = $pdo->query("SELECT COUNT(*) FROM portal_articles WHERE status = 'published' AND published_at <= NOW()");
        $total_articles = $stmt_count->fetchColumn();

        // Fetch articles for the current page
        $stmt = $pdo->prepare("SELECT a.title, a.slug, a.excerpt, a.featured_image_url, a.published_at, c.name as category_name, c.slug as category_slug
                                FROM portal_articles a
                                LEFT JOIN portal_categories c ON a.category_id = c.id
                                WHERE a.status = 'published' AND a.published_at <= NOW()
                                ORDER BY a.published_at DESC
                                LIMIT :limit OFFSET :offset");
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $articles = $stmt->fetchAll();
    } catch (\PDOException $e) {
        error_log("Error fetching articles list: " . $e->getMessage());
    }
}

// Fetch global settings for header/footer
$settings = []; // Will be populated by header_public.php
$theme = $_COOKIE['theme'] ?? 'light-theme'; // For body class

// Include header
require_once $project_root . '/includes/header_public.php';
?>

<main class="container py-5">
    <div class="row">
        <div class="col-lg-12">
            <header class="mb-5 text-center">
                <h1 class="display-4 fw-bold"><?php echo esc_html($site_title); ?></h1>
                <p class="lead text-muted"><?php echo esc_html($meta_description); ?></p>
            </header>

            <h2 class="h3 fw-bold custom-page-title">Berita dari Portal Internal</h2>
            <hr class="mt-1 mb-4">
            <?php if (!empty($articles)): ?>
                <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
                    <?php foreach ($articles as $article): ?>
                    <div class="col">
                        <div class="card h-100 shadow-sm border-0 article-card-hover">
                            <?php if (!empty($article['featured_image_url'])): ?>
                            <a href="<?php echo resolve_url('article.php?slug=' . esc_html($article['slug'])); ?>">
                                <img src="<?php echo resolve_url(esc_html($article['featured_image_url'])); ?>" class="card-img-top" alt="<?php echo esc_html($article['title']); ?>" style="height: 200px; object-fit: cover;">
                            </a>
                            <?php endif; ?>
                            <div class="card-body d-flex flex-column">
                                <?php if (!empty($article['category_name'])): ?>
                                    <a href="<?php echo resolve_url('category.php?slug=' . esc_html($article['category_slug'])); ?>" class="text-muted text-decoration-none small mb-1 text-uppercase fw-bold article-category"><?php echo esc_html($article['category_name']); ?></a>
                                <?php endif; ?>
                                <h5 class="card-title mb-2"><a href="<?php echo resolve_url('article.php?slug=' . esc_html($article['slug'])); ?>" class="text-decoration-none stretched-link article-title-link"><?php echo esc_html($article['title']); ?></a></h5>
                                <p class="card-text small text-muted mb-2"><i class="fas fa-calendar-alt me-1"></i><?php echo date('d F Y', strtotime($article['published_at'])); ?></p>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>

                <?php if ($total_articles > $limit): ?>
                <nav aria-label="Page navigation" class="mt-5">
                    <ul class="pagination justify-content-center">
                        <?php
                        $total_pages = ceil($total_articles / $limit);
                        $prev_disabled = ($page <= 1) ? "disabled" : "";
                        $next_disabled = ($page >= $total_pages) ? "disabled" : "";
                        ?>
                        <li class="page-item <?php echo $prev_disabled; ?>">
                            <a class="page-link" href="?page=<?php echo $page - 1; ?>" aria-label="Previous"><span aria-hidden="true">&laquo;</span></a>
                        </li>
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?php echo ($page == $i) ? "active" : ""; ?>"><a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a></li>
                        <?php endfor; ?>
                        <li class="page-item <?php echo $next_disabled; ?>">
                            <a class="page-link" href="?page=<?php echo $page + 1; ?>" aria-label="Next"><span aria-hidden="true">&raquo;</span></a>
                        </li>
                    </ul>
                </nav>
                <?php endif; ?>
            <?php else: ?>
                <div class="alert alert-info text-center" role="alert">
                    Tidak ada berita atau artikel yang tersedia saat ini dari portal internal.
                </div>
            <?php endif; ?>

            <hr class="my-5">

            <div id="wordpress-section">
                <header class="mb-5 text-center">
                    <h2 class="display-5 fw-bold">Konten dari Situs Utama</h2>
                    <p class="lead text-muted">Informasi, halaman, media, dan kategori yang diambil langsung dari situs WordPress kami.</p>
                </header>

                <section id="wp-posts-section" class="mb-5">
                    <h3 class="h4 fw-bold custom-page-title">Berita Terbaru dari WordPress</h3>
                    <div id="wordpress-posts-container" class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4 mt-2">
                        </div>
                </section>
                
                <section id="wp-pages-section" class="mb-5">
                    <h3 class="h4 fw-bold custom-page-title">Halaman Informasi</h3>
                    <div id="wordpress-pages-container" class="list-group mt-2">
                        </div>
                </section>

                <div class="row">
                    <div class="col-lg-8">
                        <section id="wp-media-section" class="mb-5">
                            <h3 class="h4 fw-bold custom-page-title">Galeri Media Terbaru</h3>
                            <div id="wordpress-media-container" class="row row-cols-2 row-cols-md-4 g-3 mt-2">
                                </div>
                        </section>
                    </div>
                    <div class="col-lg-4">
                        <section id="wp-categories-section" class="mb-5">
                             <h3 class="h4 fw-bold custom-page-title">Kategori Postingan</h3>
                            <ul id="wordpress-categories-container" class="list-group mt-2">
                                </ul>
                        </section>
                    </div>
                </div>

                <div id="wp-global-loading-indicator" class="text-center py-5">
                    <div class="spinner-border text-primary" style="width: 3rem; height: 3rem;" role="status">
                        <span class="visually-hidden">Memuat...</span>
                    </div>
                    <p class="mt-3">Mengambil semua data dari WordPress...</p>
                </div>
            </div>
        </div>
    </div>
</main>

<script>
document.addEventListener('DOMContentLoaded', function() {
    
    const baseApiUrl = 'https://wp.ugkmb.ac.id/wp-json/wp/v2';
    const endpoints = {
        posts: `${baseApiUrl}/posts?per_page=6&_embed`,
        pages: `${baseApiUrl}/pages?per_page=5`,
        media: `${baseApiUrl}/media?per_page=8&media_type=image`,
        categories: `${baseApiUrl}/categories?per_page=10&hide_empty=true`
    };

    const containers = {
        posts: document.getElementById('wordpress-posts-container'),
        pages: document.getElementById('wordpress-pages-container'),
        media: document.getElementById('wordpress-media-container'),
        categories: document.getElementById('wordpress-categories-container'),
    };
    
    const sections = {
        posts: document.getElementById('wp-posts-section'),
        pages: document.getElementById('wp-pages-section'),
        media: document.getElementById('wp-media-section'),
        categories: document.getElementById('wp-categories-section'),
    };

    const loadingIndicator = document.getElementById('wp-global-loading-indicator');

    // Sembunyikan semua section pada awalnya
    Object.values(sections).forEach(sec => sec.style.display = 'none');

    const renderPosts = (posts) => {
        if (!posts || posts.length === 0) return;
        sections.posts.style.display = 'block';
        let html = '';
        posts.forEach(post => {
            const featuredImage = post._embedded?.['wp:featuredmedia']?.[0]?.source_url || '<?php echo resolve_url("assets/images/default-placeholder.png"); ?>';
            const postDate = new Date(post.date).toLocaleDateString('id-ID', { day: 'numeric', month: 'long', year: 'numeric' });
            html += `
                <div class="col">
                    <div class="card h-100 shadow-sm border-0 article-card-hover">
                        <a href="${post.link}" target="_blank" rel="noopener noreferrer">
                            <img src="${featuredImage}" class="card-img-top" alt="${post.title.rendered}" style="height: 180px; object-fit: cover;">
                        </a>
                        <div class="card-body">
                            <h5 class="card-title h6"><a href="${post.link}" target="_blank" rel="noopener noreferrer" class="text-decoration-none stretched-link article-title-link">${post.title.rendered}</a></h5>
                        </div>
                        <div class="card-footer bg-transparent border-0 pt-0 pb-2 text-muted small">
                            <i class="fas fa-calendar-alt me-1"></i> ${postDate}
                        </div>
                    </div>
                </div>`;
        });
        containers.posts.innerHTML = html;
    };

    const renderPages = (pages) => {
        if (!pages || pages.length === 0) return;
        sections.pages.style.display = 'block';
        let html = '';
        pages.forEach(page => {
            html += `
                <a href="${page.link}" target="_blank" rel="noopener noreferrer" class="list-group-item list-group-item-action d-flex align-items-center">
                    <i class="fas fa-file-alt fa-fw me-3 text-primary"></i>
                    <span class="fw-medium">${page.title.rendered}</span>
                    <i class="fas fa-external-link-alt fa-xs ms-auto text-muted"></i>
                </a>`;
        });
        containers.pages.innerHTML = html;
    };
    
    const renderMedia = (media) => {
        if (!media || media.length === 0) return;
        sections.media.style.display = 'block';
        let html = '';
        media.forEach(item => {
            if (item.media_type === 'image' && item.source_url) {
                html += `
                    <div class="col">
                        <a href="${item.source_url}" data-bs-toggle="tooltip" title="${item.title.rendered}" target="_blank" rel="noopener noreferrer">
                            <img src="${item.source_url}" class="img-fluid rounded shadow-sm" alt="${item.title.rendered}" style="aspect-ratio: 1/1; object-fit: cover;">
                        </a>
                    </div>`;
            }
        });
        containers.media.innerHTML = html;
        // Inisialisasi tooltip setelah media dirender
        var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
        var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl)
        })
    };

    const renderCategories = (categories) => {
        if (!categories || categories.length === 0) return;
        sections.categories.style.display = 'block';
        let html = '';
        categories.forEach(cat => {
            if (cat.count > 0) { // Hanya tampilkan kategori yang punya postingan
                html += `
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                        <a href="${cat.link}" target="_blank" rel="noopener noreferrer" class="text-decoration-none">${cat.name}</a>
                        <span class="badge bg-primary rounded-pill">${cat.count}</span>
                    </li>`;
            }
        });
        containers.categories.innerHTML = html;
    };

    async function fetchAllWordPressData() {
        try {
            const [postsRes, pagesRes, mediaRes, categoriesRes] = await Promise.all([
                fetch(endpoints.posts),
                fetch(endpoints.pages),
                fetch(endpoints.media),
                fetch(endpoints.categories)
            ]);

            if (!postsRes.ok || !pagesRes.ok || !mediaRes.ok || !categoriesRes.ok) {
                 throw new Error('Satu atau lebih permintaan API gagal.');
            }

            const posts = await postsRes.json();
            const pages = await pagesRes.json();
            const media = await mediaRes.json();
            const categories = await categoriesRes.json();

            // Sembunyikan loading indicator setelah semua data siap
            loadingIndicator.style.display = 'none';

            // Panggil fungsi render untuk setiap jenis data
            renderPosts(posts);
            renderPages(pages);
            renderMedia(media);
            renderCategories(categories);

        } catch (error) {
            console.error('Gagal mengambil data dari WordPress:', error);
            loadingIndicator.innerHTML = '<div class="alert alert-danger">Maaf, terjadi kesalahan saat memuat konten dari situs utama.</div>';
        }
    }

    // Panggil fungsi utama untuk memulai proses
    fetchAllWordPressData();
});
</script>

<?php
require_once $project_root . '/includes/footer_public.php';
?>