<?php
// public/articles.php (Listing all articles)
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
$project_root = dirname(__DIR__);
require_once $project_root . '/includes/db_connect.php';
require_once $project_root . '/admin/includes/functions.php'; // For resolve_url, esc_html

$site_title = "Arsip Berita";
$meta_description = "Kumpulan berita dan informasi terbaru dari UGKMB.";

$articles = [];
$total_articles = 0;
$limit = 9; // Articles per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

if (isset($pdo)) {
    try {
        // Count total articles for pagination
        $stmt_count = $pdo->query("SELECT COUNT(*) FROM portal_articles WHERE status = 'published' AND published_at <= NOW()");
        $total_articles = $stmt_count->fetchColumn();

        // Fetch articles for the current page
        $stmt = $pdo->prepare("SELECT a.title, a.slug, a.excerpt, a.featured_image_url, a.published_at, c.name as category_name, c.slug as category_slug
                               FROM portal_articles a
                               LEFT JOIN portal_categories c ON a.category_id = c.id
                               WHERE a.status = 'published' AND a.published_at <= NOW()
                               ORDER BY a.published_at DESC
                               LIMIT :limit OFFSET :offset");
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $articles = $stmt->fetchAll();
    } catch (\PDOException $e) {
        error_log("Error fetching articles list: " . $e->getMessage());
        // Set a user-friendly error message if needed
    }
}

// Fetch global settings for header/footer
$settings = []; // Will be populated by header_public.php
$theme = $_COOKIE['theme'] ?? 'light-theme'; // For body class

// Include header
require_once $project_root . '/includes/header_public.php';
?>

<main class="container py-5">
    <div class="row">
        <div class="col-lg-12">
            <header class="mb-5 text-center">
                <h1 class="display-4 fw-bold"><?php echo $site_title; ?></h1>
                <p class="lead text-muted"><?php echo $meta_description; ?></p>
            </header>

            <?php if (isset($_SESSION['flash_message'])): // Display flash messages if any ?>
                <div class="alert alert-<?php echo $_SESSION['flash_message']['type']; ?> alert-dismissible fade show" role="alert">
                    <?php echo $_SESSION['flash_message']['text']; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['flash_message']); ?>
            <?php endif; ?>

            <?php if (!empty($articles)): ?>
                <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
                    <?php foreach ($articles as $article): ?>
                    <div class="col">
                        <div class="card h-100 shadow-sm border-0 article-card-hover">
                            <?php if (!empty($article['featured_image_url'])): ?>
                            <a href="<?php echo resolve_url('article.php?slug=' . esc_html($article['slug'])); ?>">
                                <img src="<?php echo resolve_url(esc_html($article['featured_image_url'])); ?>" class="card-img-top" alt="<?php echo esc_html($article['title']); ?>" style="height: 200px; object-fit: cover;">
                            </a>
                            <?php endif; ?>
                            <div class="card-body d-flex flex-column">
                                <?php if (!empty($article['category_name'])): ?>
                                    <a href="<?php echo resolve_url('category.php?slug=' . esc_html($article['category_slug'])); ?>" class="text-muted text-decoration-none small mb-1 text-uppercase fw-bold article-category"><?php echo esc_html($article['category_name']); ?></a>
                                <?php endif; ?>
                                <h5 class="card-title mb-2"><a href="<?php echo resolve_url('article.php?slug=' . esc_html($article['slug'])); ?>" class="text-decoration-none stretched-link article-title-link"><?php echo esc_html($article['title']); ?></a></h5>
                                <p class="card-text small text-muted mb-2"><i class="fas fa-calendar-alt me-1"></i><?php echo date('d F Y', strtotime($article['published_at'])); ?></p>
                                <p class="card-text flex-grow-1 mb-3 article-excerpt"><?php echo esc_html(substr(strip_tags($article['excerpt'] ?? ''), 0, 120)) . (strlen(strip_tags($article['excerpt'] ?? '')) > 120 ? '...' : ''); ?></p>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>

                <?php if ($total_articles > $limit): ?>
                <nav aria-label="Page navigation" class="mt-5">
                    <ul class="pagination justify-content-center">
                        <?php
                        $total_pages = ceil($total_articles / $limit);
                        $prev_disabled = ($page <= 1) ? "disabled" : "";
                        $next_disabled = ($page >= $total_pages) ? "disabled" : "";
                        ?>
                        <li class="page-item <?php echo $prev_disabled; ?>">
                            <a class="page-link" href="?page=<?php echo $page - 1; ?>" aria-label="Previous">
                                <span aria-hidden="true">&laquo;</span>
                            </a>
                        </li>
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?php echo ($page == $i) ? "active" : ""; ?>"><a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a></li>
                        <?php endfor; ?>
                        <li class="page-item <?php echo $next_disabled; ?>">
                            <a class="page-link" href="?page=<?php echo $page + 1; ?>" aria-label="Next">
                                <span aria-hidden="true">&raquo;</span>
                            </a>
                        </li>
                    </ul>
                </nav>
                <?php endif; ?>

            <?php else: ?>
                <div class="alert alert-info text-center" role="alert">
                    Tidak ada berita atau artikel yang tersedia saat ini.
                </div>
            <?php endif; ?>
        </div>
    </div>
</main>

<?php
require_once $project_root . '/includes/footer_public.php';
?>
