<?php
// public/study_program.php (Detail of a single study program) - Updated Version
if (session_status() == PHP_SESSION_NONE) {
    // session_start(); // Diasumsikan sudah ditangani oleh config.php
}
if (!defined('PROJECT_ROOT')) { define('PROJECT_ROOT', dirname(__DIR__)); }

require_once PROJECT_ROOT . '/includes/db_connect.php';
require_once PROJECT_ROOT . '/admin/includes/functions.php'; // For resolve_url, esc_html

$sp_slug = $_GET['slug'] ?? null;
$sp_data = null;
$faculty_data_for_sp = null;
$site_title = "Detail Program Studi";
$meta_description = "Informasi lengkap mengenai program studi.";

if (empty($sp_slug)) {
    http_response_code(404);
    $site_title = "Program Studi Tidak Ditemukan";
    // Tidak perlu set_flash_message di halaman publik, cukup tampilkan pesan error.
}

if (isset($pdo) && !empty($sp_slug)) {
    try {
        // Fetch study program details and its faculty name/slug
        $stmt_sp = $pdo->prepare("SELECT sp.*, f.name as faculty_name, f.slug as faculty_slug 
                                  FROM portal_study_programs sp
                                  LEFT JOIN portal_faculties f ON sp.faculty_id = f.id
                                  WHERE sp.slug = :slug AND sp.is_active = 1 
                                  LIMIT 1");
        $stmt_sp->bindParam(':slug', $sp_slug, PDO::PARAM_STR);
        $stmt_sp->execute();
        $sp_data = $stmt_sp->fetch(PDO::FETCH_ASSOC);

        if ($sp_data) {
            $site_title = esc_html($sp_data['name']);
            // Generate meta description from description, fallback to default
            $meta_description = !empty($sp_data['description_html']) 
                ? esc_html(substr(strip_tags($sp_data['description_html']), 0, 160)) 
                : "Informasi lengkap mengenai Program Studi " . esc_html($sp_data['name']);
            
            if (!empty($sp_data['faculty_id'])) {
                $faculty_data_for_sp = ['name' => $sp_data['faculty_name'], 'slug' => $sp_data['faculty_slug']];
            }
        } else {
            http_response_code(404);
            $site_title = "Program Studi Tidak Ditemukan";
            $meta_description = "Program Studi yang Anda cari tidak ditemukan atau saat ini tidak aktif.";
        }
    } catch (\PDOException $e) {
        error_log("Error fetching study program detail: " . $e->getMessage());
        http_response_code(500);
        $site_title = "Kesalahan Server";
        $meta_description = "Terjadi kesalahan saat memuat detail Program Studi.";
    }
}

// Fetch global settings for header/footer
$settings = []; // Akan diisi oleh header_public.php
$theme = $_COOKIE['theme'] ?? 'light-theme';

// Include header
require_once PROJECT_ROOT . '/public/includes/header_public.php';

// Fungsi untuk merender HTML yang aman dari database
function render_safe_html($html_content) {
    // PENTING: Untuk keamanan produksi, gunakan library seperti HTML Purifier di sini
    // untuk membersihkan HTML dari potensi XSS sebelum ditampilkan.
    // Untuk saat ini, kita akan langsung menampilkannya karena diasumsikan input dari admin aman.
    return $html_content;
}

?>

<main class="container py-5">
    <div class="row">
        <div class="col-lg-10 mx-auto">
            <?php if ($sp_data): ?>
                <article class="study-program-detail">
                    
                    <nav aria-label="breadcrumb" class="mb-4">
                        <ol class="breadcrumb">
                            <li class="breadcrumb-item"><a href="<?php echo resolve_url('index.php'); ?>">Beranda</a></li>
                            <?php if ($faculty_data_for_sp): ?>
                                <li class="breadcrumb-item"><a href="<?php echo resolve_url('faculties.php'); ?>">Fakultas</a></li>
                                <li class="breadcrumb-item"><a href="<?php echo resolve_url('faculty.php?slug=' . esc_html($faculty_data_for_sp['slug'])); ?>"><?php echo esc_html($faculty_data_for_sp['name']); ?></a></li>
                            <?php else: ?>
                                <li class="breadcrumb-item"><a href="<?php echo resolve_url('study_programs_list.php'); ?>">Program Studi</a></li>
                            <?php endif; ?>
                            <li class="breadcrumb-item active" aria-current="page"><?php echo esc_html($sp_data['name']); ?></li>
                        </ol>
                    </nav>

                    <header class="mb-4 text-center">
                        <?php if (!empty($sp_data['logo_url'])): ?>
                            <img src="<?php echo resolve_url(esc_html($sp_data['logo_url'])); ?>" alt="Logo <?php echo esc_html($sp_data['name']); ?>" class="mb-3" style="max-height: 120px; object-fit: contain;">
                        <?php endif; ?>
                        <h1 class="display-5 fw-bold sp-title"><?php echo esc_html($sp_data['name']); ?></h1>
                        <?php if ($faculty_data_for_sp): ?>
                            <p class="lead text-muted">
                                Bagian dari <a href="<?php echo resolve_url('faculty.php?slug=' . esc_html($faculty_data_for_sp['slug'])); ?>" class="text-decoration-none fw-medium"><?php echo esc_html($faculty_data_for_sp['name']); ?></a>
                            </p>
                        <?php endif; ?>
                        <?php if (!empty($sp_data['degree'])): ?>
                            <p class="text-muted fs-5">Jenjang: <?php echo esc_html($sp_data['degree']); ?></p>
                        <?php endif; ?>
                    </header>
                    <hr class="my-4">

                    <div class="row">
                        <div class="col-md-<?php echo (!empty($sp_data['vision_html']) || !empty($sp_data['mission_html'])) ? '8' : '12'; ?>">
                            <?php if (!empty($sp_data['description_html'])): ?>
                            <section class="mb-4 entry-content">
                                <h3 class="h4 mb-3">Deskripsi Program Studi</h3>
                                <?php echo render_safe_html($sp_data['description_html']); ?>
                            </section>
                            <?php endif; ?>
                        </div>

                        <?php if (!empty($sp_data['vision_html']) || !empty($sp_data['mission_html'])): ?>
                        <div class="col-md-4">
                            <div class="sticky-top" style="top: 20px;">
                                <?php if (!empty($sp_data['vision_html'])): ?>
                                <div class="card bg-light-subtle p-3 mb-3 shadow-sm">
                                    <h4 class="h5 card-title"><i class="fas fa-bullseye me-2 text-primary"></i>Visi</h4>
                                    <div class="card-body entry-content small p-0">
                                        <?php echo render_safe_html($sp_data['vision_html']); ?>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($sp_data['mission_html'])): ?>
                                 <div class="card bg-light-subtle p-3 shadow-sm">
                                    <h4 class="h5 card-title"><i class="fas fa-tasks me-2 text-primary"></i>Misi</h4>
                                    <div class="card-body entry-content small p-0">
                                        <?php echo render_safe_html($sp_data['mission_html']); ?>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>

                    <?php if (!empty($sp_data['head_of_program_name']) || !empty($sp_data['accreditation_status']) || !empty($sp_data['website_url']) || !empty($sp_data['brochure_url'])): ?>
                    <section class="mt-4 mb-5 sp-info-section">
                        <h3 class="h4 mb-3">Informasi Tambahan</h3>
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped">
                                <tbody>
                                    <?php if (!empty($sp_data['head_of_program_name'])): ?>
                                    <tr>
                                        <th scope="row" style="width: 30%;">Ketua Program Studi</th>
                                        <td><?php echo esc_html($sp_data['head_of_program_name']); ?></td>
                                    </tr>
                                    <?php endif; ?>
                                    <?php if (!empty($sp_data['accreditation_status'])): ?>
                                    <tr>
                                        <th scope="row">Status Akreditasi</th>
                                        <td>
                                            <?php echo esc_html($sp_data['accreditation_status']); ?>
                                            <?php if (!empty($sp_data['accreditation_number'])): ?>
                                                (SK: <?php echo esc_html($sp_data['accreditation_number']); ?>)
                                            <?php endif; ?>
                                            <?php if (!empty($sp_data['accreditation_expiry_date']) && $sp_data['accreditation_expiry_date'] != '0000-00-00'): ?>
                                                <br><small class="text-muted">Berlaku hingga: <?php echo date('d F Y', strtotime($sp_data['accreditation_expiry_date'])); ?></small>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endif; ?>
                                     <?php if (!empty($sp_data['accreditation_certificate_url'])): ?>
                                    <tr>
                                        <th scope="row">Sertifikat Akreditasi</th>
                                        <td><a href="<?php echo resolve_url(esc_html($sp_data['accreditation_certificate_url'])); ?>" target="_blank" rel="noopener noreferrer" class="btn btn-sm btn-outline-info"><i class="fas fa-file-pdf me-1"></i> Lihat Sertifikat</a></td>
                                    </tr>
                                    <?php endif; ?>
                                    <?php if (!empty($sp_data['website_url'])): ?>
                                    <tr>
                                        <th scope="row">Website Prodi</th>
                                        <td><a href="<?php echo esc_html($sp_data['website_url']); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html($sp_data['website_url']); ?></a></td>
                                    </tr>
                                    <?php endif; ?>
                                    <?php if (!empty($sp_data['brochure_url'])): ?>
                                    <tr>
                                        <th scope="row">Brosur</th>
                                        <td><a href="<?php echo resolve_url(esc_html($sp_data['brochure_url'])); ?>" target="_blank" rel="noopener noreferrer" class="btn btn-sm btn-outline-success"><i class="fas fa-file-download me-1"></i> Unduh Brosur</a></td>
                                    </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </section>
                    <?php endif; ?>
                    
                    <hr class="my-4">
                    <div class="text-center">
                        <?php if ($faculty_data_for_sp): ?>
                            <a href="<?php echo resolve_url('faculty.php?slug=' . esc_html($faculty_data_for_sp['slug'])); ?>" class="btn btn-outline-secondary me-2"><i class="fas fa-arrow-left me-1"></i> Kembali ke Fakultas <?php echo esc_html($faculty_data_for_sp['name']); ?></a>
                        <?php endif; ?>
                         <a href="<?php echo resolve_url('faculties.php'); ?>" class="btn btn-outline-primary"><i class="fas fa-university me-1"></i> Lihat Semua Fakultas & Prodi</a>
                    </div>

                </article>
            <?php else: ?>
                <div class="alert alert-warning text-center" role="alert">
                    <h4 class="alert-heading"><?php echo $site_title; ?></h4>
                    <p><?php echo $meta_description; ?></p>
                    <hr>
                    <p class="mb-0">Silakan kembali ke <a href="<?php echo resolve_url('faculties.php'); ?>" class="alert-link">daftar fakultas & program studi</a>.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</main>

<?php
require_once PROJECT_ROOT . '/includes/footer_public.php';
?>