<?php
// public/sitemap.xml.php
/*
 * Dinamis Sitemap Generator untuk UGKMB CMS
 */

// Set header XML
header("Content-Type: application/xml; charset=utf-8");

// Define PROJECT_ROOT if not already defined (e.g., from a central config)
if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__)); // Assumes this file is in public/
}

// Require db_connect which should include config.php for BASE_URL and DB constants
require_once PROJECT_ROOT . '/includes/db_connect.php'; 
// Require functions only if resolve_url or esc_html are not available from other means
// (config.php or db_connect.php might indirectly include them for public use)
// However, it's safer to include it if its functions are directly used.
if (!function_exists('resolve_url')) { // Check if already included
    require_once PROJECT_ROOT . '/admin/includes/functions.php';
}


echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" 
        xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" 
        xsi:schemaLocation="http://www.sitemaps.org/schemas/sitemap/0.9 http://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd">' . "\n";

// Helper function to output a URL entry
function sitemap_url_entry($loc, $lastmod = null, $changefreq = 'weekly', $priority = '0.8') {
    echo "  <url>\n";
    echo "    <loc>" . htmlspecialchars($loc, ENT_XML1, 'UTF-8') . "</loc>\n"; // Use htmlspecialchars for XML context
    if ($lastmod) {
        // Format to W3C Datetime (YYYY-MM-DDThh:mm:ssTZD)
        try {
            $datetime = new DateTime($lastmod);
            echo "    <lastmod>" . $datetime->format(DateTime::ATOM) . "</lastmod>\n";
        } catch (Exception $e) {
            // Log error or skip lastmod if date is invalid
            error_log("Sitemap: Invalid date for lastmod - " . $lastmod);
        }
    }
    echo "    <changefreq>" . htmlspecialchars($changefreq, ENT_XML1, 'UTF-8') . "</changefreq>\n";
    echo "    <priority>" . htmlspecialchars($priority, ENT_XML1, 'UTF-8') . "</priority>\n";
    echo "  </url>\n";
}

if (isset($pdo)) {
    // 1. Homepage
    sitemap_url_entry(resolve_url('index.php'), date('Y-m-d'), 'daily', '1.0');

    // 2. Static Pages (from portal_pages)
    try {
        $stmt_pages = $pdo->query("SELECT slug, updated_at FROM portal_pages WHERE status = 'published' AND (published_at IS NULL OR published_at <= NOW()) ORDER BY updated_at DESC");
        while ($page = $stmt_pages->fetch(PDO::FETCH_ASSOC)) {
            sitemap_url_entry(resolve_url('page.php?slug=' . $page['slug']), $page['updated_at'], 'monthly', '0.7');
        }
    } catch (\PDOException $e) {
        error_log("Sitemap error (pages): " . $e->getMessage());
    }

    // 3. Articles (from portal_articles)
    try {
        $stmt_articles = $pdo->query("SELECT slug, updated_at FROM portal_articles WHERE status = 'published' AND published_at <= NOW() ORDER BY updated_at DESC");
        while ($article = $stmt_articles->fetch(PDO::FETCH_ASSOC)) {
            sitemap_url_entry(resolve_url('article.php?slug=' . $article['slug']), $article['updated_at'], 'weekly', '0.8');
        }
    } catch (\PDOException $e) {
        error_log("Sitemap error (articles): " . $e->getMessage());
    }
    sitemap_url_entry(resolve_url('articles.php'), date('Y-m-d'), 'daily', '0.9');


    // 4. Categories (from portal_categories)
    try {
        $stmt_categories = $pdo->query("SELECT slug, updated_at FROM portal_categories ORDER BY updated_at DESC");
        while ($category = $stmt_categories->fetch(PDO::FETCH_ASSOC)) {
            sitemap_url_entry(resolve_url('category.php?slug=' . $category['slug']), $category['updated_at'], 'weekly', '0.7');
        }
    } catch (\PDOException $e) {
        error_log("Sitemap error (categories): " . $e->getMessage());
    }

    // 5. Tags (from portal_tags)
    try {
        $stmt_tags = $pdo->query("SELECT slug, updated_at FROM portal_tags ORDER BY updated_at DESC");
        while ($tag = $stmt_tags->fetch(PDO::FETCH_ASSOC)) {
            sitemap_url_entry(resolve_url('tag.php?slug=' . $tag['slug']), $tag['updated_at'], 'weekly', '0.6');
        }
    } catch (\PDOException $e) {
        error_log("Sitemap error (tags): " . $e->getMessage());
    }

    // 6. Events (from portal_events)
    try {
        $stmt_events = $pdo->query("SELECT slug, GREATEST(start_datetime, updated_at) as last_activity FROM portal_events WHERE is_published = 1 AND (start_datetime IS NULL OR start_datetime >= CURDATE() - INTERVAL 1 MONTH) ORDER BY start_datetime DESC"); // Include recent past events too
        while ($event = $stmt_events->fetch(PDO::FETCH_ASSOC)) {
            sitemap_url_entry(resolve_url('event.php?slug=' . $event['slug']), $event['last_activity'], 'weekly', '0.7');
        }
    } catch (\PDOException $e) {
        error_log("Sitemap error (events): " . $e->getMessage());
    }
    sitemap_url_entry(resolve_url('events.php'), date('Y-m-d'), 'daily', '0.8');


    // 7. Faculties (from portal_faculties)
    try {
        $stmt_faculties = $pdo->query("SELECT slug, updated_at FROM portal_faculties ORDER BY updated_at DESC");
        while ($faculty = $stmt_faculties->fetch(PDO::FETCH_ASSOC)) {
            sitemap_url_entry(resolve_url('faculty.php?slug=' . $faculty['slug']), $faculty['updated_at'], 'monthly', '0.7');
        }
    } catch (\PDOException $e) {
        error_log("Sitemap error (faculties): " . $e->getMessage());
    }
    sitemap_url_entry(resolve_url('faculties.php'), date('Y-m-d'), 'monthly', '0.8');

    // 8. Study Programs (from portal_study_programs)
    try {
        $stmt_sp = $pdo->query("SELECT slug, updated_at FROM portal_study_programs WHERE is_active = 1 ORDER BY updated_at DESC");
        while ($sp = $stmt_sp->fetch(PDO::FETCH_ASSOC)) {
            sitemap_url_entry(resolve_url('study_program.php?slug=' . $sp['slug']), $sp['updated_at'], 'monthly', '0.6');
        }
    } catch (\PDOException $e) {
        error_log("Sitemap error (study_programs): " . $e->getMessage());
    }

    // 9. Gallery Albums (from portal_gallery_albums)
    try {
        $stmt_albums = $pdo->query("SELECT slug, GREATEST(COALESCE(album_date, '1970-01-01'), updated_at) as last_activity FROM portal_gallery_albums WHERE is_published = 1 ORDER BY COALESCE(album_date, '1970-01-01') DESC");
        while ($album = $stmt_albums->fetch(PDO::FETCH_ASSOC)) {
            sitemap_url_entry(resolve_url('album.php?slug=' . $album['slug']), $album['last_activity'], 'monthly', '0.6');
        }
    } catch (\PDOException $e) {
        error_log("Sitemap error (gallery_albums): " . $e->getMessage());
    }
    sitemap_url_entry(resolve_url('gallery.php'), date('Y-m-d'), 'weekly', '0.7');
    
    // 10. Contact Page
    sitemap_url_entry(resolve_url('contact.php'), date('Y-m-d'), 'yearly', '0.5');


} else {
    sitemap_url_entry(resolve_url('index.php'), date('Y-m-d'), 'daily', '1.0');
    error_log("Sitemap generation failed: PDO object not available.");
}

echo '</urlset>';
?>