<?php
// public/gallery.php (Listing all published gallery albums)
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
$project_root = dirname(__DIR__);
require_once $project_root . '/includes/db_connect.php';
require_once $project_root . '/admin/includes/functions.php'; // For resolve_url, esc_html

$site_title = "Galeri Foto & Video";
$meta_description = "Kumpulan album foto dan video kegiatan di UGKMB.";

$albums = [];
if (isset($pdo)) {
    try {
        // Fetch all published albums, ordered by album_date or creation date
        $stmt = $pdo->query("SELECT ga.id, ga.title, ga.slug, ga.description, ga.cover_image_url, ga.album_date, 
                                    (SELECT COUNT(gi.id) FROM portal_gallery_items gi WHERE gi.album_id = ga.id) as item_count
                             FROM portal_gallery_albums ga
                             WHERE ga.is_published = 1
                             ORDER BY ga.album_date DESC, ga.created_at DESC");
        $albums = $stmt->fetchAll();
    } catch (\PDOException $e) {
        error_log("Error fetching gallery albums list: " . $e->getMessage());
        // Set a user-friendly error message if needed
    }
}

// Fetch global settings for header/footer
$settings = []; // Will be populated by header_public.php
$theme = $_COOKIE['theme'] ?? 'light-theme'; // For body class

// Include header
require_once $project_root . '/includes/header_public.php';
?>

<main class="container py-5">
    <div class="row">
        <div class="col-lg-12">
            <header class="mb-5 text-center">
                <h1 class="display-4 fw-bold"><?php echo $site_title; ?></h1>
                <p class="lead text-muted"><?php echo $meta_description; ?></p>
            </header>

            <?php if (isset($_SESSION['flash_message'])): ?>
                <div class="alert alert-<?php echo $_SESSION['flash_message']['type']; ?> alert-dismissible fade show" role="alert">
                    <?php echo $_SESSION['flash_message']['text']; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['flash_message']); ?>
            <?php endif; ?>

            <?php if (!empty($albums)): ?>
                <div class="row row-cols-1 row-cols-sm-2 row-cols-md-3 g-4">
                    <?php foreach ($albums as $album): ?>
                    <div class="col">
                        <div class="card h-100 shadow-sm border-0 gallery-album-card">
                            <a href="<?php echo resolve_url('album.php?slug=' . esc_html($album['slug'])); ?>" class="text-decoration-none">
                                <?php if (!empty($album['cover_image_url'])): ?>
                                    <img src="<?php echo resolve_url(esc_html($album['cover_image_url'])); ?>" class="card-img-top gallery-album-cover" alt="Cover <?php echo esc_html($album['title']); ?>" style="height: 220px; object-fit: cover;">
                                <?php else: ?>
                                    <div class="bg-secondary text-white d-flex align-items-center justify-content-center gallery-album-placeholder" style="height: 220px;">
                                        <i class="fas fa-images fa-3x"></i>
                                    </div>
                                <?php endif; ?>
                            </a>
                            <div class="card-body">
                                <h5 class="card-title mb-1">
                                    <a href="<?php echo resolve_url('album.php?slug=' . esc_html($album['slug'])); ?>" class="text-decoration-none stretched-link album-title-link">
                                        <?php echo esc_html($album['title']); ?>
                                    </a>
                                </h5>
                                <?php if ($album['album_date']): ?>
                                <p class="card-text small text-muted mb-2"><i class="fas fa-calendar-alt me-1"></i><?php echo date('d F Y', strtotime($album['album_date'])); ?></p>
                                <?php endif; ?>
                                <?php if (!empty($album['description'])): ?>
                                <p class="card-text small album-description-excerpt">
                                    <?php echo esc_html(substr(strip_tags($album['description']), 0, 80)) . (strlen(strip_tags($album['description'])) > 80 ? '...' : ''); ?>
                                </p>
                                <?php endif; ?>
                            </div>
                            <div class="card-footer bg-transparent border-top-0 text-muted small">
                                <?php echo esc_html($album['item_count']); ?> item
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <div class="alert alert-info text-center" role="alert">
                    Tidak ada album galeri yang tersedia saat ini.
                </div>
            <?php endif; ?>
        </div>
    </div>
</main>

<?php
require_once $project_root . '/includes/footer_public.php';
?>
