<?php
// public/article.php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__));
}

require_once PROJECT_ROOT . '/includes/db_connect.php';
// functions.php dari admin di-include oleh header_public.php jika tidak ada alternatif
// Jika ada fungsi publik khusus, bisa di include/functions_public.php
// require_once PROJECT_ROOT . '/admin/includes/functions.php'; // Untuk resolve_url, esc_html

$article_slug = $_GET['slug'] ?? null;
$article = null;
$tags = []; // Inisialisasi array tags
$related_articles = []; // Inisialisasi related articles

// Default values, akan di-override jika artikel ditemukan
$site_title = "Artikel Tidak Ditemukan";
$meta_description = "Artikel yang Anda cari tidak ditemukan atau belum dipublikasikan.";
$page_content = null; // Untuk meta keywords di header_public.php

if (empty($article_slug)) {
    http_response_code(404);
    set_flash_message('danger', 'Artikel tidak ditemukan (slug kosong).');
    // header("Location: " . resolve_url('articles.php')); // Redirect to all articles
    // exit;
}

if (isset($pdo) && !empty($article_slug)) {
    try {
        $stmt = $pdo->prepare("SELECT a.*, c.name as category_name, c.slug as category_slug, u.full_name as author_name
                               FROM portal_articles a
                               LEFT JOIN portal_categories c ON a.category_id = c.id
                               LEFT JOIN portal_users u ON a.author_id = u.id
                               WHERE a.slug = :slug AND a.status = 'published' AND a.published_at <= NOW()
                               LIMIT 1");
        $stmt->bindParam(':slug', $article_slug);
        $stmt->execute();
        $article = $stmt->fetch();

        if ($article) {
            $site_title = htmlspecialchars($article['title']); // Akan digabungkan dengan site_name di header
            $meta_description = htmlspecialchars(substr(strip_tags($article['excerpt'] ?: $article['content_html']), 0, 160));
            $page_content = $article; // Agar meta_keywords bisa diakses di header

            // Increment views_count
            $update_views = $pdo->prepare("UPDATE portal_articles SET views_count = views_count + 1 WHERE id = :id");
            $update_views->bindParam(':id', $article['id']);
            $update_views->execute();

            // Fetch tags for this article
            $stmt_tags = $pdo->prepare("SELECT t.name, t.slug FROM portal_tags t JOIN portal_article_tag pat ON t.id = pat.tag_id WHERE pat.article_id = :article_id");
            $stmt_tags->bindParam(':article_id', $article['id'], PDO::PARAM_INT);
            $stmt_tags->execute();
            $tags = $stmt_tags->fetchAll();

            // Fetch related articles (misal, dari kategori yang sama, exclude current)
            if ($article['category_id']) {
                $stmt_related = $pdo->prepare("SELECT title, slug, featured_image_url, excerpt
                                               FROM portal_articles
                                               WHERE category_id = :category_id AND id != :current_article_id AND status = 'published' AND published_at <= NOW()
                                               ORDER BY published_at DESC LIMIT 3");
                $stmt_related->execute([':category_id' => $article['category_id'], ':current_article_id' => $article['id']]);
                $related_articles = $stmt_related->fetchAll();
            }

        } else {
            http_response_code(404);
            // $site_title dan $meta_description sudah di-set default di atas
        }
    } catch (\PDOException $e) {
        error_log("Error fetching article (slug: $article_slug): " . $e->getMessage());
        http_response_code(500);
        $site_title = "Kesalahan Server";
        $meta_description = "Terjadi kesalahan saat memuat artikel.";
    }
}

// Fetch global settings & theme (sudah di-handle di header_public.php)
require_once PROJECT_ROOT . '/includes/header_public.php';
?>

<main class="main-content py-4 py-md-5">
    <div class="container">
        <div class="row">
            <div class="col-lg-8 col-xl-9 mx-auto">
                <?php if ($article): ?>
                    <article class="article-detail">
                        <header class="mb-4">
                            <?php if (!empty($article['category_name'])): ?>
                                <a href="<?php echo resolve_url('category.php?slug=' . htmlspecialchars($article['category_slug'])); ?>" class="text-uppercase text-danger fw-bold small text-decoration-none article-category-link"><?php echo htmlspecialchars($article['category_name']); ?></a>
                            <?php endif; ?>
                            <h1 class="display-5 fw-bold mt-2 mb-3 article-title"><?php echo htmlspecialchars($article['title']); ?></h1>
                            <div class="text-muted fst-italic mb-2 article-meta">
                                <span class="me-3"><i class="fas fa-calendar-alt fa-fw"></i> <?php echo date('d F Y', strtotime($article['published_at'])); ?></span>
                                <?php if (!empty($article['author_name'])): ?>
                                    <span class="me-3"><i class="fas fa-user fa-fw"></i> Oleh <?php echo htmlspecialchars($article['author_name']); ?></span>
                                <?php endif; ?>
                                <span title="Jumlah Dilihat"><i class="fas fa-eye fa-fw"></i> <?php echo number_format($article['views_count']); ?></span>
                            </div>
                        </header>

                        <?php if (!empty($article['featured_image_url'])): ?>
                        <figure class="mb-4 text-center article-featured-image">
                            <img class="img-fluid rounded shadow-sm" src="<?php echo resolve_url(htmlspecialchars($article['featured_image_url'])); ?>" alt="<?php echo htmlspecialchars($article['title']); ?>" style="max-height: 550px;">
                            <?php // Pertimbangkan untuk menambahkan caption gambar jika ada fieldnya di DB ?>
                        </figure>
                        <?php endif; ?>

                        <section class="mb-5 article-content entry-content">
                            <?php echo $article['content_html']; // Output HTML dari WYSIWYG ?>
                        </section>

                        <?php if (!empty($tags)): ?>
                        <div class="mb-4 article-tags">
                            <strong class="me-2"><i class="fas fa-tags fa-fw"></i> Tags:</strong>
                            <?php foreach ($tags as $tag): ?>
                                <a href="<?php echo resolve_url('tag.php?slug=' . htmlspecialchars($tag['slug'])); ?>" class="btn btn-sm btn-outline-secondary mb-1 me-1"><?php echo htmlspecialchars($tag['name']); ?></a>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>

                        <hr class="my-4">

                        <!-- Social Share Buttons (Contoh Sederhana, perlu JS untuk fungsionalitas penuh) -->
                        <div class="social-share-buttons mb-5 text-center">
                            <h5 class="mb-3">Bagikan Artikel Ini:</h5>
                            <a href="https://www.facebook.com/sharer/sharer.php?u=<?php echo urlencode(resolve_url('article.php?slug=' . $article['slug'])); ?>" target="_blank" class="btn btn-outline-primary btn-lg me-2" title="Bagikan ke Facebook"><i class="fab fa-facebook-f"></i></a>
                            <a href="https://twitter.com/intent/tweet?url=<?php echo urlencode(resolve_url('article.php?slug=' . $article['slug'])); ?>&text=<?php echo urlencode($article['title']); ?>" target="_blank" class="btn btn-outline-info btn-lg me-2" title="Bagikan ke Twitter/X"><i class="fab fa-twitter"></i></a>
                            <a href="https://wa.me/?text=<?php echo urlencode($article['title'] . " - " . resolve_url('article.php?slug=' . $article['slug'])); ?>" target="_blank" class="btn btn-outline-success btn-lg me-2" title="Bagikan ke WhatsApp"><i class="fab fa-whatsapp"></i></a>
                            <button class="btn btn-outline-secondary btn-lg" onclick="copyArticleLink()" title="Salin Tautan"><i class="fas fa-link"></i></button>
                        </div>

                        <?php if (!empty($related_articles)): ?>
                        <section class="related-articles mt-5 pt-4 border-top">
                            <h3 class="mb-4 section-title">Artikel Terkait</h3>
                            <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
                                <?php foreach ($related_articles as $related_article): ?>
                                <div class="col d-flex align-items-stretch">
                                    <div class="card h-100 custom-card article-card-hover">
                                        <?php if (!empty($related_article['featured_image_url'])): ?>
                                        <a href="<?php echo resolve_url('article.php?slug=' . htmlspecialchars($related_article['slug'])); ?>">
                                            <img src="<?php echo resolve_url(htmlspecialchars($related_article['featured_image_url'])); ?>" class="card-img-top custom-card-img" alt="<?php echo htmlspecialchars($related_article['title']); ?>" loading="lazy">
                                        </a>
                                        <?php endif; ?>
                                        <div class="card-body d-flex flex-column">
                                            <h5 class="card-title h6 custom-card-title"><a href="<?php echo resolve_url('article.php?slug=' . htmlspecialchars($related_article['slug'])); ?>" class="text-decoration-none stretched-link"><?php echo htmlspecialchars($related_article['title']); ?></a></h5>
                                            <p class="card-text small custom-card-excerpt flex-grow-1"><?php echo htmlspecialchars(substr(strip_tags($related_article['excerpt'] ?? ''), 0, 80)) . '...'; ?></p>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </section>
                        <?php endif; ?>


                        <?php if ($article['allow_comments']): ?>
                        <section id="comments" class="mt-5 pt-4 border-top">
                            <h3 class="mb-4 section-title">Komentar</h3>
                            <!-- Form Komentar (jika diimplementasikan) -->
                            <!-- <form id="commentForm" action="path/to/process_comment.php" method="POST">
                                <input type="hidden" name="article_id" value="<?php echo $article['id']; ?>">
                                <input type="hidden" name="csrf_token_comment" value="<?php // echo generate_csrf_token(); // Perlu cara memuat CSRF jika form di-load terpisah ?>">
                                <div class="mb-3">
                                    <label for="comment_author_name" class="form-label">Nama</label>
                                    <input type="text" class="form-control" id="comment_author_name" name="author_name" required>
                                </div>
                                <div class="mb-3">
                                    <label for="comment_author_email" class="form-label">Email</label>
                                    <input type="email" class="form-control" id="comment_author_email" name="author_email" required>
                                </div>
                                <div class="mb-3">
                                    <label for="comment_content" class="form-label">Komentar Anda</label>
                                    <textarea class="form-control" id="comment_content" name="comment_content" rows="4" required></textarea>
                                </div>
                                <button type="submit" class="btn btn-primary">Kirim Komentar</button>
                            </form>
                            <div id="commentList" class="mt-4">
                                 Daftar komentar akan dimuat di sini (mungkin via AJAX) 
                            </div> -->
                            <p class="text-muted"><i>Fitur komentar sedang dalam pengembangan.</i></p>
                        </section>
                        <?php endif; ?>

                    </article>
                <?php else: ?>
                    <div class="alert alert-warning text-center" role="alert">
                        <h4 class="alert-heading"><?php echo $site_title; ?></h4>
                        <p><?php echo $meta_description; ?></p>
                        <hr>
                        <p class="mb-0">Silakan kembali ke <a href="<?php echo resolve_url('articles.php'); ?>" class="alert-link">daftar berita</a> atau <a href="<?php echo resolve_url('index.php'); ?>" class="alert-link">beranda</a>.</p>
                    </div>
                <?php endif; ?>
            </div>
            <!-- Sidebar Opsional (jika ada) -->
            <!-- 
            <div class="col-lg-4 col-xl-3">
                <aside class="sticky-top" style="top: 80px;">
                    <div class="card mb-4">
                        <div class="card-header">Kategori Lain</div>
                        <div class="list-group list-group-flush">
                            <?php
                            // Contoh mengambil kategori lain
                            // if(isset($pdo)) {
                            //  $stmt_cats_sidebar = $pdo->query("SELECT name, slug FROM portal_categories ORDER BY name ASC LIMIT 5");
                            //  while($cat_sidebar = $stmt_cats_sidebar->fetch()) {
                            //      echo '<a href="'.resolve_url('category.php?slug='.htmlspecialchars($cat_sidebar['slug'])).'" class="list-group-item list-group-item-action">'.htmlspecialchars($cat_sidebar['name']).'</a>';
                            //  }
                            // }
                            ?>
                        </div>
                    </div>
                    <div class="card mb-4">
                        <div class="card-header">Artikel Populer</div>
                        <div class="list-group list-group-flush">
                            <?php
                            // Contoh mengambil artikel populer
                            // if(isset($pdo)) {
                            //  $stmt_pop_sidebar = $pdo->query("SELECT title, slug FROM portal_articles WHERE status='published' ORDER BY views_count DESC LIMIT 5");
                            //  while($pop_sidebar = $stmt_pop_sidebar->fetch()) {
                            //      echo '<a href="'.resolve_url('article.php?slug='.htmlspecialchars($pop_sidebar['slug'])).'" class="list-group-item list-group-item-action">'.htmlspecialchars($pop_sidebar['title']).'</a>';
                            //  }
                            // }
                            ?>
                        </div>
                    </div>
                </aside>
            </div>
             -->
        </div>
    </div>
</main>

<?php
$public_page_scripts = []; // Inisialisasi array script
if ($article) { // Hanya tambahkan script jika artikel ada
    ob_start(); // Mulai output buffering untuk menangkap script
    ?>
    <script>
    function copyArticleLink() {
        const articleUrl = window.location.href;
        navigator.clipboard.writeText(articleUrl).then(function() {
            alert('Tautan artikel berhasil disalin ke clipboard!');
        }, function(err) {
            alert('Gagal menyalin tautan. Error: ' + err);
        });
    }

    // Jika ingin AJAX untuk komentar (contoh dasar)
    // document.addEventListener('DOMContentLoaded', function() {
    //     const commentForm = document.getElementById('commentForm');
    //     if (commentForm) {
    //         commentForm.addEventListener('submit', function(e) {
    //             e.preventDefault();
    //             const formData = new FormData(this);
    //             // Lakukan fetch POST ke backend untuk submit komentar
    //             // Tanggapi dengan pesan sukses/error, dan update #commentList
    //             console.log('Submitting comment (AJAX placeholder)...', Object.fromEntries(formData));
    //             // Tampilkan pesan "Komentar sedang dikirim..."
    //             // Setelah sukses, reset form dan refresh daftar komentar
    //         });
    //     }
    // });
    </script>
    <?php
    $inline_script = ob_get_clean(); // Ambil konten buffer dan bersihkan
    $public_page_scripts[] = $inline_script; // Tambahkan sebagai blok script inline
}

require_once PROJECT_ROOT . '/includes/footer_public.php';
?>