<?php
// admin/modules/menus/process_menu.php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// 1. Inisialisasi dan Keamanan Dasar
// ----------------------------------------------------
header('Content-Type: application/json');

// Memuat file-file yang diperlukan
$project_root = dirname(__DIR__, 3);
require_once $project_root . '/includes/db_connect.php';
require_once $project_root . '/admin/includes/functions.php';

// Fungsi untuk mengirim respons JSON dan menghentikan skrip
function json_response($success, $message, $data = [], $http_code = 200) {
    http_response_code($http_code);
    echo json_encode(['success' => $success, 'message' => $message] + $data);
    exit;
}

// Cek apakah request method adalah POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    json_response(false, 'Metode request tidak valid. Hanya POST yang diizinkan.', [], 405);
}

// Cek otorisasi: pengguna harus login dan role-nya admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    json_response(false, 'Akses ditolak. Anda harus login sebagai admin.', [], 403);
}

// Cek koneksi database
if (!isset($pdo)) {
    json_response(false, 'Koneksi database tidak tersedia.', [], 500);
}

// 2. Routing Aksi
// ----------------------------------------------------
$action = $_POST['action'] ?? '';

try {
    switch ($action) {
        case 'create_group':
            handle_create_group($pdo);
            break;

        case 'update_group': // Digunakan oleh edit.php
        case 'edit_group':   // Nama lain yang mungkin digunakan
            handle_update_group($pdo);
            break;
        
        case 'delete_group':
             handle_delete_group($pdo);
             break;

        // --- Tambahkan case untuk item menu di sini nanti ---
        // case 'add_item':
        //     handle_add_menu_item($pdo);
        //     break;
        // case 'update_item_order':
        //     handle_update_item_order($pdo);
        //     break;
        // case 'delete_item':
        //     handle_delete_menu_item($pdo);
        //     break;

        default:
            json_response(false, 'Aksi tidak diketahui atau tidak valid.', ['action' => $action], 400);
    }
} catch (\PDOException $e) {
    error_log("Database Error in process_menu.php: " . $e->getMessage());
    json_response(false, 'Terjadi kesalahan pada server database.', [], 500);
} catch (\Exception $e) {
    error_log("General Error in process_menu.php: " . $e->getMessage());
    json_response(false, 'Terjadi kesalahan umum pada server.', [], 500);
}


// 3. Fungsi Handler untuk Setiap Aksi
// ----------------------------------------------------

/**
 * Menangani pembuatan grup menu baru.
 */
function handle_create_group($pdo) {
    if (!validate_csrf_token($_POST['csrf_token'] ?? '')) {
        json_response(false, 'Token keamanan tidak valid.');
    }

    $name = trim($_POST['name'] ?? '');
    $location = trim($_POST['location'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $errors = [];

    // Validasi input
    if (empty($name)) $errors['name'] = "Nama grup menu wajib diisi.";
    if (empty($location)) {
        $errors['location'] = "Kunci lokasi wajib diisi.";
    } elseif (!preg_match('/^[a-z0-9_]+$/', $location)) {
        $errors['location'] = "Kunci lokasi hanya boleh berisi huruf kecil, angka, dan underscore (_).";
    } else {
        // Cek keunikan lokasi
        $stmt = $pdo->prepare("SELECT id FROM portal_menus WHERE location = :location");
        $stmt->execute([':location' => $location]);
        if ($stmt->fetch()) {
            $errors['location'] = "Kunci lokasi '{$location}' sudah digunakan.";
        }
    }

    if (!empty($errors)) {
        json_response(false, 'Validasi gagal, silakan periksa kembali isian Anda.', ['errors' => $errors]);
    }

    // Simpan ke database
    $stmt = $pdo->prepare("INSERT INTO portal_menus (name, location, description) VALUES (:name, :location, :description)");
    $stmt->execute([
        ':name' => $name,
        ':location' => $location,
        ':description' => $description
    ]);
    
    $new_menu_id = $pdo->lastInsertId();

    json_response(true, 'Grup menu berhasil dibuat!', ['redirect_url' => 'manage_menu_items.php?menu_id=' . $new_menu_id]);
}

/**
 * Menangani pembaruan data grup menu.
 */
function handle_update_group($pdo) {
    if (!validate_csrf_token($_POST['csrf_token'] ?? '')) {
        json_response(false, 'Token keamanan tidak valid.');
    }

    $id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
    $name = trim($_POST['name'] ?? '');
    $location = trim($_POST['location'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $errors = [];

    if (!$id) json_response(false, 'ID grup menu tidak valid.');

    // Validasi input
    if (empty($name)) $errors['name'] = "Nama grup menu wajib diisi.";
    if (empty($location)) {
        $errors['location'] = "Kunci lokasi wajib diisi.";
    } elseif (!preg_match('/^[a-z0-9_]+$/', $location)) {
        $errors['location'] = "Kunci lokasi hanya boleh berisi huruf kecil, angka, dan underscore (_).";
    } else {
        // Cek keunikan lokasi, abaikan ID saat ini
        $stmt = $pdo->prepare("SELECT id FROM portal_menus WHERE location = :location AND id != :id");
        $stmt->execute([':location' => $location, ':id' => $id]);
        if ($stmt->fetch()) {
            $errors['location'] = "Kunci lokasi '{$location}' sudah digunakan oleh grup lain.";
        }
    }
    
    if (!empty($errors)) {
        json_response(false, 'Validasi gagal, silakan periksa kembali isian Anda.', ['errors' => $errors]);
    }

    // Update database
    $stmt = $pdo->prepare("UPDATE portal_menus SET name = :name, location = :location, description = :description WHERE id = :id");
    $stmt->execute([
        ':name'        => $name,
        ':location'    => $location,
        ':description' => $description,
        ':id'          => $id
    ]);

    json_response(true, 'Grup menu berhasil diperbarui!', ['redirect_url' => 'index.php']);
}

/**
 * Menangani penghapusan grup menu beserta item-itemnya.
 */
function handle_delete_group($pdo) {
    // Untuk delete, token bisa dikirim via POST body atau URL (jika metodenya GET)
    // Namun, sangat disarankan menggunakan POST untuk aksi destruktif.
    $token = $_POST['csrf_token'] ?? $_GET['csrf_token'] ?? '';
    if (!validate_csrf_token($token)) {
        json_response(false, 'Token keamanan tidak valid.');
    }

    $id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
    if (!$id) {
         $id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT); // Fallback untuk GET
    }
   
    if (!$id) {
        json_response(false, 'ID grup menu tidak valid.');
    }

    // Gunakan transaksi untuk memastikan semua data terkait terhapus
    $pdo->beginTransaction();
    try {
        // 1. Hapus semua item yang ada di dalam grup menu ini
        $stmt_items = $pdo->prepare("DELETE FROM portal_menu_items WHERE menu_id = :menu_id");
        $stmt_items->execute([':menu_id' => $id]);

        // 2. Hapus grup menu itu sendiri
        $stmt_group = $pdo->prepare("DELETE FROM portal_menus WHERE id = :id");
        $stmt_group->execute([':id' => $id]);

        // Jika semua berhasil, commit transaksi
        $pdo->commit();

        json_response(true, 'Grup menu dan semua item di dalamnya berhasil dihapus.');

    } catch (\PDOException $e) {
        // Jika terjadi error, batalkan semua perubahan
        $pdo->rollBack();
        error_log("Delete menu group error: " . $e->getMessage());
        json_response(false, 'Gagal menghapus grup menu karena kesalahan database.', [], 500);
    }
}
?>