<?php
// admin/modules/articles/index.php
if (session_status() == PHP_SESSION_NONE) {
    // Session management will be handled by header.php or config.php (e.g. via db_connect.php)
    // If not, uncomment: session_start();
}

// Ensure PROJECT_ROOT is defined (typically by config.php via db_connect.php)
if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__, 3)); // Adjust if structure differs: admin/modules/articles -> project_root
}
require_once PROJECT_ROOT . '/includes/db_connect.php'; // Should load config.php
require_once PROJECT_ROOT . '/admin/includes/functions.php'; // esc_html, resolve_url, generate_csrf_token, etc.

// Page specific variables
$page_title = "Manajemen Artikel";
// Path to admin root from current script: admin/modules/articles/ -> ../../
$module_base_path = '../../'; // Used for linking to other admin areas like categories, dashboard

// Redirect if not logged in or not authorized (admin/editor)
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['admin', 'editor'])) {
    set_flash_message('danger', 'Anda tidak memiliki izin untuk mengakses halaman ini.');
    header('Location: ' . $module_base_path . 'login.php'); // Redirect to admin login
    exit;
}

// Include common admin header and sidebar
require_once PROJECT_ROOT . '/admin/includes/header.php';
require_once PROJECT_ROOT . '/admin/includes/sidebar.php';

$articles = [];
$category_data_for_links = []; // To store category IDs if needed for links

if (isset($pdo)) {
    try {
        // Ensure you select a.category_id if you want to link to category edit page
        $stmt = $pdo->query("SELECT a.id, a.title, a.slug, a.category_id, 
                                    c.name as category_name, c.slug as category_slug_for_link, 
                                    u.full_name as author_name, a.status, a.published_at, a.created_at
                             FROM portal_articles a
                             LEFT JOIN portal_categories c ON a.category_id = c.id
                             LEFT JOIN portal_users u ON a.author_id = u.id
                             ORDER BY a.created_at DESC");
        $articles = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (\PDOException $e) {
        set_flash_message('danger', 'Gagal mengambil data artikel: ' . $e->getMessage());
        error_log("Error fetching articles for index: " . $e->getMessage());
        // $articles will remain empty, page will show "Belum ada artikel."
    }
} else {
    set_flash_message('danger', 'Koneksi database tidak tersedia.');
}
?>

<div class="container-fluid px-4">
    <?php echo generate_breadcrumbs([
        ['label' => 'Dashboard', 'url' => $module_base_path . 'index.php'],
        'Manajemen Artikel'
    ]); ?>

    <h1 class="mt-4 mb-4"><?php echo htmlspecialchars($page_title); ?></h1>
    <hr>
    <?php echo display_flash_message(); // Display success/error messages ?>

    <div class="mb-3">
        <a href="create.php" class="btn btn-primary"><i class="fas fa-plus me-1"></i> Tambah Artikel Baru</a>
    </div>

    <div class="card mb-4 shadow-sm">
        <div class="card-header"><i class="fas fa-table me-1"></i> Daftar Artikel</div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-bordered table-striped table-hover" id="dataTableArticles" width="100%" cellspacing="0">
                    <thead>
                        <tr>
                            <th>No</th>
                            <th>Judul</th>
                            <th>Kategori</th>
                            <th>Penulis</th>
                            <th>Status</th>
                            <th>Tgl Publikasi</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (count($articles) > 0): ?>
                            <?php foreach ($articles as $index => $article): ?>
                            <tr>
                                <td><?php echo $index + 1; ?></td>
                                <td><?php echo esc_html($article['title']); ?></td>
                                <td>
                                    <?php if (!empty($article['category_name']) && !empty($article['category_id'])): ?>
                                        <a href="<?php echo esc_html($module_base_path . 'modules/categories/edit.php?id=' . (int)$article['category_id']); ?>" 
                                           title="Edit Kategori <?php echo esc_html($article['category_name']); ?>">
                                            <?php echo esc_html($article['category_name']); ?>
                                        </a>
                                    <?php elseif(!empty($article['category_name'])): ?>
                                        <?php echo esc_html($article['category_name']); ?>
                                    <?php else: ?>
                                        N/A
                                    <?php endif; ?>
                                </td>
                                <td><?php echo esc_html($article['author_name'] ?? 'N/A'); ?></td>
                                <td>
                                    <?php
                                    $status_class = 'secondary'; // Default badge
                                    $status_text = ucfirst(esc_html($article['status']));
                                    switch ($article['status']) {
                                        case 'published':
                                            $status_class = 'success';
                                            break;
                                        case 'draft':
                                            $status_class = 'warning text-dark';
                                            break;
                                        case 'pending_review': // Ensure this matches your DB and processing logic
                                            $status_class = 'info text-dark';
                                            $status_text = 'Pending Review';
                                            break;
                                        case 'archived':
                                            $status_class = 'secondary';
                                            $status_text = 'Diarsip';
                                            break;
                                        // Add other cases as needed
                                    }
                                    ?>
                                    <span class="badge bg-<?php echo $status_class; ?>">
                                        <?php echo $status_text; ?>
                                    </span>
                                </td>
                                <td><?php echo !empty($article['published_at']) ? date('d M Y, H:i', strtotime($article['published_at'])) : '-'; ?></td>
                                <td class="text-center">
                                    <a href="<?php echo resolve_url('article/' . esc_html($article['slug'])); // Assuming public URL structure ?>" target="_blank" class="btn btn-sm btn-outline-info mb-1" title="Lihat Artikel di Situs Publik"><i class="fas fa-eye"></i></a>
                                    <a href="edit.php?id=<?php echo (int)$article['id']; ?>" class="btn btn-sm btn-warning mb-1" title="Edit Artikel"><i class="fas fa-edit"></i></a>
                                    <button type="button" class="btn btn-sm btn-danger mb-1 btn-delete-article"
                                            data-id="<?php echo (int)$article['id']; ?>"
                                            data-title="<?php echo esc_html(addslashes($article['title'])); ?>"
                                            data-url="process_article.php?action=delete&id=<?php echo (int)$article['id']; ?>&csrf_token=<?php echo generate_csrf_token(); ?>"
                                            title="Hapus Artikel">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="7" class="text-center">Belum ada artikel. Silakan tambahkan artikel baru.</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<?php
// Any scripts specific to this page, to be loaded by footer.php
$page_scripts = []; // Initialize or reset
$page_scripts[] = "<script>
    $(document).ready(function() {
        if ($.fn.DataTable) { // Check if DataTable is loaded
            $('#dataTableArticles').DataTable({
                responsive: true,
                language: { // Indonesian language settings for DataTable
                    search: \"Cari:\",
                    lengthMenu: \"Tampilkan _MENU_ entri\",
                    info: \"Menampilkan _START_ sampai _END_ dari _TOTAL_ entri\",
                    infoEmpty: \"Menampilkan 0 sampai 0 dari 0 entri\",
                    infoFiltered: \"(disaring dari _MAX_ total entri)\",
                    paginate: {
                        first: \"Pertama\",
                        last: \"Terakhir\",
                        next: \"Berikutnya\",
                        previous: \"Sebelumnya\"
                    },
                    zeroRecords: \"Tidak ada data yang cocok ditemukan\",
                    emptyTable: \"Tidak ada data tersedia di tabel\"
                },
                // Example: Order by the 'created_at' column descending if you add it to the table/query
                // \"order\": [[ 5, \"desc\" ]] // Adjust column index based on your table
            });
        } else {
            console.error('DataTable plugin is not loaded.');
        }

        // Delete button functionality (assuming it's handled globally or you add specific AJAX here)
        // If not handled globally in an admin-global.js or similar:
        $('#dataTableArticles').on('click', '.btn-delete-article', function() {
            const button = $(this);
            const deleteUrl = button.data('url');
            const articleTitle = button.data('title');
            const articleId = button.data('id');

            if (confirm('Anda yakin ingin menghapus artikel: \"' + articleTitle + '\"? Aksi ini tidak dapat diurungkan.')) {
                // Perform AJAX delete request
                $.ajax({
                    url: deleteUrl, // URL already contains CSRF token
                    type: 'GET', // Or POST if your delete action expects POST
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            if (typeof showAdminNotification === 'function') {
                                showAdminNotification('success', response.message);
                            } else {
                                alert(response.message);
                            }
                            // Remove row from table or reload table/page
                            // Example: button.closest('tr').remove();
                            // Or for DataTable: $('#dataTableArticles').DataTable().row(button.closest('tr')).remove().draw();
                            // For simplicity, often a page reload is easiest if not too disruptive:
                            window.location.reload();
                        } else {
                            if (typeof showAdminNotification === 'function') {
                                showAdminNotification('danger', response.message || 'Gagal menghapus artikel.');
                            } else {
                                alert(response.message || 'Gagal menghapus artikel.');
                            }
                        }
                    },
                    error: function(jqXHR, textStatus, errorThrown) {
                        let errorMsg = 'Kesalahan server saat mencoba menghapus artikel.';
                        if (jqXHR.responseJSON && jqXHR.responseJSON.message) { errorMsg = jqXHR.responseJSON.message; }
                        if (typeof showAdminNotification === 'function') {
                            showAdminNotification('danger', errorMsg);
                        } else {
                            alert(errorMsg);
                        }
                        console.error('Delete AJAX Error:', textStatus, errorThrown);
                    }
                });
            }
        });
    });
</script>";

require_once PROJECT_ROOT . '/admin/includes/footer.php'; // Includes common admin footer and potentially loads $page_scripts
?>