<?php
// admin/includes/functions.php

// Ensure session is started for CSRF and flash messages
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// --- CSRF Protection ---
if (!function_exists('generate_csrf_token')) {
    /**
     * Generates a CSRF token and stores it in the session.
     * If a token already exists, it returns the existing one.
     * @return string The generated CSRF token.
     */
    function generate_csrf_token() {
        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }
}

if (!function_exists('validate_csrf_token')) {
    /**
     * Validates a CSRF token against the one stored in the session.
     * @param string $token The token to validate.
     * @return bool True if the token is valid, false otherwise.
     */
    function validate_csrf_token($token) {
        if (isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token)) {
            return true;
        }
        return false;
    }
}

// --- AWAL PERBAIKAN ---
if (!function_exists('generate_csrf_input')) {
    /**
     * Generates a hidden HTML input field with the CSRF token.
     * @return string The HTML input field.
     */
    function generate_csrf_input() {
        $token = generate_csrf_token(); // Menggunakan fungsi yang sudah ada
        return '<input type="hidden" name="csrf_token" value="' . $token . '">';
    }
}
// --- AKHIR PERBAIKAN ---


// --- URL and Path Helpers ---
if (!function_exists('generate_slug')) {
    /**
     * Generates a URL-friendly slug from a string.
     * @param string $text The string to slugify.
     * @return string The slugified string.
     */
    function generate_slug($text) {
        // Normalisasi ke NFD (Normalization Form Canonical Decomposition) dan hapus diakritik
        if (function_exists('transliterator_transliterate')) { // Membutuhkan ekstensi intl
            $text = transliterator_transliterate('Any-Latin; Latin-ASCII; Lower()', $text);
        } else {
            // Fallback jika intl tidak ada (kurang akurat untuk semua karakter)
            $text = preg_replace('~[^\pL\d]+~u', '-', $text); // Ganti non-letter/non-digit dengan strip
            $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text); // Coba transliterasi
            $text = preg_replace('~[^-\w]+~', '', $text); // Hapus karakter yang tidak diinginkan
        }
        
        $text = preg_replace('/[^\p{L}\p{N}\s-]/u', '', mb_strtolower($text, 'UTF-8')); // Hapus karakter non-alphanumeric kecuali spasi dan strip
        $text = preg_replace('/[\s-]+/', '-', $text); // Ganti spasi dan multiple strip dengan satu strip
        $text = trim($text, '-'); // Hapus strip di awal dan akhir
        $text = preg_replace('/[^a-z0-9-]/', '', $text); // Hapus karakter non-alphanumeric (US-ASCII) tersisa

        if (empty($text)) {
            return 'n-a-' . time(); // Fallback slug jika hasil kosong
        }
        return $text;
    }
}

if (!function_exists('resolve_url')) {
    /**
     * Resolves a URL, useful for displaying images or linking assets from relative paths.
     * Assumes BASE_URL is defined in a central config file.
     * @param string $path The path or URL.
     * @return string The resolved URL.
     */
    function resolve_url($path) {
        if (empty($path)) {
            return '';
        }
        // Jika sudah URL absolut (http, https, //) atau link mailto/tel
        if (preg_match('~^(?:f|ht)tps?://~i', $path) || substr($path, 0, 2) === '//' || strpos($path, 'mailto:') === 0 || strpos($path, 'tel:') === 0) {
            return $path;
        }
        // Jika link anchor atau javascript, jangan diubah
        if (substr($path, 0, 1) === '#' || strpos($path, 'javascript:') === 0) {
            return $path;
        }

        if (!defined('BASE_URL')) {
            $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443)) ? "https://" : "http://";
            $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
            
            // Penentuan base path yang lebih aman: asumsikan aplikasi ada di root dari SCRIPT_FILENAME jika tidak di subdirectory umum
            $script_name = $_SERVER['SCRIPT_NAME']; // e.g., /myapp/admin/index.php or /index.php
            $base_path = rtrim(dirname($script_name), '/\\'); // Menghapus nama file skrip

            // Jika aplikasi Anda berada di dalam subdirektori spesifik (misal 'ugk_cms'), Anda bisa menyesuaikan ini
            // Contoh: if (strpos($base_path, '/ugk_cms') === 0) $base_path = '/ugk_cms'; else $base_path = '';

            // Jika base_path menjadi kosong (root) atau hanya /
            if ($base_path === '' || $base_path === '/') {
                $base_path_adjusted = '';
            } else {
                // Jika ada di subdirektori, pastikan base_path tidak menyertakan /admin/includes dll.
                // Logika ini mungkin perlu disesuaikan dengan struktur direktori Anda
                // Sebagai fallback yang lebih sederhana:
                $base_path_adjusted = ''; // Asumsikan BASE_URL harusnya root jika tidak terdefinisi
            }
            
            $calculated_base_url = rtrim($protocol . $host . $base_path_adjusted, '/');
            error_log("Warning: BASE_URL not defined. Using calculated fallback: " . $calculated_base_url . " for path: " . $path . ". Consider defining BASE_URL in your config.");
            return $calculated_base_url . '/' . ltrim($path, '/');
        }
        return rtrim(BASE_URL, '/') . '/' . ltrim($path, '/');
    }
}

if (!function_exists('admin_url')) {
    /**
     * Generates a URL within the admin area.
     * @param string $path_from_admin_root Path relative to the /admin/ directory.
     * @return string The full URL.
     */
    function admin_url($path_from_admin_root = '') { // Default ke root admin jika path kosong
        return resolve_url('admin/' . ltrim($path_from_admin_root, '/'));
    }
}

// --- File Upload Helper ---
if (!function_exists('upload_file')) {
    /**
     * Handles file uploads securely.
     * @param array $file_input The $_FILES['input_name'] array.
     * @param string $upload_dir_absolute The absolute path to the upload directory (from system root).
     * @param array $allowed_mime_types Array of allowed MIME types.
     * @param int $max_size_bytes Maximum file size in bytes.
     * @return array ['success' => bool, 'filepath_relative' => string|null, 'message' => string]
     */
    function upload_file($file_input, $upload_dir_absolute, $allowed_mime_types, $max_size_bytes) {
        if ($file_input['error'] !== UPLOAD_ERR_OK) {
            $phpFileUploadErrors = [
                UPLOAD_ERR_INI_SIZE   => 'File melebihi batas upload_max_filesize di php.ini.',
                UPLOAD_ERR_FORM_SIZE  => 'File melebihi batas MAX_FILE_SIZE pada form HTML.',
                UPLOAD_ERR_PARTIAL    => 'File hanya terupload sebagian.',
                UPLOAD_ERR_NO_FILE    => 'Tidak ada file yang diupload.',
                UPLOAD_ERR_NO_TMP_DIR => 'Folder temporary tidak ditemukan.',
                UPLOAD_ERR_CANT_WRITE => 'Gagal menulis file ke disk.',
                UPLOAD_ERR_EXTENSION  => 'Ekstensi PHP menghentikan proses upload file.',
            ];
            return ['success' => false, 'message' => $phpFileUploadErrors[$file_input['error']] ?? 'Error upload tidak diketahui: Kode ' . $file_input['error'], 'filepath_relative' => null];
        }

        $file_name = basename($file_input['name']);
        $file_tmp_name = $file_input['tmp_name'];
        $file_size = $file_input['size'];

        if (!function_exists('finfo_open') && !function_exists('mime_content_type')) {
             return ['success' => false, 'message' => 'Ekstensi PHP Fileinfo atau mime_content_type tidak aktif.', 'filepath_relative' => null];
        }
        
        $file_mime_type = '';
        if (function_exists('finfo_open')) {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            if ($finfo) {
                $file_mime_type = finfo_file($finfo, $file_tmp_name);
                finfo_close($finfo);
            }
        }
        if (empty($file_mime_type) && function_exists('mime_content_type')) { // Fallback
            $file_mime_type = mime_content_type($file_tmp_name);
        }
        if (empty($file_mime_type)) {
             return ['success' => false, 'message' => 'Tidak dapat mendeteksi tipe MIME file.', 'filepath_relative' => null];
        }

        if (!in_array($file_mime_type, $allowed_mime_types)) {
            return ['success' => false, 'message' => 'Tipe file tidak diizinkan (' . esc_html($file_mime_type) . '). Hanya: ' . implode(', ', array_map('esc_html', $allowed_mime_types)), 'filepath_relative' => null];
        }
        if ($file_size > $max_size_bytes) {
            return ['success' => false, 'message' => 'Ukuran file terlalu besar. Maks: ' . round($max_size_bytes / 1024 / 1024, 2) . 'MB', 'filepath_relative' => null];
        }

        if (!is_dir($upload_dir_absolute)) {
            if (!mkdir($upload_dir_absolute, 0755, true)) {
                error_log("Upload directory does not exist and could not be created: " . $upload_dir_absolute);
                return ['success' => false, 'message' => 'Gagal membuat direktori upload.', 'filepath_relative' => null];
            }
        }
        if (!is_writable($upload_dir_absolute)) {
            error_log("Upload directory not writable: " . $upload_dir_absolute);
            return ['success' => false, 'message' => 'Direktori upload tidak dapat ditulis. Periksa izin server.', 'filepath_relative' => null];
        }

        $file_extension = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        $sanitized_file_name_base = preg_replace("/[^a-zA-Z0-9_-]/", "_", pathinfo($file_name, PATHINFO_FILENAME));
        if (empty($sanitized_file_name_base)) {
            $sanitized_file_name_base = 'file';
        }
        $unique_file_name = $sanitized_file_name_base . '_' . time() . '_' . bin2hex(random_bytes(4)) . '.' . $file_extension;
        $destination_path_absolute = rtrim($upload_dir_absolute, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $unique_file_name;
        
        if (move_uploaded_file($file_tmp_name, $destination_path_absolute)) {
            $filepath_relative_to_db = null;
            if (defined('PROJECT_ROOT')) {
                $project_root_norm = rtrim(str_replace('\\', '/', PROJECT_ROOT), '/') . '/';
                $destination_norm = str_replace('\\', '/', $destination_path_absolute);
                if (strpos($destination_norm, $project_root_norm) === 0) {
                    $filepath_relative_to_db = substr($destination_norm, strlen($project_root_norm));
                } else {
                    if (strpos($destination_norm, '/uploads/') !== false) {
                         $filepath_relative_to_db = 'uploads/' . substr($destination_norm, strpos($destination_norm, '/uploads/') + strlen('/uploads/'));
                    } else {
                        $filepath_relative_to_db = $unique_file_name; // Fallback hanya nama file
                         error_log("Warning: Could not determine clear relative path for DB from ".$destination_path_absolute." relative to PROJECT_ROOT ".$project_root_norm);
                    }
                }
            } else {
                $filepath_relative_to_db = 'uploads/' . $unique_file_name; // Fallback jika PROJECT_ROOT tidak ada
                error_log("Warning: PROJECT_ROOT not defined. Storing generic relative path: ".$filepath_relative_to_db);
            }
            
            return ['success' => true, 'filepath_relative' => $filepath_relative_to_db, 'message' => 'File berhasil diupload.'];
        } else {
            error_log("Failed to move uploaded file from '" . $file_tmp_name . "' to '" . $destination_path_absolute . "'. PHP error: " . json_encode(error_get_last()));
            return ['success' => false, 'message' => 'Gagal memindahkan file yang diupload. Periksa izin server dan path.', 'filepath_relative' => null];
        }
    }
}

// --- Output Sanitization ---
if (!function_exists('esc_html')) {
    /**
     * Escapes HTML entities for secure output.
     * @param string|null $string The string to escape.
     * @return string The escaped string.
     */
    function esc_html($string) {
        return htmlspecialchars((string)$string, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    }
}

if (!function_exists('sanitize_output')) {
    /**
     * Sanitizes data for output, especially for arrays. (Recursive)
     * @param mixed $data The data to sanitize.
     * @return mixed The sanitized data.
     */
    function sanitize_output($data) {
        if (is_array($data)) {
            return array_map('sanitize_output', $data);
        } elseif (is_object($data)) {
            $cleaned_object = clone $data; // Clone untuk menjaga tipe objek jika memungkinkan
            foreach (get_object_vars($data) as $key => $value) {
                $cleaned_object->$key = sanitize_output($value);
            }
            return $cleaned_object;
        }
        return esc_html((string)$data); 
    }
}

// --- Flash Messaging System ---
if (!function_exists('set_flash_message')) {
    /**
     * Sets a flash message in the session.
     * @param string $type Type of message (e.g., 'success', 'danger', 'warning', 'info').
     * @param string $message The message content.
     */
    function set_flash_message($type, $message) {
        $_SESSION['flash_message'] = ['type' => $type, 'text' => $message];
    }
}

if (!function_exists('display_flash_message')) {
    /**
     * Displays and clears a flash message if one exists.
     * @return string HTML for the flash message or empty string.
     */
    function display_flash_message() {
        if (isset($_SESSION['flash_message'])) {
            $type = esc_html($_SESSION['flash_message']['type']); // esc_html sudah pasti ada
            $message_text = esc_html($_SESSION['flash_message']['text']);
            unset($_SESSION['flash_message']); 

            $allowed_types = ['success', 'danger', 'warning', 'info', 'primary', 'secondary', 'light', 'dark'];
            if (!in_array($type, $allowed_types)) {
                $type = 'info'; 
            }

            return '<div class="alert alert-' . $type . ' alert-dismissible fade show" role="alert">
                      ' . $message_text . '
                      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>';
        }
        return '';
    }
}

if (!function_exists('get_form_error')) {
    /**
     * Helper to get form errors for a specific field.
     * @param string $field The name of the form field.
     * @param array|null $errors_array The array containing all form errors.
     * @return string HTML for the error message or empty string.
     */
    function get_form_error($field, $errors_array = null) {
        if ($errors_array === null && isset($_SESSION['form_errors'])) {
            $errors_array = $_SESSION['form_errors'];
        }
        if (!empty($errors_array) && isset($errors_array[$field])) {
            $error_message = is_array($errors_array[$field]) ? implode('<br>', array_map('esc_html', $errors_array[$field])) : esc_html($errors_array[$field]);
            return '<div class="invalid-feedback d-block">' . $error_message . '</div>';
        }
        return '';
    }
}

if (!function_exists('old_form_data')) {
    /**
     * Helper to get old form data for repopulating forms.
     * @param string $field The name of the form field.
     * @param mixed $default_value Default value if no old data is found.
     * @param array|null $form_data_array The array containing old form data.
     * @return mixed The old value or default value. (Escaping done at output)
     */
    function old_form_data($field, $default_value = '', $form_data_array = null) {
        if ($form_data_array === null) { 
            $form_data_array = $_SESSION['form_data'] ?? []; 
        }
        return $form_data_array[$field] ?? $default_value;
    }
}

// --- Other Utility Functions ---
if (!function_exists('generate_breadcrumbs')) {
    /**
     * Generates a breadcrumb navigation.
     * @param array $crumbs Array of ['label' => 'Link Label', 'url' => 'link_url.php'] or just 'Label'.
     * @return string HTML for the breadcrumbs.
     */
    function generate_breadcrumbs($crumbs = []) {
        if (empty($crumbs)) {
            return '';
        }
        $html = '<nav aria-label="breadcrumb"><ol class="breadcrumb">'; 
        $last_key = array_key_last($crumbs);

        foreach ($crumbs as $key => $crumb) {
            if (is_array($crumb) && isset($crumb['label'])) {
                $label = esc_html($crumb['label']);
                if ($key === $last_key || empty($crumb['url'])) {
                    $html .= '<li class="breadcrumb-item active" aria-current="page">' . $label . '</li>';
                } else {
                    $url = esc_html($crumb['url']); 
                    $html .= '<li class="breadcrumb-item"><a href="' . $url . '">' . $label . '</a></li>';
                }
            } elseif (is_string($crumb)) { 
                 $html .= '<li class="breadcrumb-item active" aria-current="page">' . esc_html($crumb) . '</li>';
            }
        }
        $html .= '</ol></nav>';
        return $html;
    }
}

if (!function_exists('clear_form_session_data')) {
    // Function to clear form session data after processing or display
    function clear_form_session_data() {
        unset($_SESSION['form_data']);
        unset($_SESSION['form_errors']);
    }
}
?>