<?php
// admin/modules/users/process_user.php

// Pastikan config.php dimuat paling awal dan hanya sekali
if (file_exists(dirname(__DIR__, 3) . '/config.php')) {
    require_once dirname(__DIR__, 3) . '/config.php';
} elseif (file_exists(dirname(__FILE__, 4) . '/config.php')) { 
    require_once dirname(__FILE__, 4) . '/config.php';
} else {
    error_log("CRITICAL FAILURE: config.php not found from process_user.php.");
    // Jangan output JSON jika config saja tidak ada, karena header belum tentu diset
    die("Konfigurasi sistem tidak ditemukan. Harap hubungi administrator.");
}

// Fungsi untuk deteksi AJAX (jika belum ada di functions.php global)
if (!function_exists('is_ajax_request')) {
    function is_ajax_request() {
        return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
    }
}

$is_ajax = is_ajax_request();

// Set header JSON jika AJAX, tapi hanya jika belum ada output lain
if ($is_ajax && !headers_sent()) {
    header('Content-Type: application/json');
}

// Respons default, selalu sertakan token CSRF baru
$response = ['success' => false, 'message' => 'Aksi tidak dikenal atau gagal diproses.', 'new_csrf_token' => generate_csrf_token()]; 

// Pengecekan Sesi
if (!isset($_SESSION['user_id'])) {
    $response['message'] = 'Sesi Anda telah berakhir atau tidak valid. Silakan login kembali.';
    if ($is_ajax) { 
        http_response_code(401); // Unauthorized
        echo json_encode($response); 
        exit; 
    }
    set_flash_message('danger', $response['message']);
    header('Location: ' . admin_url('login.php'));
    exit;
}

// Pengecekan Koneksi Database
if (!isset($pdo)) {
    $response['message'] = 'Koneksi database gagal. Operasi tidak dapat dilanjutkan.';
    error_log("CRITICAL: PDO object not available in process_user.php.");
    if ($is_ajax) { 
        http_response_code(503); // Service Unavailable
        echo json_encode($response); 
        exit; 
    }
    set_flash_message('danger', $response['message']);
    header('Location: ' . admin_url('modules/users/index.php'));
    exit;
}

$action = $_POST['action'] ?? $_GET['action'] ?? null;
$module_redirect_url_base = admin_url('modules/users/index.php');


// --- Validasi Role untuk Aksi Tertentu ---
$allowed_for_admin_only = ['create', 'edit', 'delete'];
if (in_array($action, $allowed_for_admin_only) && (!isset($_SESSION['role']) || $_SESSION['role'] !== 'admin')) {
    $response['message'] = 'Akses ditolak. Anda tidak memiliki izin untuk melakukan aksi ini.';
    if ($is_ajax) { http_response_code(403); echo json_encode($response); exit; }
    set_flash_message('danger', $response['message']);
    header('Location: ' . $module_redirect_url_base); 
    exit;
}


// --- Fungsi Validasi Input (Sertakan definisi lengkapnya di sini atau di functions.php) ---
if (!function_exists('validate_user_input_admin')) {
    function validate_user_input_admin($data, $is_edit = false, $user_id_to_edit = null, $pdo_conn = null) {
        $errors = [];
        if (empty($data['username'])) $errors['username'] = 'Username wajib diisi.';
        elseif (strlen($data['username']) < 3 || strlen($data['username']) > 50) $errors['username'] = 'Username harus antara 3 dan 50 karakter.';
        elseif (!preg_match('/^[a-zA-Z0-9_]+$/', $data['username'])) $errors['username'] = 'Username hanya boleh berisi huruf, angka, dan underscore.';
        else {
            // Cek duplikasi username
            $sql_check_user = "SELECT id FROM portal_users WHERE username = :username";
            $params_check_user = [':username' => $data['username']];
            if ($is_edit && $user_id_to_edit) {
                $sql_check_user .= " AND id != :id";
                $params_check_user[':id'] = $user_id_to_edit;
            }
            $stmt_check_user = $pdo_conn->prepare($sql_check_user);
            $stmt_check_user->execute($params_check_user);
            if ($stmt_check_user->fetch()) {
                $errors['username'] = 'Username "' . esc_html($data['username']) . '" sudah digunakan.';
            }
        }

        if (empty($data['email'])) $errors['email'] = 'Email wajib diisi.';
        elseif (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) $errors['email'] = 'Format email tidak valid.';
        elseif (strlen($data['email']) > 100) $errors['email'] = 'Email maksimal 100 karakter.';
        else {
            // Cek duplikasi email
            $sql_check_email = "SELECT id FROM portal_users WHERE email = :email";
            $params_check_email = [':email' => $data['email']];
            if ($is_edit && $user_id_to_edit) {
                $sql_check_email .= " AND id != :id";
                $params_check_email[':id'] = $user_id_to_edit;
            }
            $stmt_check_email = $pdo_conn->prepare($sql_check_email);
            $stmt_check_email->execute($params_check_email);
            if ($stmt_check_email->fetch()) {
                $errors['email'] = 'Email "' . esc_html($data['email']) . '" sudah terdaftar.';
            }
        }
        
        if (!$is_edit && empty($data['password'])) $errors['password'] = 'Password wajib diisi untuk pengguna baru.';
        if (!empty($data['password']) && (strlen($data['password']) < 6)) $errors['password'] = 'Password minimal 6 karakter.';
        if (!empty($data['password']) && $data['password'] !== $data['confirm_password']) $errors['confirm_password'] = 'Konfirmasi password tidak cocok.';

        if (empty($data['role'])) $errors['role'] = 'Role wajib dipilih.';
        elseif (!in_array($data['role'], ['admin', 'editor', 'author'])) $errors['role'] = 'Role tidak valid.';
        
        if (empty($data['full_name'])) $errors['full_name'] = 'Nama Lengkap wajib diisi.';
        elseif (strlen($data['full_name']) > 100) $errors['full_name'] = 'Nama Lengkap maksimal 100 karakter.';

        return $errors;
    }
}

if (!function_exists('validate_profile_input')) {
    function validate_profile_input($data, $user_id_to_edit, $pdo_conn = null) {
        $errors = [];
        // Username tidak boleh diubah oleh pengguna sendiri di form profil
        if (empty($data['email'])) $errors['email'] = 'Email wajib diisi.';
        elseif (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) $errors['email'] = 'Format email tidak valid.';
        elseif (strlen($data['email']) > 100) $errors['email'] = 'Email maksimal 100 karakter.';
        else {
            // Cek duplikasi email, kecuali untuk user saat ini
            $stmt_check_email = $pdo_conn->prepare("SELECT id FROM portal_users WHERE email = :email AND id != :id");
            $stmt_check_email->execute([':email' => $data['email'], ':id' => $user_id_to_edit]);
            if ($stmt_check_email->fetch()) {
                $errors['email'] = 'Email "' . esc_html($data['email']) . '" sudah digunakan oleh pengguna lain.';
            }
        }
        
        // Password hanya divalidasi jika diisi (artinya ingin diubah)
        if (!empty($data['current_password']) || !empty($data['new_password']) || !empty($data['confirm_new_password'])) {
            if (empty($data['current_password'])) {
                $errors['current_password'] = 'Password saat ini wajib diisi jika ingin mengubah password.';
            } else {
                // Verifikasi password saat ini
                $stmt_user = $pdo_conn->prepare("SELECT password_hash FROM portal_users WHERE id = :id");
                $stmt_user->execute([':id' => $user_id_to_edit]);
                $user = $stmt_user->fetch();
                if (!$user || !password_verify($data['current_password'], $user['password_hash'])) {
                    $errors['current_password'] = 'Password saat ini salah.';
                }
            }
            if (empty($data['new_password'])) $errors['new_password'] = 'Password baru wajib diisi.';
            elseif (strlen($data['new_password']) < 6) $errors['new_password'] = 'Password baru minimal 6 karakter.';
            if ($data['new_password'] !== $data['confirm_new_password']) $errors['confirm_new_password'] = 'Konfirmasi password baru tidak cocok.';
        }
        
        if (empty($data['full_name'])) $errors['full_name'] = 'Nama Lengkap wajib diisi.';
        elseif (strlen($data['full_name']) > 100) $errors['full_name'] = 'Nama Lengkap maksimal 100 karakter.';

        return $errors;
    }
}
// --- End Fungsi Validasi Input ---


// --- Logika Utama ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validasi CSRF sudah dilakukan di atas untuk POST
    
    $id_to_process = isset($_POST['id']) ? (int)$_POST['id'] : null;

    if ($action === 'create' || $action === 'edit') {
        if ($_SESSION['role'] !== 'admin') { /* ... (sudah dihandle di atas) ... */ }

        $form_data_input_admin = [
            'username' => trim($_POST['username'] ?? ''),
            'email' => trim($_POST['email'] ?? ''),
            'password' => $_POST['password'] ?? '', // Akan di-hash jika diisi
            'confirm_password' => $_POST['confirm_password'] ?? '',
            'full_name' => trim($_POST['full_name'] ?? ''),
            'role' => $_POST['role'] ?? '',
            'is_active' => isset($_POST['is_active']) ? 1 : 0,
        ];
        
        $is_edit_action_admin = ($action === 'edit');
        $validation_errors_admin = validate_user_input_admin($form_data_input_admin, $is_edit_action_admin, $id_to_process, $pdo);
        
        $redirect_on_error_admin = $is_edit_action_admin ? admin_url('modules/users/edit.php?id=' . $id_to_process) : admin_url('modules/users/create.php');

        if (!empty($validation_errors_admin)) {
            $response['message'] = 'Validasi gagal. Harap periksa input Anda.';
            $response['errors'] = $validation_errors_admin;
            if (!$is_ajax) {
                $_SESSION['form_errors_user_admin'] = $validation_errors_admin;
                $_SESSION['form_data_user_admin'] = $form_data_input_admin;
            }
            // Keluar dari switch, akan dihandle di akhir
        } else {
            // Proses ke database
            try {
                if ($action === 'create') {
                    $password_hash_admin = password_hash($form_data_input_admin['password'], PASSWORD_DEFAULT);
                    $sql_admin = "INSERT INTO portal_users (username, email, password_hash, full_name, role, is_active, created_at, updated_at) 
                                  VALUES (:username, :email, :password_hash, :full_name, :role, :is_active, NOW(), NOW())";
                    $stmt_admin = $pdo->prepare($sql_admin);
                    $stmt_admin->bindParam(':password_hash', $password_hash_admin);
                } else { // edit
                    if (!$id_to_process) { throw new Exception("ID pengguna tidak valid untuk edit."); }
                    $sql_admin = "UPDATE portal_users SET username = :username, email = :email, full_name = :full_name, role = :role, is_active = :is_active";
                    if (!empty($form_data_input_admin['password'])) { // Jika password diubah
                        $password_hash_admin = password_hash($form_data_input_admin['password'], PASSWORD_DEFAULT);
                        $sql_admin .= ", password_hash = :password_hash";
                    }
                    $sql_admin .= ", updated_at = NOW() WHERE id = :id";
                    $stmt_admin = $pdo->prepare($sql_admin);
                    if (!empty($form_data_input_admin['password'])) {
                        $stmt_admin->bindParam(':password_hash', $password_hash_admin);
                    }
                    $stmt_admin->bindParam(':id', $id_to_process, PDO::PARAM_INT);
                }

                $stmt_admin->bindParam(':username', $form_data_input_admin['username']);
                $stmt_admin->bindParam(':email', $form_data_input_admin['email']);
                $stmt_admin->bindParam(':full_name', $form_data_input_admin['full_name']);
                $stmt_admin->bindParam(':role', $form_data_input_admin['role']);
                $stmt_admin->bindParam(':is_active', $form_data_input_admin['is_active'], PDO::PARAM_INT);

                if ($stmt_admin->execute()) {
                    $response['success'] = true;
                    $response['message'] = 'Pengguna berhasil ' . ($action === 'create' ? 'dibuat.' : 'diperbarui.');
                    $response['redirect_url'] = $module_redirect_url_base; // Redirect ke daftar pengguna
                } else {
                    $errorInfoAdmin = $stmt_admin->errorInfo();
                    $response['message'] = 'Gagal menyimpan pengguna ke database. ' . ($errorInfoAdmin[2] ?? '');
                    error_log("DB Error saving user (admin, action: {$action}): " . print_r($errorInfoAdmin, true));
                }
            } catch (\PDOException $e_admin_save) {
                error_log("PDOException saving user (admin, action: {$action}): " . $e_admin_save->getMessage());
                $response['message'] = 'Terjadi kesalahan database saat menyimpan pengguna.';
                if ($is_ajax) http_response_code(500);
            } catch (\Exception $e_general_admin_save) {
                 error_log("Exception saving user (admin, action: {$action}): " . $e_general_admin_save->getMessage());
                $response['message'] = 'Terjadi kesalahan umum: ' . $e_general_admin_save->getMessage();
                if ($is_ajax) http_response_code(500);
            }
        }

    } elseif ($action === 'edit_profile') {
        // ID pengguna yang diedit adalah pengguna yang sedang login
        $id_to_process_profile = $_SESSION['user_id']; 

        $form_data_input_profile = [
            'email' => trim($_POST['email'] ?? ''),
            'full_name' => trim($_POST['full_name'] ?? ''),
            'current_password' => $_POST['current_password'] ?? '',
            'new_password' => $_POST['new_password'] ?? '',
            'confirm_new_password' => $_POST['confirm_new_password'] ?? '',
        ];
        
        $validation_errors_profile = validate_profile_input($form_data_input_profile, $id_to_process_profile, $pdo);
        $redirect_on_error_profile = admin_url('modules/users/edit_profile.php');

        if (!empty($validation_errors_profile)) {
            $response['message'] = 'Validasi gagal. Harap periksa input Anda.';
            $response['errors'] = $validation_errors_profile;
            if (!$is_ajax) {
                $_SESSION['form_errors_user_profile'] = $validation_errors_profile;
                $_SESSION['form_data_user_profile'] = $form_data_input_profile; // Simpan tanpa password
            }
        } else {
            try {
                $sql_profile = "UPDATE portal_users SET email = :email, full_name = :full_name";
                $params_profile = [
                    ':email' => $form_data_input_profile['email'],
                    ':full_name' => $form_data_input_profile['full_name'],
                    ':id' => $id_to_process_profile
                ];

                if (!empty($form_data_input_profile['new_password'])) {
                    // Password sudah divalidasi di validate_profile_input (termasuk current_password)
                    $new_password_hash_profile = password_hash($form_data_input_profile['new_password'], PASSWORD_DEFAULT);
                    $sql_profile .= ", password_hash = :password_hash";
                    $params_profile[':password_hash'] = $new_password_hash_profile;
                }
                $sql_profile .= ", updated_at = NOW() WHERE id = :id";
                
                $stmt_profile = $pdo->prepare($sql_profile);
                if ($stmt_profile->execute($params_profile)) {
                    // Update session jika nama berubah
                    $_SESSION['full_name'] = $form_data_input_profile['full_name'];
                    // Jika email juga disimpan di session dan boleh diubah, update juga
                    // $_SESSION['email'] = $form_data_input_profile['email'];

                    $response['success'] = true;
                    $response['message'] = 'Profil Anda berhasil diperbarui.';
                    $response['updated_full_name'] = $form_data_input_profile['full_name']; 
                } else {
                    $errorInfoProfile = $stmt_profile->errorInfo();
                    $response['message'] = 'Gagal memperbarui profil. ' . ($errorInfoProfile[2] ?? '');
                    error_log("DB Error updating profile: " . print_r($errorInfoProfile, true));
                }
            } catch (\PDOException $e_profile_save) {
                error_log("PDOException updating profile: " . $e_profile_save->getMessage());
                $response['message'] = 'Terjadi kesalahan database saat memperbarui profil.';
                if ($is_ajax) http_response_code(500);
            }
        }

    } else {
        $response['message'] = 'Aksi POST tidak dikenal atau tidak diizinkan.';
        if ($is_ajax) http_response_code(400);
    }

} elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && $action === 'delete') {
    // Validasi CSRF sudah dilakukan di atas untuk GET delete
    if ($_SESSION['role'] !== 'admin') { /* ... (sudah dihandle di atas) ... */ }

    $id_to_delete = isset($_GET['id']) ? (int)$_GET['id'] : 0;

    if ($id_to_delete <= 0) {
        $response['message'] = 'ID pengguna tidak valid untuk dihapus.';
        if ($is_ajax) http_response_code(400);
    } elseif ($id_to_delete == $_SESSION['user_id']) {
        $response['message'] = 'Anda tidak dapat menghapus akun Anda sendiri.';
        if ($is_ajax) http_response_code(403);
    } elseif ($id_to_delete == 1) { // Asumsi ID 1 adalah super admin yang tidak boleh dihapus
        $response['message'] = 'Pengguna utama (ID 1) tidak dapat dihapus.';
        if ($is_ajax) http_response_code(403);
    } else {
        try {
            $stmt_delete_user = $pdo->prepare("DELETE FROM portal_users WHERE id = :id");
            $stmt_delete_user->bindParam(':id', $id_to_delete, PDO::PARAM_INT);
            if ($stmt_delete_user->execute()) {
                if ($stmt_delete_user->rowCount() > 0) {
                    $response['success'] = true;
                    $response['message'] = 'Pengguna berhasil dihapus.';
                } else {
                    $response['message'] = 'Pengguna tidak ditemukan atau sudah dihapus.';
                    // Tidak perlu set success = false jika rowCount = 0 karena operasi delete berhasil tapi tidak ada yang diubah
                }
            } else {
                $errorInfoDeleteUser = $stmt_delete_user->errorInfo();
                $response['message'] = 'Gagal menghapus pengguna. ' . ($errorInfoDeleteUser[2] ?? '');
                error_log("DB Error deleting user: " . print_r($errorInfoDeleteUser, true));
            }
        } catch (\PDOException $e_delete_user) {
            error_log("PDOException deleting user: " . $e_delete_user->getMessage());
            $response['message'] = 'Terjadi kesalahan database saat menghapus pengguna.';
            if ($is_ajax) http_response_code(500);
        }
    }
} else {
    $response['message'] = 'Permintaan tidak valid atau aksi tidak didukung.';
    if ($is_ajax && !empty($action)) http_response_code(400); // Bad request jika ada action tapi tidak cocok
    elseif ($is_ajax) http_response_code(405); // Method Not Allowed jika tidak ada action
}


// Kirim respons atau lakukan redirect
if ($is_ajax) {
    if (!headers_sent()) {
        if ($response['success'] && http_response_code() >= 400) {
             // Jika success true tapi ada http error code, override jadi 200
             http_response_code(200);
        } elseif (!$response['success'] && http_response_code() < 400 ) {
            // Jika gagal tapi belum ada kode error spesifik, set ke 400 atau 422 (jika ada errors) atau 500
             http_response_code(isset($response['errors']) ? 422 : (isset($response['critical_error']) ? 500 : 400));
        }
    }
    echo json_encode($response);
} else {
    // Redirect untuk non-AJAX
    $redirect_url = $module_redirect_url_base; // Default
    if ($response['success']) {
        set_flash_message('success', $response['message']);
        if (!empty($response['redirect_url'])) { // Jika ada URL redirect spesifik dari add/edit sukses
            $redirect_url = $response['redirect_url'];
        }
    } else {
        set_flash_message('danger', $response['message']);
        // Jika validasi gagal, kembali ke form sebelumnya dengan data dan error
        if (($action === 'create' || $action === 'edit') && !empty($response['errors'])) {
            // $_SESSION['form_errors_user_admin'] dan $_SESSION['form_data_user_admin'] sudah di-set
            $id_for_redirect_fail = $_POST['id'] ?? ($action === 'edit' ? $id_to_process : null);
            if ($id_for_redirect_fail) {
                $redirect_url = admin_url('modules/users/edit.php?id=' . $id_for_redirect_fail);
            } else {
                $redirect_url = admin_url('modules/users/create.php');
            }
        } elseif ($action === 'edit_profile' && !empty($response['errors'])) {
            // $_SESSION['form_errors_user_profile'] dan $_SESSION['form_data_user_profile'] sudah di-set
            $redirect_url = admin_url('modules/users/edit_profile.php');
        }
    }
    
    if (!headers_sent()){
        header('Location: ' . $redirect_url);
    } else {
        error_log("Headers already sent in process_user.php, cannot redirect to: " . $redirect_url . ". Last error: " . json_encode(error_get_last()));
        echo "<p>Operasi selesai. <a href='" . esc_html($redirect_url) . "'>Kembali</a></p>";
        if (!$response['success'] && isset($response['message'])) {
            echo "<p style='color:red;'>Pesan Error: " . esc_html($response['message']) . "</p>";
        }
    }
}
exit;
?>