<?php
// admin/modules/menus/process_menu_item.php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Define project root if not already defined
if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__, 3));
}
require_once PROJECT_ROOT . '/includes/db_connect.php';
require_once PROJECT_ROOT . '/admin/includes/functions.php';

// --- HELPER FUNCTIONS ---

// Function to check if the request is an AJAX request
function is_ajax_request() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
}

// Function to build a hierarchical menu tree from a flat array
function build_menu_tree(array &$elements, $parentId = null) {
    $branch = [];
    foreach ($elements as &$element) {
        if ($element['parent_id'] == $parentId) {
            $children = build_menu_tree($elements, $element['id']);
            if ($children) {
                $element['children'] = $children;
            }
            $branch[] = $element;
        }
    }
    return $branch;
}

// Function to get all descendant IDs of a given parent item (for validation)
function get_all_descendant_ids($pdo, $itemId) {
    $descendants = [];
    $items_to_check = [$itemId];
    while (!empty($items_to_check)) {
        $currentItemId = array_shift($items_to_check);
        
        $stmt = $pdo->prepare("SELECT id FROM portal_menu_items WHERE parent_id = ?");
        $stmt->execute([$currentItemId]);
        $children = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        if (!empty($children)) {
            $descendants = array_merge($descendants, $children);
            $items_to_check = array_merge($items_to_check, $children);
        }
    }
    return $descendants;
}


// --- INITIAL CHECKS & SETUP ---

$is_ajax = is_ajax_request();
if ($is_ajax) {
    header('Content-Type: application/json');
}

// Security: Ensure user is a logged-in admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Akses ditolak. Anda harus login sebagai admin.']);
    exit;
}

// Security: Check for database connection
if (!isset($pdo)) {
    http_response_code(503);
    echo json_encode(['success' => false, 'message' => 'Koneksi database tidak tersedia.']);
    exit;
}

// Determine action from POST or GET
$action = $_POST['action'] ?? $_GET['action'] ?? null;

// --- CSRF VALIDATION for state-changing actions ---
$state_changing_actions = ['add_item', 'edit_item', 'delete_item', 'update_order'];
if (in_array($action, $state_changing_actions)) {
    $token = $_POST['csrf_token'] ?? $_GET['csrf_token'] ?? null;
    if (!validate_csrf_token($token)) {
        http_response_code(403);
        echo json_encode([
            'success' => false, 
            'message' => 'Sesi tidak valid atau telah kedaluwarsa. Silakan muat ulang halaman.',
            'new_csrf_token' => generate_csrf_token() // Provide a new token
        ]);
        exit;
    }
}

// Prepare a standard response structure
$response = [
    'success' => false, 
    'message' => 'Aksi tidak valid atau tidak dikenal.', 
    'new_csrf_token' => generate_csrf_token()
];


// --- ACTION ROUTER ---

try {
    switch ($action) {
        
        // ACTION: Get all menu items in a hierarchical structure
        case 'get_menu_items_hierarchical':
            $menu_id = isset($_GET['menu_id']) ? (int)$_GET['menu_id'] : 0;
            if ($menu_id > 0) {
                $stmt = $pdo->prepare("SELECT * FROM portal_menu_items WHERE menu_id = :menu_id ORDER BY parent_id, item_order");
                $stmt->execute([':menu_id' => $menu_id]);
                $all_items = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $hierarchical_items = build_menu_tree($all_items);
                
                $response['success'] = true;
                $response['message'] = 'Data item menu berhasil dimuat.';
                $response['menu_items_hierarchical'] = $hierarchical_items;
            } else {
                $response['message'] = 'ID Menu tidak valid.';
            }
            break;

        // ACTION: Add a new menu item
        case 'add_item':
            $menu_id = isset($_POST['menu_id']) ? (int)$_POST['menu_id'] : 0;
            $title = trim($_POST['title'] ?? '');
            $url = trim($_POST['url'] ?? '#');
            
            if ($menu_id <= 0 || empty($title) || empty($url)) {
                $response['message'] = 'Data tidak lengkap. Judul dan URL wajib diisi.';
                break;
            }

            $sql = "INSERT INTO portal_menu_items (menu_id, title, url, target, parent_id, item_order, css_class, icon_class, created_at, updated_at) 
                    VALUES (:menu_id, :title, :url, :target, :parent_id, :item_order, :css_class, :icon_class, NOW(), NOW())";
            $stmt = $pdo->prepare($sql);
            
            $parent_id = !empty($_POST['parent_id']) ? (int)$_POST['parent_id'] : null;
            
            $stmt->execute([
                ':menu_id' => $menu_id,
                ':title' => $title,
                ':url' => $url,
                ':target' => $_POST['target'] ?? '_self',
                ':parent_id' => $parent_id,
                ':item_order' => (int)($_POST['item_order'] ?? 0),
                ':css_class' => trim($_POST['css_class'] ?? ''),
                ':icon_class' => trim($_POST['icon_class'] ?? '')
            ]);
            
            $response['success'] = true;
            $response['message'] = 'Item menu berhasil ditambahkan.';
            break;

        // ACTION: Edit an existing menu item
        case 'edit_item':
            $item_id = isset($_POST['item_id']) ? (int)$_POST['item_id'] : 0;
            $title = trim($_POST['title'] ?? '');
            $url = trim($_POST['url'] ?? '#');
            $parent_id = !empty($_POST['parent_id']) ? (int)$_POST['parent_id'] : null;

            if ($item_id <= 0 || empty($title) || empty($url)) {
                $response['message'] = 'Data tidak lengkap atau ID item tidak valid.';
                break;
            }

            // CRITICAL: Prevent circular dependency
            if ($item_id === $parent_id) {
                $response['message'] = 'Item tidak bisa menjadi induk dari dirinya sendiri.';
                break;
            }
            if ($parent_id) {
                $descendants = get_all_descendant_ids($pdo, $item_id);
                if (in_array($parent_id, $descendants)) {
                    $response['message'] = 'Tidak dapat menjadikan turunan sebagai induk (circular dependency).';
                    break;
                }
            }

            $sql = "UPDATE portal_menu_items SET 
                        title = :title, url = :url, target = :target, parent_id = :parent_id, 
                        item_order = :item_order, css_class = :css_class, icon_class = :icon_class, 
                        updated_at = NOW()
                    WHERE id = :id";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                ':title' => $title,
                ':url' => $url,
                ':target' => $_POST['target'] ?? '_self',
                ':parent_id' => $parent_id,
                ':item_order' => (int)($_POST['item_order'] ?? 0),
                ':css_class' => trim($_POST['css_class'] ?? ''),
                ':icon_class' => trim($_POST['icon_class'] ?? ''),
                ':id' => $item_id
            ]);

            $response['success'] = true;
            $response['message'] = 'Item menu berhasil diperbarui.';
            break;

        // ACTION: Delete a menu item and all its children (relies on ON DELETE CASCADE)
        case 'delete_item':
            $item_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            if ($item_id > 0) {
                // Assuming the foreign key for `parent_id` has `ON DELETE CASCADE`,
                // deleting the parent will automatically delete all its children recursively.
                $stmt = $pdo->prepare("DELETE FROM portal_menu_items WHERE id = :id");
                $stmt->execute([':id' => $item_id]);
                
                if ($stmt->rowCount() > 0) {
                    $response['success'] = true;
                    $response['message'] = 'Item menu dan semua sub-itemnya berhasil dihapus.';
                } else {
                    $response['message'] = 'Item menu tidak ditemukan atau sudah dihapus.';
                }
            } else {
                $response['message'] = 'ID item tidak valid.';
            }
            break;

        // ACTION: Update the order and parentage of items from drag-and-drop
        case 'update_order':
            $menu_id = isset($_POST['menu_id']) ? (int)$_POST['menu_id'] : 0;
            $order_data_json = $_POST['order_data'] ?? '[]';
            $order_data = json_decode($order_data_json, true);

            if ($menu_id <= 0 || !is_array($order_data)) {
                $response['message'] = 'Data urutan tidak valid atau ID menu tidak ada.';
                break;
            }

            $pdo->beginTransaction();
            
            $sql = "UPDATE portal_menu_items SET parent_id = :parent_id, item_order = :item_order, updated_at = NOW() WHERE id = :id AND menu_id = :menu_id";
            $stmt = $pdo->prepare($sql);
            
            foreach ($order_data as $item) {
                if (!isset($item['id']) || !isset($item['item_order'])) continue; // Skip invalid entries
                $stmt->execute([
                    ':parent_id' => $item['parent_id'] ?: null,
                    ':item_order' => (int)$item['item_order'],
                    ':id' => (int)$item['id'],
                    ':menu_id' => $menu_id
                ]);
            }
            
            $pdo->commit();
            $response['success'] = true;
            $response['message'] = 'Urutan menu berhasil disimpan.';
            break;
    }
} catch (PDOException $e) {
    // Rollback transaction if it's active
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("Menu Item Processing Error: " . $e->getMessage());
    http_response_code(500);
    $response['message'] = 'Terjadi kesalahan pada server. Silakan coba lagi.';
}

// Final output
echo json_encode($response);
exit;
