<?php
// admin/modules/faculties/process_faculty.php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__, 3));
}
require_once PROJECT_ROOT . '/includes/db_connect.php';
require_once PROJECT_ROOT . '/admin/includes/functions.php';

function is_ajax_request_faculty() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
}
$is_ajax = is_ajax_request_faculty();
if ($is_ajax) {
    header('Content-Type: application/json');
}

$module_redirect_url_base = 'index.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    if ($is_ajax) { http_response_code(403); echo json_encode(['success' => false, 'message' => 'Akses ditolak.']); exit; }
    set_flash_message('danger', 'Anda tidak memiliki izin untuk aksi ini.'); header('Location: ../../login.php'); exit;
}

if (!isset($pdo)) {
    if ($is_ajax) { http_response_code(503); echo json_encode(['success' => false, 'message' => 'Koneksi database gagal.']); exit; }
    set_flash_message('danger', 'Koneksi database gagal.'); header('Location: ' . $module_redirect_url_base); exit;
}

// --- Validation Function ---
function validate_faculty_input($data, $is_edit = false, $faculty_id_to_edit = null, $pdo_conn = null) {
    $errors = [];
    if (empty($data['name'])) { $errors['name'] = 'Nama fakultas wajib diisi.'; }
    elseif (strlen($data['name']) > 150) { $errors['name'] = 'Nama fakultas maksimal 150 karakter.'; }
    else {
        $sql_check_name = "SELECT id FROM portal_faculties WHERE name = :name";
        $params_check = [':name' => $data['name']];
        if ($is_edit && $faculty_id_to_edit) { $sql_check_name .= " AND id != :id"; $params_check[':id'] = $faculty_id_to_edit;}
        $stmt_check = $pdo_conn->prepare($sql_check_name); $stmt_check->execute($params_check);
        if ($stmt_check->fetch()) { $errors['name'] = 'Nama fakultas "' . esc_html($data['name']) . '" sudah ada.';}
    }
    if (!empty($data['slug'])) {
        if (!preg_match('/^[a-z0-9]+(?:-[a-z0-9]+)*$/', $data['slug'])) { $errors['slug'] = 'Slug hanya boleh berisi huruf kecil, angka, dan tanda hubung (-).'; }
        elseif (strlen($data['slug']) > 170) { $errors['slug'] = 'Slug maksimal 170 karakter.';}
        else {
            $sql_check_slug = "SELECT id FROM portal_faculties WHERE slug = :slug";
            $params_check_slug = [':slug' => $data['slug']];
            if ($is_edit && $faculty_id_to_edit) { $sql_check_slug .= " AND id != :id"; $params_check_slug[':id'] = $faculty_id_to_edit;}
            $stmt_check_slug = $pdo_conn->prepare($sql_check_slug); $stmt_check_slug->execute($params_check_slug);
            if ($stmt_check_slug->fetch()) { $errors['slug'] = 'Slug "' . esc_html($data['slug']) . '" sudah digunakan.';}
        }
    } elseif ($is_edit && empty($data['slug'])) { $errors['slug'] = 'Slug wajib diisi saat edit jika dikosongkan.';}
    if (!empty($data['dean_name']) && strlen($data['dean_name']) > 100) { $errors['dean_name'] = 'Nama dekan maksimal 100 karakter.';}
    if (!empty($data['website_url'])) {
        if (!filter_var($data['website_url'], FILTER_VALIDATE_URL)) { $errors['website_url'] = 'Format URL website tidak valid.';}
        elseif (strlen($data['website_url']) > 255) { $errors['website_url'] = 'URL Website maksimal 255 karakter.';}
    }
    if (!isset($data['faculty_order']) || !is_numeric($data['faculty_order']) || (int)$data['faculty_order'] < 0) { $errors['faculty_order'] = 'Urutan fakultas harus berupa angka non-negatif.';}
    return $errors;
}


if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        if ($is_ajax) { http_response_code(403); echo json_encode(['success' => false, 'message' => 'Kesalahan CSRF.']); exit; }
        set_flash_message('danger', 'Kesalahan CSRF token.');
        $action_redirect = $_POST['action'] ?? 'create'; $id_redirect = $_POST['id'] ?? null;
        header('Location: ' . ($action_redirect === 'edit' && $id_redirect ? 'edit.php?id=' . $id_redirect : 'create.php')); exit;
    }

    $action = $_POST['action'] ?? null;
    $faculty_id = isset($_POST['id']) ? (int)$_POST['id'] : null;
    $redirect_on_error_url = ($action === 'edit' && $faculty_id ? 'edit.php?id=' . $faculty_id : 'create.php');

    $form_data_input = [
        'name' => trim($_POST['name'] ?? ''),
        'slug' => trim($_POST['slug'] ?? ''),
        'description_html' => $_POST['description_html'] ?? null,
        'dean_name' => trim($_POST['dean_name'] ?? ''),
        'website_url' => trim($_POST['website_url'] ?? ''),
        'faculty_order' => isset($_POST['faculty_order']) ? (int)$_POST['faculty_order'] : 0,
    ];

    if ($action === 'create' && empty($form_data_input['slug'])) {
        $form_data_input['slug'] = generate_slug($form_data_input['name']);
        if(empty($form_data_input['slug'])) $form_data_input['slug'] = 'fakultas-' . time(); // Fallback
    }

    $errors = validate_faculty_input($form_data_input, ($action === 'edit'), $faculty_id, $pdo);

    if (!empty($errors)) {
        if ($is_ajax) {
            http_response_code(422); echo json_encode(['success' => false, 'message' => 'Validasi gagal.', 'errors' => $errors]);
        } else {
            $_SESSION['form_errors'] = $errors; $_SESSION['form_data'] = $form_data_input;
            set_flash_message('danger', 'Gagal menyimpan fakultas.');
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    }

    try {
        if ($action === 'create') {
            $sql = "INSERT INTO portal_faculties (name, slug, description_html, dean_name, website_url, faculty_order, created_at, updated_at)
                    VALUES (:name, :slug, :description_html, :dean_name, :website_url, :faculty_order, NOW(), NOW())";
            $stmt = $pdo->prepare($sql);
        } elseif ($action === 'edit' && $faculty_id) {
            $sql = "UPDATE portal_faculties SET name = :name, slug = :slug, description_html = :description_html,
                    dean_name = :dean_name, website_url = :website_url, faculty_order = :faculty_order, updated_at = NOW()
                    WHERE id = :id";
            $stmt = $pdo->prepare($sql);
            $stmt->bindParam(':id', $faculty_id, PDO::PARAM_INT);
        } else { throw new Exception("Aksi tidak valid atau ID fakultas tidak ada untuk edit."); }

        $stmt->bindParam(':name', $form_data_input['name']);
        $stmt->bindParam(':slug', $form_data_input['slug']);
        $stmt->bindParam(':description_html', $form_data_input['description_html']);
        $stmt->bindParam(':dean_name', $form_data_input['dean_name']);
        $stmt->bindParam(':website_url', $form_data_input['website_url']);
        $stmt->bindParam(':faculty_order', $form_data_input['faculty_order'], PDO::PARAM_INT);
        $stmt->execute();

        if ($is_ajax) {
            echo json_encode(['success' => true, 'message' => 'Fakultas berhasil ' . ($action === 'create' ? 'dibuat' : 'diperbarui') . '.', 'redirect_url' => $module_redirect_url_base]);
        } else {
            set_flash_message('success', 'Fakultas berhasil ' . ($action === 'create' ? 'dibuat' : 'diperbarui') . '.');
            header('Location: ' . $module_redirect_url_base);
        }
        exit;

    } catch (PDOException | Exception $e) {
        error_log("Error processing faculty ($action): " . $e->getMessage());
        if ($is_ajax) {
            http_response_code(500); echo json_encode(['success' => false, 'message' => 'Terjadi kesalahan server: ' . $e->getMessage()]);
        } else {
            set_flash_message('danger', 'Terjadi kesalahan server: ' . $e->getMessage());
            $_SESSION['form_data'] = $form_data_input;
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    }

} elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'delete') {
    if (!isset($_GET['csrf_token']) || !validate_csrf_token($_GET['csrf_token'])) {
        if ($is_ajax) { http_response_code(403); echo json_encode(['success' => false, 'message' => 'Kesalahan CSRF.']); exit; }
        set_flash_message('danger', 'Kesalahan CSRF token.'); header('Location: ' . $module_redirect_url_base); exit;
    }

    $faculty_id_to_delete = isset($_GET['id']) ? (int)$_GET['id'] : 0;
    if ($faculty_id_to_delete <= 0) {
        if ($is_ajax) { http_response_code(400); echo json_encode(['success' => false, 'message' => 'ID Fakultas tidak valid.']); exit; }
        set_flash_message('danger', 'ID fakultas tidak valid.'); header('Location: ' . $module_redirect_url_base); exit;
    }

    try {
        $stmt_check_prodi = $pdo->prepare("SELECT COUNT(*) FROM portal_study_programs WHERE faculty_id = :faculty_id");
        $stmt_check_prodi->bindParam(':faculty_id', $faculty_id_to_delete, PDO::PARAM_INT);
        $stmt_check_prodi->execute();
        $prodi_count = $stmt_check_prodi->fetchColumn();
        $additional_message = "";
        // Asumsi foreign key ON DELETE SET NULL, jadi penghapusan tetap bisa dilakukan
        if ($prodi_count > 0) {
            $additional_message = ' Afiliasi fakultas untuk ' . $prodi_count . ' program studi terkait akan diatur ulang (faculty_id akan menjadi NULL).';
        }

        $stmt_delete = $pdo->prepare("DELETE FROM portal_faculties WHERE id = :id");
        $stmt_delete->bindParam(':id', $faculty_id_to_delete, PDO::PARAM_INT);
        $stmt_delete->execute();

        if ($stmt_delete->rowCount() > 0) {
            if ($is_ajax) { echo json_encode(['success' => true, 'message' => 'Fakultas berhasil dihapus.' . $additional_message]); }
            else { set_flash_message('success', 'Fakultas berhasil dihapus.' . $additional_message); header('Location: ' . $module_redirect_url_base); }
        } else {
            $fail_msg = 'Fakultas tidak ditemukan atau tidak dapat dihapus.';
            if ($is_ajax) { echo json_encode(['success' => false, 'message' => $fail_msg]); }
            else { set_flash_message('warning', $fail_msg); header('Location: ' . $module_redirect_url_base); }
        }
    } catch (PDOException $e) {
        error_log("Error deleting faculty: " . $e->getMessage());
        $error_message_user = 'Gagal menghapus fakultas: Kesalahan database.';
        if ($e->getCode() == '23000') { // Jika ada constraint RESTRICT yang tidak terduga
             $error_message_user = 'Gagal menghapus fakultas. Pastikan tidak ada program studi yang masih terafiliasi jika tidak ada aturan ON DELETE SET NULL.';
        }
        if ($is_ajax) { http_response_code(500); echo json_encode(['success' => false, 'message' => $error_message_user]); }
        else { set_flash_message('danger', $error_message_user); header('Location: ' . $module_redirect_url_base); }
    }
    exit;
} else {
    if ($is_ajax) { http_response_code(400); echo json_encode(['success' => false, 'message' => 'Permintaan tidak valid.']); }
    else { set_flash_message('danger', 'Permintaan tidak valid.'); header('Location: ' . $module_redirect_url_base); }
    exit;
}
?>