<?php
/**
 * process_event.php - VERSI LENGKAP DENGAN FUNGSI DELETE
 * Menangani pembuatan (create), pembaruan (edit), dan penghapusan (delete) data acara.
 */

// 1. Muat Konfigurasi Inti
require_once dirname(__DIR__, 3) . '/config.php';

// 2. Tentukan Jenis Permintaan
$is_ajax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
if ($is_ajax) {
    header('Content-Type: application/json');
}

// 3. Otentikasi & Otorisasi
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['admin', 'editor'])) {
    $errorMessage = 'Akses ditolak. Anda tidak memiliki izin untuk melakukan aksi ini.';
    if ($is_ajax) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => $errorMessage]);
    } else {
        set_flash_message('danger', $errorMessage);
        header('Location: ' . admin_url('login.php'));
    }
    exit;
}

// 4. Proses Berdasarkan Metode HTTP
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    process_post_request($pdo, $is_ajax);
} elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'delete') {
    // Rute untuk menangani permintaan hapus dari AJAX
    process_delete_request($pdo, $is_ajax);
} else {
    $errorMessage = 'Metode permintaan tidak valid.';
    if ($is_ajax) {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => $errorMessage]);
    } else {
        set_flash_message('danger', $errorMessage);
        header('Location: index.php');
    }
    exit;
}

/**
 * Fungsi untuk memproses permintaan POST (Create & Edit)
 */
function process_post_request($pdo, $is_ajax) {
    // [ FUNGSI UNTUK CREATE DAN EDIT TETAP SAMA SEPERTI SEBELUMNYA ]
    // (Kode di bawah ini adalah salinan dari versi sebelumnya yang sudah benar)
    
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        http_response_code(403); echo json_encode(['success' => false, 'message' => 'Sesi tidak valid.']); exit;
    }

    $input = [
        'id'               => isset($_POST['id']) ? (int)$_POST['id'] : null,
        'action'           => $_POST['action'] ?? 'create',
        'title'            => trim($_POST['title'] ?? ''),
        'slug'             => trim($_POST['slug'] ?? ''),
        'description_html' => $_POST['description_html'] ?? '',
        'start_datetime'   => trim($_POST['start_datetime'] ?? ''),
        'end_datetime'     => !empty($_POST['end_datetime']) ? trim($_POST['end_datetime']) : null,
        'location'         => trim($_POST['location'] ?? ''),
        'organizer'        => trim($_POST['organizer'] ?? ''),
        'contact_person'   => trim($_POST['contact_person'] ?? ''),
        'contact_email'    => trim($_POST['contact_email'] ?? ''),
        'contact_phone'    => trim($_POST['contact_phone'] ?? ''),
        'event_url'        => trim($_POST['event_url'] ?? ''),
        'is_published'     => isset($_POST['is_published']) ? (int)$_POST['is_published'] : 0,
        'old_image'        => $_POST['old_featured_image_url'] ?? null,
        'user_id'          => $_SESSION['user_id'],
    ];

    if (empty($input['slug'])) {
        $input['slug'] = generate_slug($input['title']);
    }

    $errors = [];
    if (empty($input['title'])) $errors['title'] = 'Judul acara wajib diisi.';
    if (empty($input['start_datetime'])) $errors['start_datetime'] = 'Tanggal mulai wajib diisi.';

    try {
        $sql_check_slug = "SELECT id FROM portal_events WHERE slug = :slug";
        $params_check = [':slug' => $input['slug']];
        if ($input['action'] === 'edit' && $input['id']) {
            $sql_check_slug .= " AND id != :id";
            $params_check[':id'] = $input['id'];
        }
        $stmt_check = $pdo->prepare($sql_check_slug);
        $stmt_check->execute($params_check);
        if ($stmt_check->fetch()) {
            $errors['slug'] = 'Slug ini sudah digunakan.';
        }
    } catch (PDOException $e) { /* Abaikan */ }

    $input['featured_image_url'] = handle_image_upload_event_proc($input['old_image'], $errors);

    if (!empty($errors)) {
        http_response_code(422);
        echo json_encode(['success' => false, 'message' => 'Validasi gagal.', 'errors' => $errors]);
        exit;
    }

    save_event_to_db_proc($input, $pdo, $is_ajax);
}

/**
 * =================================================================
 * FUNGSI BARU: Untuk memproses permintaan DELETE
 * =================================================================
 */
function process_delete_request($pdo, $is_ajax) {
    // 1. Validasi CSRF Token
    if (!isset($_GET['csrf_token']) || !validate_csrf_token($_GET['csrf_token'])) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Aksi tidak diizinkan (sesi tidak valid).']);
        exit;
    }

    // 2. Validasi ID
    $event_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
    if ($event_id <= 0) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'ID acara tidak valid.']);
        exit;
    }

    try {
        $pdo->beginTransaction();

        // 3. Ambil path gambar sebelum dihapus dari DB
        $stmt_get_image = $pdo->prepare("SELECT featured_image_url FROM portal_events WHERE id = :id");
        $stmt_get_image->execute([':id' => $event_id]);
        $image_path = $stmt_get_image->fetchColumn();

        // 4. Hapus record dari database
        $stmt_delete = $pdo->prepare("DELETE FROM portal_events WHERE id = :id");
        $stmt_delete->execute([':id' => $event_id]);

        // 5. Cek apakah penghapusan berhasil
        if ($stmt_delete->rowCount() > 0) {
            // Jika berhasil, hapus file gambar dari server
            if (!empty($image_path) && file_exists(PROJECT_ROOT . '/' . $image_path)) {
                @unlink(PROJECT_ROOT . '/' . $image_path);
            }
            $pdo->commit();
            echo json_encode(['success' => true, 'message' => 'Acara berhasil dihapus.']);
        } else {
            $pdo->rollBack();
            http_response_code(404);
            echo json_encode(['success' => false, 'message' => 'Acara tidak ditemukan atau sudah dihapus.']);
        }
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("DATABASE ERROR on event delete: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Gagal menghapus acara karena kesalahan database.']);
    }
    exit;
}

// Helper functions (dengan nama sedikit diubah agar tidak konflik jika ada duplikat)
function handle_image_upload_event_proc($old_image_path, &$errors) {
    if (isset($_FILES['featured_image_file']) && $_FILES['featured_image_file']['error'] == UPLOAD_ERR_OK) {
        $upload_result = upload_file($_FILES['featured_image_file'], PROJECT_ROOT . '/uploads/events_featured/', ['image/jpeg', 'image/png', 'image/gif', 'image/webp'], 2 * 1024 * 1024);
        if ($upload_result['success']) {
            if (!empty($old_image_path) && file_exists(PROJECT_ROOT . '/' . $old_image_path)) { @unlink(PROJECT_ROOT . '/' . $old_image_path); }
            return 'uploads/events_featured/' . $upload_result['filename'];
        } else {
            $errors['featured_image_file'] = $upload_result['message']; return $old_image_path;
        }
    }
    return $old_image_path;
}

function save_event_to_db_proc($input, $pdo, $is_ajax) {
    $params = [
        ':title' => $input['title'], ':slug' => $input['slug'], ':description_html' => $input['description_html'], ':start_datetime' => $input['start_datetime'], ':end_datetime' => $input['end_datetime'], ':location' => $input['location'], ':organizer' => $input['organizer'], ':contact_person' => $input['contact_person'], ':contact_email' => $input['contact_email'], ':contact_phone' => $input['contact_phone'], ':event_url' => $input['event_url'], ':is_published' => $input['is_published'], ':featured_image_url' => $input['featured_image_url'], ':user_id' => $input['user_id']
    ];
    if ($input['action'] === 'create') {
        $sql = "INSERT INTO portal_events (title, slug, description_html, start_datetime, end_datetime, location, organizer, contact_person, contact_email, contact_phone, event_url, is_published, featured_image_url, created_by_user_id, created_at, updated_at) VALUES (:title, :slug, :description_html, :start_datetime, :end_datetime, :location, :organizer, :contact_person, :contact_email, :contact_phone, :event_url, :is_published, :featured_image_url, :user_id, NOW(), NOW())";
    } else {
        $sql = "UPDATE portal_events SET title = :title, slug = :slug, description_html = :description_html, start_datetime = :start_datetime, end_datetime = :end_datetime, location = :location, organizer = :organizer, contact_person = :contact_person, contact_email = :contact_email, contact_phone = :contact_phone, event_url = :event_url, is_published = :is_published, featured_image_url = :featured_image_url, updated_by_user_id = :user_id, updated_at = NOW() WHERE id = :id";
        $params[':id'] = $input['id'];
    }
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $message = 'Acara berhasil ' . ($input['action'] === 'create' ? 'dibuat' : 'diperbarui') . '.';
        echo json_encode(['success' => true, 'message' => $message, 'redirect_url' => 'index.php']);
        exit;
    } catch (PDOException $e) {
        error_log("DB ERROR: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Kesalahan database.', 'error_detail' => $e->getMessage()]);
        exit;
    }
}
?>