// admin/assets/js/admin-global.js
// ... (fungsi showAdminNotification dan confirmAdminAction seperti sebelumnya) ...
// admin/assets/js/admin-global.js
// Berisi fungsi-fungsi JavaScript global yang sering digunakan di Admin Panel

/**
 * Menampilkan notifikasi admin (toast atau alert sederhana).
 * Menggunakan Bootstrap Alert yang dimodifikasi agar muncul sebagai toast.
 *
 * @param {string} type Tipe notifikasi ('success', 'danger'/'error', 'warning', 'info').
 * @param {string} message Pesan notifikasi.
 * @param {number} [duration=5000] Durasi tampilan dalam milidetik. 0 untuk tidak hilang otomatis.
 */
function showAdminNotification(type, message, duration = 5000) {
    const notificationContainerId = 'adminGlobalNotificationArea';
    let container = document.getElementById(notificationContainerId);

    if (!container) {
        container = document.createElement('div');
        container.id = notificationContainerId;
        container.style.position = 'fixed';
        container.style.top = '80px'; // Sesuaikan dengan tinggi navbar admin + sedikit padding
        container.style.right = '20px';
        container.style.zIndex = '1090'; // Pastikan di atas elemen lain (modal Bootstrap 1050-1060)
        container.style.width = 'auto';
        container.style.maxWidth = '400px'; 
        document.body.appendChild(container);
    }

    const alertElement = document.createElement('div');
    let alertTypeClass = 'info'; // Default
    let iconClass = 'fas fa-info-circle'; // Default icon

    switch (type.toLowerCase()) {
        case 'success':
            alertTypeClass = 'success';
            iconClass = 'fas fa-check-circle';
            break;
        case 'danger':
        case 'error':
            alertTypeClass = 'danger';
            iconClass = 'fas fa-times-circle';
            break;
        case 'warning':
            alertTypeClass = 'warning';
            iconClass = 'fas fa-exclamation-triangle';
            break;
        case 'info':
            alertTypeClass = 'info';
            iconClass = 'fas fa-info-circle';
            break;
    }

    // Gunakan class Bootstrap Alert dan tambahkan beberapa style untuk membuatnya seperti toast
    alertElement.className = `alert alert-${alertTypeClass} alert-dismissible fade show shadow-sm admin-toast-notification`;
    alertElement.setAttribute('role', 'alert');
    alertElement.style.marginBottom = '10px';
    alertElement.style.display = 'flex'; // Untuk alignment ikon dan teks
    alertElement.style.alignItems = 'flex-start'; // Align ikon ke atas jika teks panjang

    alertElement.innerHTML = `
        <div style="flex-shrink: 0; margin-right: 10px; font-size: 1.2em; margin-top: 2px;">
            <i class="${iconClass}"></i>
        </div>
        <div style="flex-grow: 1;">${escapeHtmlJs(message)}</div>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close" style="margin-left: 15px;"></button>
    `;
    
    // Prepend agar notifikasi baru muncul di atas
    if (container.firstChild) {
        container.insertBefore(alertElement, container.firstChild);
    } else {
        container.appendChild(alertElement);
    }

    // Auto-dismiss notifikasi
    if (duration > 0) {
        setTimeout(() => {
            // Pastikan elemen masih ada di DOM sebelum mencoba menutupnya
            if (alertElement && alertElement.parentNode) { 
                if (typeof bootstrap !== 'undefined' && typeof bootstrap.Alert !== 'undefined') {
                    const bsAlertInstance = bootstrap.Alert.getOrCreateInstance(alertElement);
                    if (bsAlertInstance) {
                        bsAlertInstance.close();
                    } else { 
                        alertElement.remove(); // Fallback
                    }
                } else { 
                    alertElement.remove(); // Fallback jika Bootstrap JS tidak ada
                }
            }
        }, duration);
    }
}

/**
 * Helper function untuk escape HTML entities di JavaScript agar aman ditampilkan di DOM.
 * @param {*} unsafe Input yang mungkin string atau tipe lain.
 * @returns {string} String yang sudah di-escape atau representasi string dari input.
 */
function escapeHtmlJs(unsafe) {
    if (typeof unsafe !== 'string') {
        // console.warn('escapeHtmlJs called with non-string value:', unsafe);
        return String(unsafe); // Coba konversi ke string
    }
    return unsafe
         .replace(/&/g, "&")
         .replace(/</g, "<")
         .replace(/>/g, ">")
         .replace(/"/g, """)
         .replace(/'/g, "'");
}

/**
 * Menampilkan error validasi pada form dari respons AJAX.
 * @param {jQuery} formElement Objek jQuery dari elemen form.
 * @param {object} errors Objek error dengan key adalah nama field dan value adalah pesan error (bisa string atau array).
 */
function displayAdminFormErrors(formElement, errors) {
    if (!(formElement instanceof jQuery) || formElement.length === 0) {
        console.error("displayAdminFormErrors expects a valid jQuery object for formElement.");
        return;
    }

    // Hapus error sebelumnya yang spesifik untuk form ini
    formElement.find('.is-invalid').removeClass('is-invalid');
    formElement.find('.custom-form-error-message').remove();
    formElement.find('.invalid-feedback:not(.custom-form-error-message)').html(''); // Kosongkan pesan error bawaan

    let firstErrorField = null;
    let generalErrorsHtml = '';

    $.each(errors, function(key, value) {
        const field = formElement.find(`[name="${key}"], [name="${key}[]"], #${key.replace(/\./g, '\\.')}`); // Handle dot notation in key
        const errorMessage = Array.isArray(value) ? value.join('<br>') : value;
        
        if (field.length > 0) {
            const fieldElement = field.first();
            fieldElement.addClass('is-invalid');

            let feedbackElement = fieldElement.siblings('.invalid-feedback.d-block');
            if (!feedbackElement.length) feedbackElement = fieldElement.siblings('.invalid-feedback');
            if (!feedbackElement.length) { // Jika tidak ada sama sekali, buat yang baru
                const errorDiv = $('<div class="invalid-feedback custom-form-error-message d-block"></div>');
                // Penempatan feedback untuk radio/checkbox group
                if (fieldElement.is(':radio') || fieldElement.is(':checkbox')) {
                    const fieldParent = fieldElement.closest('.form-check'); // atau .form-group
                    if (fieldParent.length) fieldParent.parent().append(errorDiv);
                    else errorDiv.insertAfter(fieldElement.last());
                } else {
                     errorDiv.insertAfter(fieldElement);
                }
                feedbackElement = errorDiv;
            }
            
            feedbackElement.html(escapeHtmlJs(errorMessage)).show();

            if (!firstErrorField) firstErrorField = fieldElement;
        } else {
            // Kumpulkan error umum jika field tidak ditemukan
            generalErrorsHtml += `<p class="text-danger small mb-1">${escapeHtmlJs(key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase()))}: ${escapeHtmlJs(errorMessage)}</p>`;
        }
    });

    // Tampilkan error umum
    const generalErrorContainer = formElement.find('#generalFormErrors'); // Butuh <div id="generalFormErrors"></div> di form
    if (generalErrorsHtml && generalErrorContainer.length) {
        generalErrorContainer.html(generalErrorsHtml).show();
        if (!firstErrorField) firstErrorField = formElement; // Fokus ke form jika hanya error umum
    } else if (generalErrorsHtml) {
        // Jika container #generalFormErrors tidak ada, tampilkan sebagai notifikasi global
        showAdminNotification('danger', `Ada beberapa kesalahan input: <br>${generalErrorsHtml}`, 7000);
    }


    // Fokus ke field pertama yang error
    if (firstErrorField && typeof firstErrorField.focus === 'function') {
        setTimeout(() => {
            const fieldToFocus = firstErrorField.is(':visible') ? firstErrorField : firstErrorField.closest(':visible');
            if (fieldToFocus.length && (fieldToFocus.is('input:not([type="hidden"]), select, textarea'))) {
                fieldToFocus.focus();
                 $('html, body').animate({ scrollTop: fieldToFocus.offset().top - 120 }, 300); // Scroll dengan offset navbar
            } else if (fieldToFocus.length) {
                // Jika bukan input field, scroll ke elemennya
                 $('html, body').animate({ scrollTop: fieldToFocus.offset().top - 120 }, 300);
            }
        }, 100);
    }
}

/**
 * Mengonfirmasi sebuah aksi dengan dialog confirm standar browser.
 * @param {string} message Pesan konfirmasi.
 * @param {function} callbackIfConfirmed Fungsi yang akan dijalankan jika user mengonfirmasi.
 * @param {function} [callbackIfCancelled] (Opsional) Fungsi jika user batal.
 */
function confirmAdminAction(message, callbackIfConfirmed, callbackIfCancelled) {
    if (window.confirm(message)) { // Gunakan window.confirm agar jelas
        if (typeof callbackIfConfirmed === 'function') {
            callbackIfConfirmed();
        }
    } else {
        if (typeof callbackIfCancelled === 'function') {
            callbackIfCancelled();
        }
    }
}

/**
 * Mengupdate nilai CSRF token pada semua form yang memiliki input dengan nama 'csrf_token'.
 * Panggil ini setelah request AJAX yang mengembalikan 'new_csrf_token' dalam respons JSON.
 * @param {string} newToken Nilai CSRF token yang baru.
 */
function updateAllCsrfTokens(newToken) {
    if (newToken) {
        $('input[name="csrf_token"]').val(newToken);
        // Jika Anda menyimpan token di variabel JS global (misal `currentCsrfToken` di `manage_menu_items.js`)
        // Anda perlu cara untuk mengupdate variabel itu juga.
        // Salah satu cara adalah dengan event custom atau memanggil fungsi setter global jika ada.
        // Contoh:
        if (typeof window.currentCsrfToken !== 'undefined') {
            window.currentCsrfToken = newToken; 
            // console.log('Global currentCsrfToken updated:', window.currentCsrfToken);
        }
         if (typeof window.csrfToken !== 'undefined') { // Alternatif nama variabel global
            window.csrfToken = newToken; 
        }
    }
}

/**
 * Fungsi debounce untuk menunda eksekusi fungsi hingga user berhenti mengetik/berinteraksi.
 * @param {function} func Fungsi yang akan di-debounce.
 * @param {number} delay Waktu tunda dalam milidetik.
 * @returns {function} Fungsi yang sudah di-debounce.
 */
function debounce(func, delay) {
    let timeout;
    return function(...args) {
        const context = this;
        clearTimeout(timeout);
        timeout = setTimeout(() => func.apply(context, args), delay);
    };
}

/**
 * Fungsi throttle untuk membatasi frekuensi eksekusi fungsi.
 * @param {function} func Fungsi yang akan di-throttle.
 * @param {number} limit Batas waktu dalam milidetik antar eksekusi.
 * @returns {function} Fungsi yang sudah di-throttle.
 */
function throttle(func, limit) {
    let inThrottle;
    return function(...args) {
        const context = this;
        if (!inThrottle) {
            func.apply(context, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

// Inisialisasi komponen global saat DOM siap (jika admin-global.js dimuat setelah jQuery)
// Jika admin-global.js dimuat sebelum jQuery, pindahkan ini ke dalam blok jQuery(function($){});
// atau pastikan jQuery sudah ada.
if (typeof $ !== 'undefined') {
    $(function () {
        // Inisialisasi tooltip Bootstrap (jika digunakan)
        if (typeof bootstrap !== 'undefined' && typeof bootstrap.Tooltip !== 'undefined') {
            const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl, {
                    container: 'body' // Untuk menghindari masalah z-index dengan elemen lain
                });
            });
        }

        // Inisialisasi popover Bootstrap (jika digunakan)
        if (typeof bootstrap !== 'undefined' && typeof bootstrap.Popover !== 'undefined') {
            const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
            popoverTriggerList.map(function (popoverTriggerEl) {
                return new bootstrap.Popover(popoverTriggerEl);
            });
        }
    });
} else {
    console.warn('jQuery not loaded when admin-global.js executed. Some initializations might be skipped.');
}


// Pesan bahwa file sudah dimuat (berguna untuk debugging urutan load)
console.log('admin-global.js loaded successfully.');
function displayAdminFormErrors(formElement, errors) {
    if (!formElement || typeof errors !== 'object' || $.isEmptyObject(errors)) {
        console.warn("displayAdminFormErrors: Invalid form element or no errors provided.");
        return;
    }

    // Bersihkan error sebelumnya di form ini
    formElement.find('.is-invalid').removeClass('is-invalid');
    formElement.find('.invalid-feedback').html(''); // Kosongkan pesan error bawaan Bootstrap
    formElement.find('.custom-form-error-message').remove(); // Hapus error custom lama jika ada

    let generalErrorContainer = formElement.find('#generalFormErrors'); // Cari kontainer error umum
    if (generalErrorContainer.length) {
        generalErrorContainer.empty().hide(); // Kosongkan dan sembunyikan
    }

    $.each(errors, function(key, value) {
        const fieldName = key;
        let field = formElement.find('[name="' + fieldName + '"]');
        if (!field.length) {
            field = formElement.find('[name="' + fieldName + '[]"]'); // Coba untuk array field
        }

        if (fieldName === '_general' || fieldName.startsWith('_')) { 
            if (!generalErrorContainer.length) {
                // Jika tidak ada #generalFormErrors, buat atau tampilkan di tempat lain
                console.warn("No #generalFormErrors container found in the form to display: ", value);
                // Anda bisa tambahkan elemen error umum secara dinamis jika perlu
                // $('<div>').addClass('alert alert-danger mt-2').html(Array.isArray(value) ? value.join('<br>') : value).prependTo(formElement);
            } else {
                generalErrorContainer.append('<div>' + (Array.isArray(value) ? value.join('<br>') : value) + '</div>').show();
            }
        } else if (field.length > 0) {
            field.addClass('is-invalid');
            // Cari elemen invalid-feedback yang sudah ada ATAU buat yang baru
            let feedbackElement = field.siblings('.invalid-feedback');
            if (!feedbackElement.length) {
                let parentContainer = field.closest('.input-group, .form-check');
                if (parentContainer.length) {
                    feedbackElement = parentContainer.siblings('.invalid-feedback');
                    if (!feedbackElement.length) {
                         feedbackElement = $('<div class="invalid-feedback custom-form-error-message"></div>').insertAfter(parentContainer);
                    }
                } else {
                     feedbackElement = $('<div class="invalid-feedback custom-form-error-message"></div>').insertAfter(field.last());
                }
            }
            feedbackElement.html(Array.isArray(value) ? value.join('<br>') : value).show();
        } else {
            // Jika field spesifik tidak ditemukan, tampilkan sebagai error umum
            if (!generalErrorContainer.length) {
                 console.warn("No #generalFormErrors container found, and field '"+fieldName+"' not found.");
            } else {
                 generalErrorContainer.append('<div><strong>' + fieldName.replace(/_/g, ' ') + ':</strong> ' + (Array.isArray(value) ? value.join('<br>') : value) + '</div>').show();
            }
            console.warn("Form field not found for error key:", fieldName, "Value:", value);
        }
    });
}
// admin/assets/js/admin-global.js (atau file JS global admin lainnya)

/**
 * Escapes HTML entities in a string for safe output in HTML attributes or content from JavaScript.
 * @param {string} unsafe The string to escape.
 * @returns {string} The escaped string.
 */
function escapeHtmlJs(unsafe) {
    if (typeof unsafe !== 'string') {
        // Handle non-string inputs gracefully, e.g., convert to string or return empty
        if (unsafe === null || typeof unsafe === 'undefined') return '';
        unsafe = String(unsafe);
    }
    return unsafe
         .replace(/&/g, "&")
         .replace(/</g, "<")
         .replace(/>/g, ">")
         .replace(/"/g, """)
         .replace(/'/g, "'");
}

// Fungsi JavaScript global lainnya seperti initializeSlugGenerator, initializeImagePreview,
// showAdminNotification, displayAdminFormErrors juga sebaiknya ada di sini.
// ...