<?php
// public/study_programs_list.php (Listing all active study programs grouped by faculty)
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
$project_root = dirname(__DIR__);
require_once $project_root . '/includes/db_connect.php';
require_once $project_root . '/admin/includes/functions.php'; // For resolve_url, esc_html

$site_title = "Daftar Program Studi";
$meta_description = "Jelajahi semua program studi yang tersedia di UGKMB, dikelompokkan berdasarkan fakultas.";

$study_programs_by_faculty = [];

if (isset($pdo)) {
    try {
        // Fetch all active study programs along with their faculty information
        $stmt = $pdo->query("SELECT 
                                sp.id AS sp_id, sp.name AS sp_name, sp.slug AS sp_slug, 
                                sp.degree AS sp_degree, sp.accreditation_status AS sp_accreditation, 
                                sp.logo_url AS sp_logo_url,
                                f.id AS faculty_id, f.name AS faculty_name, f.slug AS faculty_slug
                             FROM portal_study_programs sp
                             JOIN portal_faculties f ON sp.faculty_id = f.id
                             WHERE sp.is_active = 1
                             ORDER BY f.faculty_order ASC, f.name ASC, sp.program_order ASC, sp.name ASC");
        
        $all_study_programs = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Group study programs by faculty
        foreach ($all_study_programs as $program) {
            $faculty_id = $program['faculty_id'];
            if (!isset($study_programs_by_faculty[$faculty_id])) {
                $study_programs_by_faculty[$faculty_id] = [
                    'faculty_name' => $program['faculty_name'],
                    'faculty_slug' => $program['faculty_slug'],
                    'programs' => []
                ];
            }
            $study_programs_by_faculty[$faculty_id]['programs'][] = $program;
        }

    } catch (\PDOException $e) {
        error_log("Error fetching study programs list: " . $e->getMessage());
        // Set a user-friendly error message if needed for the page
        set_flash_message('danger', 'Terjadi kesalahan saat memuat daftar program studi.');
    }
}

// Fetch global settings for header/footer
$settings = []; // Will be populated by header_public.php
$theme = $_COOKIE['theme'] ?? 'light-theme'; // For body class

// Include header
require_once PROJECT_ROOT . '/public/includes/header_public.php';
?>

<main class="container py-5">
    <div class="row">
        <div class="col-lg-12">
            <header class="mb-5 text-center">
                <h1 class="display-4 fw-bold"><?php echo $site_title; ?></h1>
                <p class="lead text-muted"><?php echo $meta_description; ?></p>
            </header>

            <?php echo display_flash_message(); // Display any flash messages ?>

            <?php if (!empty($study_programs_by_faculty)): ?>
                <?php foreach ($study_programs_by_faculty as $faculty_id => $faculty_group): ?>
                    <section class="mb-5 faculty-group">
                        <h2 class="mb-4 section-title h3">
                            <a href="<?php echo resolve_url('faculty.php?slug=' . esc_html($faculty_group['faculty_slug'])); ?>" class="text-decoration-none">
                                <i class="fas fa-university me-2"></i>Fakultas <?php echo esc_html($faculty_group['faculty_name']); ?>
                            </a>
                        </h2>
                        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
                            <?php foreach ($faculty_group['programs'] as $sp): ?>
                            <div class="col">
                                <div class="card h-100 shadow-sm text-center study-program-card article-card-hover">
                                    <?php if (!empty($sp['sp_logo_url'])): ?>
                                        <div class="text-center p-3">
                                            <img src="<?php echo resolve_url(esc_html($sp['sp_logo_url'])); ?>" class="card-img-top" alt="Logo <?php echo esc_html($sp['sp_name']); ?>" style="max-height: 80px; width: auto; object-fit: contain;">
                                        </div>
                                    <?php else: ?>
                                        <div class="text-center p-3 text-muted">
                                            <i class="fas fa-graduation-cap fa-3x"></i>
                                        </div>
                                    <?php endif; ?>
                                    <div class="card-body">
                                        <h5 class="card-title h6">
                                            <a href="<?php echo resolve_url('study_program.php?slug=' . esc_html($sp['sp_slug'])); ?>" class="text-decoration-none stretched-link">
                                                <?php echo esc_html($sp['sp_name']); ?>
                                            </a>
                                        </h5>
                                        <p class="card-text small text-muted mb-1"><?php echo esc_html($sp['sp_degree'] ?? ''); ?></p>
                                        <?php if (!empty($sp['sp_accreditation'])): ?>
                                        <p class="card-text small"><span class="badge bg-info text-dark">Akreditasi: <?php echo esc_html($sp['sp_accreditation']); ?></span></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </section>
                    <?php if (next($study_programs_by_faculty)): // Add a divider if not the last faculty group ?>
                        <hr class="my-5">
                    <?php endif; ?>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="alert alert-info text-center" role="alert">
                    Tidak ada informasi program studi yang aktif tersedia saat ini.
                </div>
            <?php endif; ?>
            
            <div class="mt-5 text-center">
                <a href="<?php echo resolve_url('faculties.php'); ?>" class="btn btn-outline-secondary"><i class="fas fa-arrow-left me-1"></i> Kembali ke Daftar Fakultas</a>
            </div>
        </div>
    </div>
</main>

<?php
require_once $project_root . '/includes/footer_public.php';
?>