<?php
// public/search.php

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

$project_root = dirname(__DIR__);

// Sertakan functions.php di awal untuk memastikan fungsi tersedia global
// Pastikan file ini sudah diperbaiki dengan if(!function_exists(...)) untuk semua fungsinya
if (file_exists($project_root . '/admin/includes/functions.php')) {
    require_once $project_root . '/admin/includes/functions.php'; 
} else {
    // Fallback minimal jika functions.php tidak ada
    if (!function_exists('esc_html')) {
        function esc_html($string) { return htmlspecialchars((string)$string, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
    }
    if (!function_exists('resolve_url')) {
        function resolve_url($path) { 
            if (defined('BASE_URL')) return rtrim(BASE_URL, '/') . '/' . ltrim($path, '/');
            // Fallback yang sangat sederhana jika BASE_URL tidak ada
            if (substr($path, 0, 1) === '/') return $path;
            return '/' . $path; 
        }
    }
     if (!function_exists('strip_tags')) { // PHP built-in, tapi untuk jaga-jaga
        function strip_tags($str, $allowed_tags = null) { return \strip_tags($str, $allowed_tags); }
    }
}
require_once $project_root . '/includes/db_connect.php'; 

// --- AJAX Request Handler ---
if (isset($_GET['ajax_search']) && $_GET['ajax_search'] == '1') {
    header('Content-Type: application/json');

    $response = ['success' => false, 'message' => 'Parameter pencarian tidak valid.'];

    $search_query_from_ajax = trim($_GET['q'] ?? '');
    // $limit_per_type sekarang diabaikan di PHP karena LIMIT 3 dihardcode di SQL
    // $limit_per_type = isset($_GET['limit_per_type']) ? max(1, (int)$_GET['limit_per_type']) : 3; 

    if (!isset($pdo)) {
        $response['message'] = 'Koneksi database gagal.';
        error_log("Search AJAX Error: PDO object not available.");
        echo json_encode($response);
        exit;
    }

    if (empty($search_query_from_ajax)) {
        $response['success'] = true; 
        $response['message'] = 'Kata kunci pencarian tidak boleh kosong.';
        $response['results'] = [];
        $response['total_found'] = 0;
        echo json_encode($response);
        exit;
    }

    try {
        $fts_param_value = $search_query_from_ajax; 
        $like_param_value = "%" . $search_query_from_ajax . "%";

        $results_accumulator = []; 
        $base_url_for_json = (defined('BASE_URL') ? rtrim(BASE_URL, '/') . '/' : '');
        if (empty($base_url_for_json) && function_exists('resolve_url')) {
            $base_url_for_json = rtrim(resolve_url(''), '/') . '/';
        }
        
        // Menggunakan query SQL yang Anda berikan dengan LIMIT 3 dihardcode
        $search_sources_sql = [
            'Artikel' => "SELECT 
                            'Artikel' AS result_type, a.id, a.title AS result_title, 
                            CONCAT('public/article.php?slug=', a.slug) AS target_path,
                            a.excerpt AS result_description, a.featured_image_url, 
                            a.published_at AS result_date, a.content_html AS full_content,
                            MATCH(a.title, a.content_html, a.excerpt) AGAINST (:search_keyword_fts IN NATURAL LANGUAGE MODE) AS relevance_score
                        FROM portal_articles a
                        LEFT JOIN portal_categories pc ON a.category_id = pc.id
                        WHERE 
                            a.status = 'published' AND a.published_at <= NOW()
                            AND (
                                MATCH(a.title, a.content_html, a.excerpt) AGAINST (:search_keyword_fts_where IN NATURAL LANGUAGE MODE)
                                OR pc.name LIKE :search_keyword_like_cat
                            )
                        ORDER BY relevance_score DESC, a.published_at DESC LIMIT 3",
            'Halaman' => "SELECT 
                            'Halaman' AS result_type, p.id, p.title AS result_title, 
                            CONCAT('public/page.php?slug=', p.slug) AS target_path,
                            SUBSTRING(p.content_html, 1, 250) AS result_description_with_html,
                            NULL AS featured_image_url, p.updated_at AS result_date,
                            MATCH(p.title, p.content_html) AGAINST (:search_keyword_fts IN NATURAL LANGUAGE MODE) as relevance_score
                        FROM portal_pages p
                        WHERE 
                            p.status = 'published' AND (p.published_at IS NULL OR p.published_at <= NOW())
                            AND MATCH(p.title, p.content_html) AGAINST (:search_keyword_fts_where IN NATURAL LANGUAGE MODE)
                        ORDER BY relevance_score DESC, p.title ASC LIMIT 3",
            'Dokumen' => "SELECT
                            'Dokumen' AS result_type, d.id, d.title AS result_title,
                            d.file_url AS target_path, d.description AS result_description,
                            d.file_mime_type, NULL AS featured_image_url, d.publish_date AS result_date 
                        FROM portal_documents d
                        WHERE
                            d.is_public = 1
                            AND (d.title LIKE :search_keyword_like OR d.description LIKE :search_keyword_like OR d.file_name_original LIKE :search_keyword_like)
                        ORDER BY d.publish_date DESC, d.title ASC LIMIT 3",
            'Agenda' => "SELECT 
                            'Agenda' AS result_type, e.id, e.title AS result_title, 
                            IF(e.event_url IS NOT NULL AND e.event_url != '', e.event_url, CONCAT('public/event.php?slug=', e.slug)) AS target_path,
                            e.description_html AS result_description_full, 
                            e.featured_image_url, e.start_datetime AS result_date
                        FROM portal_events e
                        WHERE 
                            e.is_published = 1
                            AND (e.title LIKE :search_keyword_like OR e.description_html LIKE :search_keyword_like OR e.location LIKE :search_keyword_like OR e.organizer LIKE :search_keyword_like)
                        ORDER BY e.start_datetime DESC LIMIT 3",
            'Menu Navigasi' => "SELECT 
                                'Menu Navigasi' AS result_type, mi.id, mi.title AS result_title, 
                                mi.url AS target_path, 
                                CONCAT('Tautan navigasi: ', mi.title) AS result_description,
                                NULL AS featured_image_url, NULL AS result_date
                            FROM portal_menu_items mi
                            WHERE (mi.title LIKE :search_keyword_like OR mi.url LIKE :search_keyword_like)
                            ORDER BY mi.item_order ASC, mi.title ASC LIMIT 3",
            'Link Cepat' => "SELECT 
                                'Link Cepat' AS result_type, ql.id, ql.title AS result_title, 
                                ql.url AS target_path, ql.description AS result_description,
                                ql.icon_class, NULL AS featured_image_url, NULL AS result_date
                            FROM portal_quick_links ql
                            WHERE 
                                ql.is_active = 1 
                                AND (ql.title LIKE :search_keyword_like OR ql.description LIKE :search_keyword_like OR ql.url LIKE :search_keyword_like)
                            ORDER BY ql.link_order ASC, ql.title ASC LIMIT 3"
        ];
        
        foreach ($search_sources_sql as $type => $sql) {
            try {
                $stmt = $pdo->prepare($sql);
                
                if (strpos($sql, ':search_keyword_fts') !== false) {
                    $stmt->bindValue(':search_keyword_fts', $fts_param_value, PDO::PARAM_STR);
                }
                if (strpos($sql, ':search_keyword_fts_where') !== false) {
                    $stmt->bindValue(':search_keyword_fts_where', $fts_param_value, PDO::PARAM_STR);
                }
                if (strpos($sql, ':search_keyword_like') !== false) {
                    $stmt->bindValue(':search_keyword_like', $like_param_value, PDO::PARAM_STR);
                }
                if (strpos($sql, ':search_keyword_like_cat') !== false) { 
                    $stmt->bindValue(':search_keyword_like_cat', $like_param_value, PDO::PARAM_STR);
                }

                if (!$stmt->execute()) {
                    error_log("SQL execution failed for type '{$type}': " . implode(", ", $stmt->errorInfo()) . " Query: " . $sql);
                    continue; 
                }
                $items_data = $stmt->fetchAll(PDO::FETCH_ASSOC);

                foreach ($items_data as $item_raw) {
                    $item_url = $item_raw['target_path'] ?? '#';
                    if (function_exists('resolve_url')) {
                        if (!(preg_match('~^(?:f|ht)tps?://~i', $item_url) || substr($item_url, 0, 1) === '#' || strpos($item_url, 'javascript:') === 0)) {
                            $item_url = resolve_url($item_url);
                        }
                    } else { 
                        if (!(preg_match('~^(?:f|ht)tps?://~i', $item_url) || substr($item_url, 0, 1) === '#' || strpos($item_url, 'javascript:') === 0) && !empty($base_url_for_json)) {
                             $item_url = $base_url_for_json . ltrim($item_url, '/');
                        }
                    }

                    $excerpt = '';
                    $content_for_excerpt_php = '';
                    if (isset($item_raw['result_description_full'])) { 
                        $content_for_excerpt_php = $item_raw['result_description_full'];
                    } elseif (isset($item_raw['result_description'])) { 
                        $content_for_excerpt_php = $item_raw['result_description'];
                    } elseif (isset($item_raw['full_content']) && $type === 'Artikel') { 
                        $content_for_excerpt_php = $item_raw['full_content'];
                    }
                    
                    $clean_excerpt = function_exists('strip_tags') ? strip_tags((string)$content_for_excerpt_php) : (string)$content_for_excerpt_php;
                    $excerpt_length = 120;
                    $clean_excerpt = trim(preg_replace('/\s+/', ' ', $clean_excerpt));
                    $excerpt = mb_substr($clean_excerpt, 0, $excerpt_length, 'UTF-8') . (mb_strlen($clean_excerpt, 'UTF-8') > $excerpt_length ? '...' : '');

                    $results_accumulator[] = [
                        'type' => $item_raw['result_type'] ?? $type,
                        'title' => $item_raw['result_title'] ?? 'Tanpa Judul',
                        'url' => $item_url,
                        'excerpt' => $excerpt,
                        'image' => (isset($item_raw['featured_image_url']) && !empty($item_raw['featured_image_url'])) ? (function_exists('resolve_url') ? resolve_url($item_raw['featured_image_url']) : $base_url_for_json . ltrim($item_raw['featured_image_url'], '/')) : null,
                        'date' => $item_raw['result_date'] ?? null,
                        'category_name' => $item_raw['category_name'] ?? null, 
                        'category_slug' => $item_raw['category_slug'] ?? null, 
                        'icon_class' => $item_raw['icon_class'] ?? null, 
                        'file_mime_type' => $item_raw['file_mime_type'] ?? null,
                        'relevance_score' => $item_raw['relevance_score'] ?? 0
                    ];
                }
            } catch (\PDOException $e_source_exec) {
                 error_log("PDOException during execution for type '{$type}': " . $e_source_exec->getMessage() . " Query: " . $sql);
            }
        }
        
        if (!empty($results_accumulator)) {
            usort($results_accumulator, function($a, $b) {
                $score_a = $a['relevance_score'] ?? 0;
                $score_b = $b['relevance_score'] ?? 0;
                if ($score_b != $score_a) {
                    return $score_b <=> $score_a;
                }
                $date_a_ts = isset($a['date']) && !empty($a['date']) ? strtotime($a['date']) : 0;
                $date_b_ts = isset($b['date']) && !empty($b['date']) ? strtotime($b['date']) : 0;
                if ($date_a_ts != $date_b_ts) {
                    return $date_b_ts <=> $date_a_ts; 
                }
                return strcmp(strtolower($a['title'] ?? ''), strtolower($b['title'] ?? ''));
            });
        }
        
        $total_found_overall = count($results_accumulator);

        $response['success'] = true;
        $response['message'] = $total_found_overall > 0 ? 'Pencarian berhasil.' : 'Tidak ada hasil yang cocok ditemukan.';
        $response['results'] = $results_accumulator; 
        $response['total_found'] = $total_found_overall; 
        $response['base_url'] = $base_url_for_json; 
        
    } catch (\PDOException $e) { 
        error_log("AJAX Search Main PDOException for query '{$search_query_from_ajax}': " . $e->getMessage());
        $response['message'] = 'Terjadi kesalahan teknis (DB) saat memproses pencarian.';
    } catch (\Exception $ex) { 
        error_log("AJAX Search Main Exception for query '{$search_query_from_ajax}': " . $ex->getMessage());
        $response['message'] = 'Terjadi kesalahan umum saat pencarian.';
    }

    echo json_encode($response);
    exit; 
}

// --- HTML Page Rendering (Non-AJAX Request) ---
$search_query_initial = trim($_GET['q'] ?? ''); 
$site_title = "Pencarian";
global $settings; 
$meta_description = "Cari informasi di " . (isset($settings['site_name']) ? esc_html($settings['site_name']) : "UGKMB"); 

if (function_exists('esc_html') && !empty($search_query_initial)) {
    $site_title = "Pencarian untuk: \"" . esc_html($search_query_initial) . "\"";
}
$page_title_for_layout = $site_title; 

if (file_exists($project_root . '/includes/header_public.php')) {
    require_once $project_root . '/includes/header_public.php';
} else {
    echo "<!DOCTYPE html><html><head><title>Error</title></head><body><p>Error: File header_public.php tidak ditemukan.</p>";
    // Hentikan jika header krusial, atau biarkan berlanjut dengan tampilan minimal
}
?>

<main class="flex-grow">
    <section class="py-8 md:py-12 bg-ugkmb_light_gray dark:bg-ugkmb_dark_bg_custom">
        <div class="container mx-auto px-4">
            <header class="mb-8 md:mb-10 animate-on-scroll fade-in-up-scroll">
                <h1 class="text-2xl md:text-3xl lg:text-4xl font-bold text-ugkmb_primary dark:text-ugkmb_secondary text-center md:text-left" id="searchPageTitle">
                    <?php echo esc_html($site_title); ?>
                </h1>
                
                <form id="liveSearchForm" class="mt-6 max-w-2xl mx-auto md:mx-0">
                    <div class="flex items-center border-2 border-ugkmb_primary dark:border-ugkmb_secondary rounded-lg overflow-hidden shadow-sm focus-within:ring-2 focus-within:ring-ugkmb_secondary dark:focus-within:ring-ugkmb_primary">
                        <input type="search" id="searchQueryInput" name="q" class="w-full px-4 py-3 text-gray-700 dark:text-gray-200 bg-white dark:bg-slate-800 focus:outline-none placeholder-gray-500 dark:placeholder-gray-400 text-sm md:text-base" 
                               placeholder="Cari artikel, halaman, dokumen, agenda..." 
                               aria-label="Kata kunci pencarian" 
                               value="<?php echo esc_html($search_query_initial); ?>">
                        <button class="ugkmb_primary_bg hover:bg-blue-700 dark:bg-ugkmb_secondary dark:hover:opacity-90 text-white px-6 py-3 text-sm md:text-base font-semibold" type="submit" id="searchButton">
                            <i class="fas fa-search mr-1 md:mr-2"></i> Cari
                        </button>
                    </div>
                </form>
            </header>
            
            <div id="searchResultsContainer" class="mt-4">
                <p class="text-center text-muted dark:text-gray-400" id="searchStatusText">
                    <?php if (empty($search_query_initial)): ?>
                        Masukkan kata kunci untuk memulai pencarian.
                    <?php else: ?>
                        <i class="fas fa-spinner fa-spin me-2"></i>Memuat hasil untuk "<?php echo esc_html($search_query_initial); ?>"...
                    <?php endif; ?>
                </p>
            </div>
            
            <div id="searchPaginationContainer" class="mt-4 d-flex justify-content-center"></div>

            <div class="mt-8 text-center">
                <a href="<?php echo resolve_url('index.php'); ?>" class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white ugkmb_primary_bg hover:bg-blue-700 dark:ugkmb_secondary_bg dark:hover:opacity-90 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-ugkmb_primary">
                    <i class="fas fa-home me-2"></i> Kembali ke Beranda
                </a>
            </div>
        </div>
    </section>
</main>

<?php
$page_scripts = []; 
$page_scripts[] = '
<script>
$(document).ready(function() {
    // Pastikan console.log ini muncul di browser Anda
    console.log("Search page JS loaded. Using hardcoded LIMIT 3 for each SQL type."); 

    function localEscapeHtml(unsafe) { 
        if (typeof unsafe !== "string") {
            if (unsafe === null || typeof unsafe === "undefined") return "";
            try { unsafe = String(unsafe); } catch (e) { return ""; }
        }
        return unsafe.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/\'/g, "&#039;");
    }
    function localFormatDate(dateString) { 
        if (!dateString) return "-"; 
        try { 
            const date = new Date(dateString); 
            if (isNaN(date.getTime())) { return dateString; } 
            // Format tanggal yang lebih umum dan singkat untuk daftar
            return date.toLocaleDateString("id-ID", { day: "2-digit", month: "short", year: "numeric" }); 
        } catch (e) { return dateString; } 
    }
    function localHighlightText(text, query) { 
        if (!query || typeof text !== "string") return text; 
        try {
            // Buat regex yang aman dari query, split by space for multiple word highlight
            const queryParts = query.trim().split(/\\s+/).filter(Boolean);
            if (queryParts.length === 0) return text;

            const escapedQueryParts = queryParts.map(part => part.replace(/[.*+?^${}()|[\]\\\\]/g, "\\\\$&"));
            const regex = new RegExp("(" + escapedQueryParts.join("|") + ")", "gi"); 
            return text.replace(regex, \'<mark class="bg-yellow-200 dark:bg-yellow-500 dark:text-black px-0.5 rounded">$1</mark>\'); 
        } catch (e) {
            console.error("Error highlighting text:", e);
            return text; // Return original text if regex fails
        }
    }

    let currentSearchQuery = $("#searchQueryInput").val().trim();
    let searchDebounceTimer;
    // const searchLimitPerTypeFromJS = 3; // Dihapus karena LIMIT 3 dihardcode di PHP

    function updateSearchPageTitleAndURL(query) {
        let siteName = "' . (isset($settings['site_name']) ? esc_html($settings['site_name']) : 'UGKMB') . '";
        let newTitle = siteName + " - Pencarian"; 
        let displayQuery = localEscapeHtml(query);
        if (query) { 
            newTitle = "Hasil untuk: " + displayQuery + " - " + siteName; 
            $("#searchPageTitle").html("Hasil Pencarian untuk: \\"" + displayQuery + "\\""); 
        } else { 
            $("#searchPageTitle").html("Pencarian Informasi"); 
        }
        document.title = newTitle;
        if (history.pushState) { 
            let newUrl = window.location.protocol + "//" + window.location.host + window.location.pathname; 
            if (query) { newUrl += "?q=" + encodeURIComponent(query); } 
            if (newUrl !== window.location.href.split("#")[0]) { 
                window.history.pushState({path:newUrl, query: query}, "", newUrl); 
            } 
        }
    }

    function executeSearch(query) {
        query = query.trim();
        updateSearchPageTitleAndURL(query);
        currentSearchQuery = query; 

        if (query === "") { 
            $("#searchResultsContainer").html("<p class=\\"text-center text-muted dark:text-gray-400\\">Masukkan kata kunci untuk memulai pencarian.</p>"); 
            $("#searchPaginationContainer").html(""); // Pastikan paginasi juga dikosongkan
            $("#searchStatusText").show(); return; 
        }

        $("#searchStatusText").hide();
        $("#searchResultsContainer").html("<p class=\\"text-center text-blue-600 dark:text-sky-400\\"><i class=\\"fas fa-spinner fa-spin me-2\\"></i>Mencari hasil untuk \\"" + localEscapeHtml(query) + "\\"...</p>");
        $("#searchPaginationContainer").html(""); // Kosongkan paginasi saat pencarian baru
        
        $.ajax({
            url: "' . resolve_url('public/search.php') . '", 
            type: "GET",
            data: { ajax_search: 1, q: query /* , limit_per_type: sudah dihardcode di PHP */ },
            dataType: "json",
            success: function(response) {
                console.log("AJAX Response (User SQL, Hardcoded LIMIT 3):", response); 
                $("#searchResultsContainer").empty();
                if (response.success) {
                    if (response.results && response.results.length > 0) {
                        let resultsHtmlOutput = \'<p class="mb-4 text-sm text-gray-600 dark:text-gray-400">Menampilkan \'+ response.total_found + \' hasil terkait untuk "<strong>\' + localEscapeHtml(query) + \'</strong>":</p>\';
                        
                        const resultsByType = {};
                        response.results.forEach(function(item) {
                            if (!resultsByType[item.type]) resultsByType[item.type] = [];
                            resultsByType[item.type].push(item);
                        });

                        const typeDisplayOrder = ["Artikel", "Halaman", "Dokumen", "Agenda", "Menu Navigasi", "Link Cepat"]; 
                        const typeIcons = {
                            "Artikel": "fas fa-newspaper", "Halaman": "fas fa-file-alt", "Dokumen": "fas fa-file-pdf",
                            "Agenda": "fas fa-calendar-check",
                            "Menu Navigasi": "fas fa-bars", "Link Cepat": "fas fa-link"
                        };

                        let anyResultsDisplayed = false;
                        typeDisplayOrder.forEach(function(typeKey) {
                            if (resultsByType[typeKey] && resultsByType[typeKey].length > 0) {
                                anyResultsDisplayed = true;
                                resultsHtmlOutput += \'<div class="search-results-group mb-8">\';
                                resultsHtmlOutput += \'<h2 class="text-xl md:text-2xl font-semibold text-ugkmb_primary dark:text-ugkmb_secondary border-b-2 border-ugkmb_primary/30 dark:border-ugkmb_secondary/30 pb-2 mb-4"><i class="\'+ (typeIcons[typeKey] || "fas fa-info-circle") +\' me-2 opacity-80"></i>\' + localEscapeHtml(typeKey) + \' <span class="text-base font-normal text-gray-500 dark:text-gray-400">(\' + resultsByType[typeKey].length + \')</span></h2>\';
                                resultsHtmlOutput += \'<div class="space-y-4">\';
                                resultsByType[typeKey].forEach(function(item) {
                                    let itemUrlResolved = item.url || "#"; 
                                    if (response.base_url && !(itemUrlResolved.startsWith("http") || itemUrlResolved.startsWith("#") || itemUrlResolved.startsWith("mailto:") || itemUrlResolved.startsWith("tel:") || itemUrlResolved.startsWith("javascript:"))) {
                                        itemUrlResolved = response.base_url + itemUrlResolved.replace(/^\\/+/, "");
                                    }

                                    let targetBlankAttr = (itemUrlResolved.startsWith("http") && !itemUrlResolved.includes(window.location.hostname) && (response.base_url && !itemUrlResolved.startsWith(response.base_url))) ? \' target="_blank" rel="noopener noreferrer"\' : "";
                                    
                                    let imageHtmlResult = "";
                                    if ((item.type === "Artikel" || item.type === "Agenda") && item.image) { 
                                        let resolvedImage = item.image;
                                        if (response.base_url && !item.image.startsWith("http")) {
                                            resolvedImage = response.base_url + item.image.replace(/^\\/+/, "");
                                        }
                                        imageHtmlResult = \'<div class="flex-shrink-0 w-20 h-20 sm:w-24 sm:h-24 md:w-28 md:h-28 mr-4 mb-3 sm:mb-0 overflow-hidden rounded-md shadow-sm"><img src="\' + localEscapeHtml(resolvedImage) + \'" alt="\' + localEscapeHtml(item.title) + \'" class="w-full h-full object-cover"></div>\'; 
                                    } else if (item.type === "Dokumen" && item.file_mime_type) { 
                                        let docIcon = "fas fa-file"; 
                                        if(item.file_mime_type.includes("pdf")) docIcon = "fas fa-file-pdf text-red-500"; 
                                        else if(item.file_mime_type.includes("word")) docIcon = "fas fa-file-word text-blue-500"; 
                                        else if(item.file_mime_type.includes("excel") || item.file_mime_type.includes("spreadsheet")) docIcon = "fas fa-file-excel text-green-500"; 
                                        imageHtmlResult = \'<div class="flex-shrink-0 w-16 h-16 mr-4 mb-3 sm:mb-0 flex items-center justify-center text-3xl text-gray-400 dark:text-gray-500"><i class="\'+docIcon+\'"></i></div>\'; 
                                    }

                                    resultsHtmlOutput += \'<div class="search-result-item bg-white dark:bg-slate-800 p-3.5 md:p-5 rounded-xl shadow-md hover:shadow-lg transition-shadow duration-200">\';
                                    resultsHtmlOutput += \'<div class="flex flex-col sm:flex-row">\';
                                    resultsHtmlOutput += imageHtmlResult;
                                    resultsHtmlOutput += \'<div class="flex-1">\';
                                    resultsHtmlOutput += \'<h3 class="text-md md:text-lg font-semibold text-ugkmb_text_dark dark:text-gray-100 mb-0.5 leading-tight"><a href="\' + localEscapeHtml(itemUrlResolved) + \'"\' + targetBlankAttr + \' class="hover:text-ugkmb_primary dark:hover:text-ugkmb_secondary">\'+ localHighlightText(localEscapeHtml(item.title), query) +\'</a></h3>\';
                                    if (item.category_name && item.type === "Artikel" && item.category_slug) { resultsHtmlOutput += \'<p class="text-xs text-gray-500 dark:text-gray-400 mb-1"><i class="fas fa-folder-open fa-fw opacity-70"></i> <a href="\' + (response.base_url || "") + "public/category.php?slug=" + localEscapeHtml(item.category_slug) + \'" class="hover:underline">\' + localEscapeHtml(item.category_name) + \'</a></p>\';}
                                    if (item.date) { resultsHtmlOutput += \'<p class="text-xs text-gray-500 dark:text-gray-400 mb-1.5"><i class="fas fa-calendar-alt fa-fw opacity-70"></i> \' + localFormatDate(item.date) + \'</p>\'; }
                                    if (item.excerpt) { resultsHtmlOutput += \'<p class="text-xs md:text-sm text-gray-600 dark:text-gray-300 leading-relaxed line-clamp-2 md:line-clamp-3 mb-1.5">\'+ localHighlightText(localEscapeHtml(item.excerpt), query) +\'</p>\'; }
                                    resultsHtmlOutput += \'<a href="\' + localEscapeHtml(itemUrlResolved) + \'"\' + targetBlankAttr + \' class="text-xs text-ugkmb_primary dark:text-ugkmb_secondary hover:underline truncate block">\'+ localEscapeHtml(itemUrlResolved) +\'</a>\';
                                    resultsHtmlOutput += \'</div></div></div>\'; 
                                });
                                resultsHtmlOutput += \'</div></div>\'; 
                            }
                        });
                        if (!anyResultsDisplayed && query) { 
                            resultsHtmlOutput = \'<div class="alert alert-info text-center dark:bg-sky-800 dark:text-sky-200 dark:border-sky-700" role="alert">Tidak ada hasil yang ditemukan untuk "<strong>\' + localEscapeHtml(query) + \'</strong>".</div>\';
                        } else if (!query && !anyResultsDisplayed) {
                            resultsHtmlOutput = "<p class=\\"text-center text-muted dark:text-gray-400\\">Masukkan kata kunci untuk memulai pencarian.</p>";
                        }
                        $("#searchResultsContainer").html(resultsHtmlOutput);
                    } else { 
                        if(query) { 
                            $("#searchResultsContainer").html(\'<div class="alert alert-info text-center dark:bg-sky-800 dark:text-sky-200 dark:border-sky-700" role="alert">Tidak ada hasil yang ditemukan untuk "<strong>\' + localEscapeHtml(query) + \'</strong>".</div>\');
                        } else { 
                            $("#searchResultsContainer").html("<p class=\\"text-center text-muted dark:text-gray-400\\">Masukkan kata kunci untuk memulai pencarian.</p>");
                        }
                    }
                } else { 
                    $("#searchResultsContainer").html(\'<div class="alert alert-danger text-center dark:bg-red-800 dark:text-red-200 dark:border-red-700" role="alert">\' + localEscapeHtml(response.message || "Terjadi kesalahan saat memuat hasil pencarian.") + \'</div>\');
                }
            },
            error: function(jqXHR, textStatus, errorThrown) {
                console.error("Search AJAX error details:", textStatus, errorThrown, jqXHR.responseText);
                $("#searchResultsContainer").html(\'<div class="alert alert-danger text-center dark:bg-red-800 dark:text-red-200 dark:border-red-700" role="alert">Kesalahan koneksi atau server. Silakan coba lagi. (\' + localEscapeHtml(textStatus) + \')</div>\');
            }
        });
    }

    $("#liveSearchForm").on("submit", function(e) { e.preventDefault(); currentSearchQuery = $("#searchQueryInput").val().trim(); executeSearch(currentSearchQuery); });
    $("#searchQueryInput").on("input", function() { clearTimeout(searchDebounceTimer); const queryFromInput = $(this).val().trim(); searchDebounceTimer = setTimeout(function() { if(queryFromInput !== currentSearchQuery || (queryFromInput === "" && currentSearchQuery !== "") ){ currentSearchQuery = queryFromInput; executeSearch(currentSearchQuery); } else if (queryFromInput === "" && currentSearchQuery === "") { $("#searchResultsContainer").html("<p class=\\"text-center text-muted dark:text-gray-400\\">Masukkan kata kunci.</p>"); updateSearchPageTitleAndURL(""); } }, 700); });
    $(window).on("popstate", function(event) { 
        const state = event.originalEvent.state;
        if (state && typeof state.query !== "undefined") { 
            $("#searchQueryInput").val(state.query); 
            executeSearch(state.query); 
        } else {
            const params = new URLSearchParams(window.location.search);
            const qParam = params.get("q") || "";
            $("#searchQueryInput").val(qParam);
            if (qParam !== currentSearchQuery || (qParam === "" && currentSearchQuery !== "")) { 
                executeSearch(qParam);
            } else if (!qParam) {
                $("#searchResultsContainer").html("<p class=\\"text-center text-muted dark:text-gray-400\\">Masukkan kata kunci.</p>");
                updateSearchPageTitleAndURL("");
            }
        }
    });

    if (currentSearchQuery) { updateSearchPageTitleAndURL(currentSearchQuery); executeSearch(currentSearchQuery); } 
    else { updateSearchPageTitleAndURL(""); }
});
</script>
';

if (file_exists($project_root . '/includes/footer_public.php')) {
    require_once $project_root . '/includes/footer_public.php'; 
} else {
    echo "<p>Error: File footer_public.php tidak ditemukan.</p></body></html>";
}
?>