<?php
// public/faculty.php (Detail of a single faculty and its study programs)
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
$project_root = dirname(__DIR__);
require_once $project_root . '/includes/db_connect.php';
require_once $project_root . '/admin/includes/functions.php'; // For resolve_url, esc_html

$faculty_slug = $_GET['slug'] ?? null;
$faculty_data = null;
$study_programs = [];
$site_title = "Detail Fakultas";
$meta_description = "Informasi detail mengenai fakultas dan program studi yang dinaungi.";

if (empty($faculty_slug)) {
    http_response_code(404);
    set_flash_message('danger', 'Fakultas tidak ditemukan (slug kosong).');
    // header("Location: " . resolve_url('faculties.php')); // Redirect to faculties list
    // exit;
    $site_title = "Fakultas Tidak Ditemukan";
}

if (isset($pdo) && !empty($faculty_slug)) {
    try {
        // Fetch faculty details
        $stmt_faculty = $pdo->prepare("SELECT * FROM portal_faculties WHERE slug = :slug LIMIT 1");
        $stmt_faculty->bindParam(':slug', $faculty_slug);
        $stmt_faculty->execute();
        $faculty_data = $stmt_faculty->fetch();

        if ($faculty_data) {
            $site_title = esc_html($faculty_data['name']);
            $meta_description = !empty($faculty_data['description_html']) ? esc_html(substr(strip_tags($faculty_data['description_html']), 0, 160)) : "Detail Fakultas " . esc_html($faculty_data['name']);

            // Fetch active study programs under this faculty
            $stmt_sp = $pdo->prepare("SELECT id, name, slug, degree, accreditation_status, logo_url 
                                      FROM portal_study_programs 
                                      WHERE faculty_id = :faculty_id AND is_active = 1 
                                      ORDER BY program_order ASC, name ASC");
            $stmt_sp->bindParam(':faculty_id', $faculty_data['id'], PDO::PARAM_INT);
            $stmt_sp->execute();
            $study_programs = $stmt_sp->fetchAll();

        } else {
            http_response_code(404);
            $site_title = "Fakultas Tidak Ditemukan";
            $meta_description = "Fakultas yang Anda cari tidak ditemukan.";
        }
    } catch (\PDOException $e) {
        error_log("Error fetching faculty detail: " . $e->getMessage());
        http_response_code(500);
        $site_title = "Kesalahan Server";
        $meta_description = "Terjadi kesalahan saat memuat detail fakultas.";
    }
}

// Fetch global settings for header/footer
$settings = []; // Will be populated by header_public.php
$theme = $_COOKIE['theme'] ?? 'light-theme'; // For body class

// Include header
require_once $project_root . '/includes/header_public.php';
?>

<main class="container py-5">
    <div class="row">
        <div class="col-lg-10 mx-auto">
            <?php if ($faculty_data): ?>
                <article class="faculty-detail-article">
                    <header class="mb-4 text-center">
                        <h1 class="display-4 fw-bold faculty-title"><?php echo esc_html($faculty_data['name']); ?></h1>
                        <?php if (!empty($faculty_data['dean_name'])): ?>
                            <p class="lead text-muted">Dekan: <?php echo esc_html($faculty_data['dean_name']); ?></p>
                        <?php endif; ?>
                        <?php if (!empty($faculty_data['website_url'])): ?>
                            <p><a href="<?php echo esc_html($faculty_data['website_url']); ?>" class="btn btn-outline-primary" target="_blank" rel="noopener noreferrer"><i class="fas fa-globe me-1"></i> Kunjungi Website Fakultas</a></p>
                        <?php endif; ?>
                    </header>
                    <hr class="my-4">

                    <?php if (!empty($faculty_data['description_html'])): ?>
                    <section class="mb-5 faculty-description entry-content">
                        <h3 class="mb-3">Tentang Fakultas</h3>
                        <?php echo $faculty_data['description_html']; // Output HTML content, ensure sanitized on input ?>
                    </section>
                    <?php endif; ?>

                    <?php if (!empty($study_programs)): ?>
                    <section class="mb-5 study-programs-section">
                        <h3 class="mb-4 text-center section-title">Program Studi di <?php echo esc_html($faculty_data['name']); ?></h3>
                        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
                            <?php foreach ($study_programs as $sp): ?>
                            <div class="col">
                                <div class="card h-100 shadow-sm text-center study-program-card">
                                    <?php if (!empty($sp['logo_url'])): ?>
                                        <div class="text-center p-3">
                                            <img src="<?php echo resolve_url(esc_html($sp['logo_url'])); ?>" class="card-img-top" alt="Logo <?php echo esc_html($sp['name']); ?>" style="max-height: 80px; width: auto; object-fit: contain;">
                                        </div>
                                    <?php else: ?>
                                        <div class="text-center p-3 text-muted">
                                            <i class="fas fa-graduation-cap fa-3x"></i>
                                        </div>
                                    <?php endif; ?>
                                    <div class="card-body">
                                        <h5 class="card-title h6"><a href="<?php echo resolve_url('study_program.php?slug=' . esc_html($sp['slug'])); ?>" class="text-decoration-none stretched-link"><?php echo esc_html($sp['name']); ?></a></h5>
                                        <p class="card-text small text-muted mb-1"><?php echo esc_html($sp['degree'] ?? ''); ?></p>
                                        <?php if (!empty($sp['accreditation_status'])): ?>
                                        <p class="card-text small"><span class="badge bg-info text-dark">Akreditasi: <?php echo esc_html($sp['accreditation_status']); ?></span></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </section>
                    <?php elseif (!empty($faculty_data['description_html'])): // Only show this if description exists but no prodi ?>
                         <p class="text-center text-muted">Belum ada program studi yang terdaftar untuk fakultas ini.</p>
                    <?php endif; ?>
                    
                    <hr class="my-4">
                    <div class="text-center">
                         <a href="<?php echo resolve_url('faculties.php'); ?>" class="btn btn-outline-secondary"><i class="fas fa-arrow-left me-1"></i> Kembali ke Daftar Fakultas</a>
                    </div>

                </article>
            <?php else: ?>
                <div class="alert alert-warning text-center" role="alert">
                    <h4 class="alert-heading">Fakultas Tidak Ditemukan</h4>
                    <p><?php echo $meta_description; ?></p>
                    <hr>
                    <p class="mb-0">Silakan kembali ke <a href="<?php echo resolve_url('faculties.php'); ?>" class="alert-link">daftar fakultas</a>.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</main>

<?php
require_once $project_root . '/includes/footer_public.php';
?>
