<?php
// public/events.php (Listing all events)
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
$project_root = dirname(__DIR__);
require_once $project_root . '/includes/db_connect.php';
require_once $project_root . '/admin/includes/functions.php'; // For resolve_url, esc_html

$site_title = "Agenda Acara";
$meta_description = "Daftar agenda dan acara mendatang dari UGKMB.";

$events = [];
$total_events = 0;
$limit = 10; // Events per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Filter for upcoming or past events (optional)
$filter_type = $_GET['filter'] ?? 'upcoming'; // 'upcoming', 'past', 'all'

if (isset($pdo)) {
    try {
        $where_clause = "WHERE e.is_published = 1 "; // Alias tabel e.
        $order_by_clause = "ORDER BY e.start_datetime "; // Alias tabel e.

        if ($filter_type === 'upcoming') {
            $where_clause .= "AND (e.end_datetime IS NULL OR e.end_datetime >= NOW()) AND e.start_datetime >= CURDATE() ";
            $order_by_clause .= "ASC";
        } elseif ($filter_type === 'past') {
            $where_clause .= "AND (e.end_datetime < NOW() OR (e.end_datetime IS NULL AND e.start_datetime < CURDATE())) ";
            $order_by_clause .= "DESC";
        } else { // 'all'
             $order_by_clause .= "DESC";
        }


        // Count total events for pagination based on filter
        // Perhatikan penambahan alias 'e' di sini juga jika ada ambiguitas
        $stmt_count = $pdo->query("SELECT COUNT(*) FROM portal_events e " . $where_clause);
        $total_events = $stmt_count->fetchColumn();

        // Fetch events for the current page
        $stmt = $pdo->prepare("SELECT e.id, e.title, e.slug, e.description_html, e.start_datetime, e.end_datetime, e.location, e.featured_image_url, e.organizer
                               FROM portal_events e
                               " . $where_clause . "
                               " . $order_by_clause . "
                               LIMIT :limit OFFSET :offset");
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $events = $stmt->fetchAll();
    } catch (\PDOException $e) {
        error_log("Error fetching events list: " . $e->getMessage());
        // Set a user-friendly error message if needed
    }
}

// Fetch global settings for header/footer
$settings = []; // Will be populated by header_public.php
$theme = $_COOKIE['theme'] ?? 'light-theme'; // For body class

// Include header
require_once PROJECT_ROOT . '/public/includes/header_public.php';
?>

<main class="container py-5">
    <div class="row">
        <div class="col-lg-12">
            <header class="mb-5 text-center">
                <h1 class="display-4 fw-bold"><?php echo $site_title; ?></h1>
                <p class="lead text-muted"><?php echo $meta_description; ?></p>
                <div class="btn-group mt-3" role="group" aria-label="Filter Acara">
                    <a href="?filter=upcoming" class="btn <?php echo ($filter_type === 'upcoming' || ($filter_type !== 'past' && $filter_type !== 'all')) ? 'btn-primary' : 'btn-outline-primary'; ?>">Akan Datang</a>
                    <a href="?filter=past" class="btn <?php echo ($filter_type === 'past') ? 'btn-primary' : 'btn-outline-primary'; ?>">Sudah Lewat</a>
                    <a href="?filter=all" class="btn <?php echo ($filter_type === 'all') ? 'btn-primary' : 'btn-outline-primary'; ?>">Semua Acara</a>
                </div>
            </header>

            <?php if (isset($_SESSION['flash_message'])): ?>
                <div class="alert alert-<?php echo $_SESSION['flash_message']['type']; ?> alert-dismissible fade show" role="alert">
                    <?php echo $_SESSION['flash_message']['text']; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['flash_message']); ?>
            <?php endif; ?>

            <?php if (!empty($events)): ?>
                <div class="list-group shadow-sm">
                    <?php foreach ($events as $event): ?>
                    <a href="<?php echo resolve_url('event.php?slug=' . esc_html($event['slug'])); ?>" class="list-group-item list-group-item-action p-3 p-md-4 event-list-item article-card-hover">
                        <div class="row align-items-center g-3">
                            <div class="col-md-3 mb-3 mb-md-0 text-center">
                                <?php if (!empty($event['featured_image_url'])): ?>
                                    <img src="<?php echo resolve_url(esc_html($event['featured_image_url'])); ?>" alt="<?php echo esc_html($event['title']); ?>" class="img-fluid rounded" style="max-height: 150px; width:auto; object-fit: cover;">
                                <?php else: ?>
                                    <div class="bg-light d-flex align-items-center justify-content-center rounded" style="height: 120px; width: 100%;">
                                        <i class="fas fa-calendar-alt fa-3x text-muted"></i>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="col-md-9">
                                <h4 class="mb-1 event-item-title"><?php echo esc_html($event['title']); ?></h4>
                                <p class="text-muted small mb-2 event-item-date">
                                    <i class="fas fa-calendar-alt fa-fw"></i> <?php echo date('d M Y, H:i', strtotime($event['start_datetime'])); ?>
                                    <?php if ($event['end_datetime'] && date('Y-m-d', strtotime($event['start_datetime'])) != date('Y-m-d', strtotime($event['end_datetime']))): ?>
                                        - <?php echo date('d M Y, H:i', strtotime($event['end_datetime'])); ?>
                                    <?php elseif ($event['end_datetime']): ?>
                                        - <?php echo date('H:i', strtotime($event['end_datetime'])); ?>
                                    <?php endif; ?>
                                </p>
                                <?php if (!empty($event['location'])): ?>
                                    <p class="text-muted small mb-1 event-item-location"><i class="fas fa-map-marker-alt fa-fw"></i> <?php echo esc_html($event['location']); ?></p>
                                <?php endif; ?>
                                <?php if (!empty($event['organizer'])): ?>
                                    <p class="text-muted small mb-2 event-item-organizer"><i class="fas fa-sitemap fa-fw"></i> <?php echo esc_html($event['organizer']); ?></p>
                                <?php endif; ?>
                                <p class="mb-0 event-item-excerpt d-none d-md-block">
                                    <?php
                                    $excerpt = !empty($event['description_html']) ? strip_tags($event['description_html']) : 'Klik untuk detail acara...';
                                    echo esc_html(substr($excerpt, 0, 150)) . (strlen($excerpt) > 150 ? '...' : '');
                                    ?>
                                </p>
                            </div>
                        </div>
                    </a>
                    <?php endforeach; ?>
                </div>

                <?php if ($total_events > $limit): ?>
                <nav aria-label="Page navigation events" class="mt-5">
                    <ul class="pagination justify-content-center">
                        <?php
                        $total_pages = ceil($total_events / $limit);
                        $prev_disabled = ($page <= 1) ? "disabled" : "";
                        $next_disabled = ($page >= $total_pages) ? "disabled" : "";
                        ?>
                        <li class="page-item <?php echo $prev_disabled; ?>">
                            <a class="page-link" href="?filter=<?php echo esc_html($filter_type); ?>&page=<?php echo $page - 1; ?>" aria-label="Previous">
                                <span aria-hidden="true">«</span>
                            </a>
                        </li>
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?php echo ($page == $i) ? "active" : ""; ?>"><a class="page-link" href="?filter=<?php echo esc_html($filter_type); ?>&page=<?php echo $i; ?>"><?php echo $i; ?></a></li>
                        <?php endfor; ?>
                        <li class="page-item <?php echo $next_disabled; ?>">
                            <a class="page-link" href="?filter=<?php echo esc_html($filter_type); ?>&page=<?php echo $page + 1; ?>" aria-label="Next">
                                <span aria-hidden="true">»</span>
                            </a>
                        </li>
                    </ul>
                </nav>
                <?php endif; ?>

            <?php else: ?>
                <div class="alert alert-info text-center" role="alert">
                    Tidak ada acara yang ditemukan untuk filter "<?php echo esc_html($filter_type); ?>".
                </div>
            <?php endif; ?>
        </div>
    </div>
</main>

<?php
require_once $project_root . '/includes/footer_public.php';
?>