<?php
// public/articles.php (FINAL ADVANCED VERSION with Filters & Search)
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
$project_root = dirname(__DIR__);
require_once $project_root . '/includes/db_connect.php';
require_once $project_root . '/admin/includes/functions.php';

$site_title = "Arsip Berita & Artikel";
$meta_description = "Jelajahi, cari, dan filter semua berita dan informasi dari portal internal dan situs utama UGKMB.";

// --- KODE ASLI UNTUK MENGAMBIL DATA DARI DATABASE LOKAL (TIDAK DIUBAH) ---
$articles = [];
$total_articles = 0;
if (isset($pdo)) {
    try {
        $stmt_count = $pdo->query("SELECT COUNT(*) FROM portal_articles WHERE status = 'published' AND published_at <= NOW()");
        $total_articles = $stmt_count->fetchColumn();
        $stmt = $pdo->prepare("SELECT a.title, a.slug, a.featured_image_url, a.published_at, c.name as category_name, c.slug as category_slug FROM portal_articles a LEFT JOIN portal_categories c ON a.category_id = c.id WHERE a.status = 'published' AND a.published_at <= NOW() ORDER BY a.published_at DESC");
        $stmt->execute();
        $articles = $stmt->fetchAll();
    } catch (\PDOException $e) {
        error_log("Error fetching all articles list: " . $e->getMessage());
    }
}

require_once $project_root . '/public/includes/header_public.php';
?>

<style>
    .article-card-hover {
        border-radius: 0.75rem;
        transition: box-shadow 0.3s ease, transform 0.3s ease;
        box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        border: none;
    }
    .article-card-hover:hover {
        transform: translateY(-5px);
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
    }
    .section-subtitle {
        font-size: 1.1rem;
        font-weight: 500;
    }
    .filter-controls {
        background-color: var(--bs-tertiary-bg);
        padding: 1rem;
        border-radius: 0.75rem;
    }
</style>

<main class="container py-5">
    <header class="mb-5 text-center">
        <h1 class="display-4 fw-bold"><?php echo esc_html($site_title); ?></h1>
        <p class="lead text-muted"><?php echo esc_html($meta_description); ?></p>
    </header>

    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2 class="h3 fw-bold custom-page-title mb-0">Arsip Portal Internal</h2>
        <span class="badge bg-secondary rounded-pill section-subtitle"><?php echo $total_articles; ?> Artikel</span>
    </div>
    <hr class="mt-1 mb-4">
    
    <?php if (!empty($articles)): ?>
        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
            <?php foreach ($articles as $article): ?>
            <div class="col">
                <div class="card h-100 article-card-hover">
                    <?php if (!empty($article['featured_image_url'])): ?>
                    <a href="<?php echo resolve_url('article.php?slug=' . esc_html($article['slug'])); ?>"><img src="<?php echo resolve_url(esc_html($article['featured_image_url'])); ?>" class="card-img-top" alt="<?php echo esc_html($article['title']); ?>" style="height: 200px; object-fit: cover; border-top-left-radius: 0.75rem; border-top-right-radius: 0.75rem;"></a>
                    <?php endif; ?>
                    <div class="card-body d-flex flex-column">
                        <h5 class="card-title mb-2"><a href="<?php echo resolve_url('article.php?slug=' . esc_html($article['slug'])); ?>" class="text-decoration-none stretched-link article-title-link"><?php echo esc_html($article['title']); ?></a></h5>
                        <p class="card-text small text-muted mt-auto"><i class="fas fa-calendar-alt me-1"></i><?php echo date('d F Y', strtotime($article['published_at'])); ?></p>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    <?php else: ?>
        <div class="alert alert-info text-center" role="alert">Tidak ada berita dari portal internal.</div>
    <?php endif; ?>

    <hr class="my-5">
    
    <header class="mb-5 text-center">
        <h2 class="display-5 fw-bold">Arsip dari Situs Utama</h2>
        <p class="lead text-muted">Gunakan filter dan pencarian untuk menjelajahi semua konten dari WordPress.</p>
    </header>

    <div class="filter-controls p-3 mb-4 rounded-3">
        <div class="row g-2 align-items-center">
            <div class="col-md-4">
                <input type="search" id="wp-search-input" class="form-control" placeholder="Cari artikel di WordPress...">
            </div>
            <div class="col-md-3">
                <select id="wp-category-filter" class="form-select" aria-label="Filter Kategori">
                    <option value="">Semua Kategori</option>
                    </select>
            </div>
            <div class="col-md-3">
                 <select id="wp-tag-filter" class="form-select" aria-label="Filter Tag">
                    <option value="">Semua Tag</option>
                    </select>
            </div>
            <div class="col-md-2 text-end">
                <span id="wp-total-count" class="badge bg-primary rounded-pill section-subtitle" style="display: none;"></span>
            </div>
        </div>
    </div>

    <section id="wp-posts-section">
        <div id="wordpress-posts-container" class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
            </div>
        <div id="wp-posts-loader" class="text-center py-5">
            <div class="spinner-border text-primary" style="width: 3rem; height: 3rem;" role="status"></div>
            <p class="mt-3 lead" id="wp-loader-text">Memuat...</p>
        </div>
    </section>
</main>

<script>
document.addEventListener('DOMContentLoaded', function() {
    
    const baseApiUrl = 'https://wp.ugkmb.ac.id/wp-json/wp/v2';
    const container = document.getElementById('wordpress-posts-container');
    const loader = document.getElementById('wp-posts-loader');
    const totalCountBadge = document.getElementById('wp-total-count');
    
    // Elemen Filter
    const searchInput = document.getElementById('wp-search-input');
    const categoryFilter = document.getElementById('wp-category-filter');
    const tagFilter = document.getElementById('wp-tag-filter');

    let searchTimeout;

    const renderPosts = (posts) => {
        if (posts.length === 0) {
            return '<div class="col-12"><div class="alert alert-warning text-center">Tidak ada artikel yang cocok dengan kriteria Anda.</div></div>';
        }
        return posts.map(post => {
            const featuredImage = post._embedded?.['wp:featuredmedia']?.[0]?.source_url || '<?php echo resolve_url("assets/images/default-placeholder.png"); ?>';
            const postDate = new Date(post.date).toLocaleDateString('id-ID', { day: 'numeric', month: 'long', year: 'numeric' });
            return `
                <div class="col">
                    <div class="card h-100 article-card-hover">
                        <a href="${post.link}" target="_blank" rel="noopener noreferrer">
                            <img src="${featuredImage}" class="card-img-top" alt="${post.title.rendered}" style="height: 200px; object-fit: cover; border-top-left-radius: 0.75rem; border-top-right-radius: 0.75rem;">
                        </a>
                        <div class="card-body d-flex flex-column">
                            <h5 class="card-title h6 mb-2"><a href="${post.link}" target="_blank" rel="noopener noreferrer" class="text-decoration-none stretched-link article-title-link">${post.title.rendered}</a></h5>
                            <p class="card-text small text-muted mt-auto"><i class="fas fa-calendar-alt me-1"></i>${postDate}</p>
                        </div>
                    </div>
                </div>`;
        }).join('');
    };

    const fetchPosts = async () => {
        loader.style.display = 'block';
        container.innerHTML = '';
        totalCountBadge.style.display = 'none';

        // Bangun URL API berdasarkan filter yang aktif
        let apiUrl = new URL(`${baseApiUrl}/posts`);
        apiUrl.searchParams.append('per_page', '100'); // Ambil hingga 100 per halaman
        apiUrl.searchParams.append('_embed', 'true');

        if (searchInput.value) apiUrl.searchParams.append('search', searchInput.value);
        if (categoryFilter.value) apiUrl.searchParams.append('categories', categoryFilter.value);
        if (tagFilter.value) apiUrl.searchParams.append('tags', tagFilter.value);
        
        try {
            const response = await fetch(apiUrl);
            if (!response.ok) throw new Error('Gagal mengambil data postingan.');

            const totalPosts = parseInt(response.headers.get('X-WP-Total'));
            totalCountBadge.textContent = `${totalPosts} Artikel`;
            totalCountBadge.style.display = 'inline-block';

            const posts = await response.json();
            
            loader.style.display = 'none';
            container.innerHTML = renderPosts(posts);

        } catch (error) {
            console.error("Error fetching WordPress posts:", error);
            loader.style.display = 'none';
            container.innerHTML = `<div class="col-12"><div class="alert alert-danger">Gagal memuat arsip.</div></div>`;
        }
    };

    const populateFilter = async (endpoint, selectElement) => {
        try {
            const response = await fetch(`${baseApiUrl}${endpoint}?per_page=100&hide_empty=true`);
            if (!response.ok) throw new Error(`Gagal mengambil data ${endpoint}`);
            const items = await response.json();
            items.forEach(item => {
                const option = document.createElement('option');
                option.value = item.id;
                option.textContent = item.name;
                selectElement.appendChild(option);
            });
        } catch (error) {
            console.error(error);
        }
    };

    // --- Event Listeners ---
    searchInput.addEventListener('keyup', () => {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(fetchPosts, 500); // Tunggu 500ms setelah user berhenti mengetik
    });

    categoryFilter.addEventListener('change', fetchPosts);
    tagFilter.addEventListener('change', fetchPosts);

    // --- Inisialisasi ---
    const initializePage = () => {
        populateFilter('/categories', categoryFilter);
        populateFilter('/tags', tagFilter);
        fetchPosts(); // Panggil pertama kali untuk memuat semua postingan
    };

    initializePage();
});
</script>

<?php
require_once $project_root . '/includes/footer_public.php';
?>