<?php
// public/api_explorer.php
// Halaman ini mendemonstrasikan pengambilan data dari berbagai endpoint WordPress REST API.

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__));
}
require_once PROJECT_ROOT . '/admin/includes/functions.php';

$site_title = "WordPress API Explorer";
$meta_description = "Melihat data langsung dari berbagai endpoint WordPress REST API.";

require_once PROJECT_ROOT . '/public/includes/header_public.php';
?>

<style>
    .api-section {
        margin-bottom: 3rem;
        padding: 2rem;
        border-radius: 0.75rem;
        background-color: var(--bs-light-bg-subtle);
        border: 1px solid var(--bs-border-color-translucent);
    }
    .api-section h2 {
        border-bottom: 2px solid var(--bs-primary);
        padding-bottom: 0.5rem;
        margin-bottom: 1.5rem;
        font-size: 1.75rem;
    }
    .endpoint-url {
        font-family: 'Courier New', Courier, monospace;
        background-color: var(--bs-secondary-bg-subtle);
        padding: 0.25rem 0.5rem;
        border-radius: 0.25rem;
        font-size: 0.9rem;
        color: var(--bs-danger);
        word-break: break-all;
    }
    pre {
        background-color: #282c34;
        color: #abb2bf;
        padding: 1rem;
        border-radius: 0.5rem;
        max-height: 300px;
        overflow-y: auto;
        font-size: 0.85rem;
    }
    .auth-required::after {
        content: '🔒 Memerlukan Autentikasi';
        font-size: 0.8rem;
        font-weight: bold;
        color: var(--bs-warning-text-emphasis);
        background-color: var(--bs-warning-bg-subtle);
        padding: 0.2rem 0.5rem;
        border-radius: 20px;
        margin-left: 0.5rem;
    }
    .loader {
        text-align: center;
        padding: 2rem;
    }
</style>

<main class="container py-5">
    <header class="text-center mb-5">
        <h1 class="display-4 fw-bold"><?php echo $site_title; ?></h1>
        <p class="lead text-muted"><?php echo $meta_description; ?></p>
    </header>

    <section class="api-section">
        <h2>Konten (Posts, Pages, Media)</h2>
        
        <h4 class="mt-4">Posts</h4>
        <p>Endpoint: <code class="endpoint-url">/posts</code></p>
        <div id="posts-container"><div class="loader spinner-border text-primary"></div></div>

        <h4 class="mt-4">Pages</h4>
        <p>Endpoint: <code class="endpoint-url">/pages</code></p>
        <div id="pages-container"><div class="loader spinner-border text-primary"></div></div>

        <h4 class="mt-4">Media</h4>
        <p>Endpoint: <code class="endpoint-url">/media</code></p>
        <div id="media-container"><div class="loader spinner-border text-primary"></div></div>
    </section>

    <section class="api-section">
        <h2>Taksonomi (Kategori & Tag)</h2>

        <h4 class="mt-4">Categories</h4>
        <p>Endpoint: <code class="endpoint-url">/categories</code></p>
        <div id="categories-container"><div class="loader spinner-border text-primary"></div></div>

        <h4 class="mt-4">Tags</h4>
        <p>Endpoint: <code class="endpoint-url">/tags</code></p>
        <div id="tags-container"><div class="loader spinner-border text-primary"></div></div>
    </section>

    <section class="api-section">
        <h2 class="auth-required">Pengguna (Users)</h2>
        <p class="text-muted">Mengakses data pengguna biasanya memerlukan hak akses khusus dan tidak akan menampilkan data untuk pengunjung publik.</p>
        
        <h4 class="mt-4">Users</h4>
        <p>Endpoint: <code class="endpoint-url">/users</code></p>
        <div id="users-container"><div class="loader spinner-border text-primary"></div></div>
    </section>
    
    <section class="api-section">
        <h2>Lain-lain</h2>

        <h4 class="mt-4">Comments</h4>
        <p>Endpoint: <code class="endpoint-url">/comments</code></p>
        <div id="comments-container"><div class="loader spinner-border text-primary"></div></div>
        
        <h4 class="mt-4 auth-required">Settings</h4>
        <p>Endpoint: <code class="endpoint-url">/settings</code></p>
        <div id="settings-container"><div class="loader spinner-border text-primary"></div></div>

        <h4 class="mt-4">Post Types</h4>
        <p>Endpoint: <code class="endpoint-url">/types</code></p>
        <div id="types-container"><div class="loader spinner-border text-primary"></div></div>

        <h4 class="mt-4">Taxonomies</h4>
        <p>Endpoint: <code class="endpoint-url">/taxonomies</code></p>
        <div id="taxonomies-container"><div class="loader spinner-border text-primary"></div></div>
    </section>

</main>

<script>
document.addEventListener('DOMContentLoaded', function() {
    
    const baseApiUrl = 'https://wp.ugkmb.ac.id/wp-json/wp/v2';

    // Helper untuk fetch data dan render
    const fetchAndRender = async (endpoint, containerId, renderFunction) => {
        const container = document.getElementById(containerId);
        if (!container) return;

        try {
            const response = await fetch(`${baseApiUrl}${endpoint}`);
            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.message || `HTTP error! status: ${response.status}`);
            }
            const data = await response.json();
            container.innerHTML = renderFunction(data);
        } catch (error) {
            console.error(`Error fetching ${endpoint}:`, error);
            container.innerHTML = `<div class="alert alert-warning"><strong>Gagal Memuat:</strong> ${error.message}</div>`;
        }
    };

    // Fungsi render generik untuk menampilkan data JSON mentah
    const renderJson = (data) => `<pre>${JSON.stringify(data, null, 2)}</pre>`;

    // Fungsi render yang lebih baik untuk daftar item
    const renderList = (data, titleKey = 'title', linkKey = 'link') => {
        if (data.length === 0) return '<p class="text-muted">Tidak ada data yang ditemukan.</p>';
        
        let listHtml = '<div class="list-group">';
        data.forEach(item => {
            const title = item[titleKey]?.rendered || item[titleKey] || item.name || '(No Title)';
            const link = item[linkKey] || '#';
            listHtml += `<a href="${link}" target="_blank" rel="noopener noreferrer" class="list-group-item list-group-item-action">${title} <small class="text-muted">(ID: ${item.id})</small></a>`;
        });
        listHtml += '</div>';
        return listHtml;
    };

    // Fungsi render untuk menampilkan nama dari objek
    const renderObjectKeys = (data) => {
        const keys = Object.keys(data);
        if (keys.length === 0) return '<p class="text-muted">Tidak ada data yang ditemukan.</p>';
        
        let badgesHtml = '<div class="d-flex flex-wrap gap-2">';
        keys.forEach(key => {
            badgesHtml += `<span class="badge bg-secondary">${key}</span>`;
        });
        badgesHtml += '</div>';
        return badgesHtml;
    };

    // --- Panggil semua fetch & render ---

    // Konten
    fetchAndRender('/posts?per_page=5&_embed', 'posts-container', (data) => renderList(data, 'title'));
    fetchAndRender('/pages?per_page=5', 'pages-container', (data) => renderList(data, 'title'));
    fetchAndRender('/media?per_page=5', 'media-container', (data) => renderList(data, 'title'));

    // Taksonomi
    fetchAndRender('/categories?per_page=5', 'categories-container', (data) => renderList(data, 'name'));
    fetchAndRender('/tags?per_page=5', 'tags-container', (data) => renderList(data, 'name'));
    
    // Pengguna
    fetchAndRender('/users?per_page=5', 'users-container', (data) => renderList(data, 'name'));

    // Lain-lain
    fetchAndRender('/comments?per_page=5', 'comments-container', (data) => renderList(data, 'author_name', 'link'));
    fetchAndRender('/settings', 'settings-container', renderJson);
    fetchAndRender('/types', 'types-container', renderObjectKeys);
    fetchAndRender('/taxonomies', 'taxonomies-container', renderObjectKeys);

});
</script>

<?php
require_once PROJECT_ROOT . '/includes/footer_public.php';
?>