<?php
// public/album.php (Detail of a single gallery album and its items)
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
$project_root = dirname(__DIR__);
require_once $project_root . '/includes/db_connect.php';
require_once $project_root . '/admin/includes/functions.php'; // For resolve_url, esc_html

$album_slug = $_GET['slug'] ?? null;
$album_data = null;
$gallery_items = [];
$site_title = "Detail Album Galeri";
$meta_description = "Lihat koleksi foto dan video dalam album ini.";

if (empty($album_slug)) {
    http_response_code(404);
    set_flash_message('danger', 'Album tidak ditemukan (slug kosong).');
    // header("Location: " . resolve_url('gallery.php')); // Redirect to gallery list
    // exit;
    $site_title = "Album Tidak Ditemukan";
}

if (isset($pdo) && !empty($album_slug)) {
    try {
        // Fetch album details
        $stmt_album = $pdo->prepare("SELECT * FROM portal_gallery_albums WHERE slug = :slug AND is_published = 1 LIMIT 1");
        $stmt_album->bindParam(':slug', $album_slug);
        $stmt_album->execute();
        $album_data = $stmt_album->fetch();

        if ($album_data) {
            $site_title = esc_html($album_data['title']);
            $meta_description = !empty($album_data['description']) ? esc_html(substr(strip_tags($album_data['description']), 0, 160)) : "Koleksi item dalam album " . esc_html($album_data['title']);

            // Fetch items for this album
            $stmt_items = $pdo->prepare("SELECT * FROM portal_gallery_items 
                                         WHERE album_id = :album_id 
                                         ORDER BY item_order ASC, created_at ASC");
            $stmt_items->bindParam(':album_id', $album_data['id'], PDO::PARAM_INT);
            $stmt_items->execute();
            $gallery_items = $stmt_items->fetchAll();
        } else {
            http_response_code(404);
            $site_title = "Album Tidak Ditemukan";
            $meta_description = "Album galeri yang Anda cari tidak ditemukan atau belum dipublikasikan.";
        }
    } catch (\PDOException $e) {
        error_log("Error fetching album detail: " . $e->getMessage());
        http_response_code(500);
        $site_title = "Kesalahan Server";
        $meta_description = "Terjadi kesalahan saat memuat detail album.";
    }
}

// Fetch global settings for header/footer
$settings = []; // Will be populated by header_public.php
$theme = $_COOKIE['theme'] ?? 'light-theme'; // For body class

// Include header
require_once $project_root . '/includes/header_public.php';
?>

<main class="container py-5">
    <div class="row">
        <div class="col-lg-12">
            <?php if ($album_data): ?>
                <header class="mb-4 text-center">
                    <h1 class="display-4 fw-bold album-title"><?php echo esc_html($album_data['title']); ?></h1>
                    <?php if ($album_data['album_date']): ?>
                        <p class="text-muted"><i class="fas fa-calendar-alt me-1"></i> Tanggal Album: <?php echo date('d F Y', strtotime($album_data['album_date'])); ?></p>
                    <?php endif; ?>
                    <?php if (!empty($album_data['description'])): ?>
                        <p class="lead text-muted album-description"><?php echo nl2br(esc_html($album_data['description'])); ?></p>
                    <?php endif; ?>
                </header>
                <hr class="my-4">

                <?php if (!empty($gallery_items)): ?>
                    <div class="row row-cols-1 row-cols-sm-2 row-cols-md-3 row-cols-lg-4 g-4 gallery-items-grid">
                        <?php foreach ($gallery_items as $item): ?>
                        <div class="col">
                            <div class="card h-100 shadow-sm gallery-item-card">
                                <?php if ($item['item_type'] === 'image' && !empty($item['item_url'])): ?>
                                    <a href="<?php echo resolve_url(esc_html($item['item_url'])); ?>" data-bs-toggle="lightbox" data-gallery="gallery-album-<?php echo $album_data['id']; ?>" data-title="<?php echo esc_html($item['title'] ?? ''); ?>" data-footer="<?php echo esc_html($item['caption'] ?? ''); ?>">
                                        <img src="<?php echo resolve_url(esc_html($item['item_url'])); ?>" class="card-img-top gallery-item-image" alt="<?php echo esc_html($item['title'] ?? 'Gallery Image'); ?>" style="height: 200px; object-fit: cover;">
                                    </a>
                                <?php elseif ($item['item_type'] === 'video_embed' && !empty($item['item_url'])): ?>
                                    <div class="ratio ratio-16x9 gallery-item-video-embed">
                                        <?php
                                        $allowed_video_hosts = ['youtube.com', 'youtube-nocookie.com', 'vimeo.com'];
                                        $parsed_url = parse_url($item['item_url']);
                                        $is_allowed_video = false;
                                        if (isset($parsed_url['host'])) {
                                            foreach ($allowed_video_hosts as $allowed_host) {
                                                if (stripos($parsed_url['host'], $allowed_host) !== false) {
                                                    $is_allowed_video = true;
                                                    break;
                                                }
                                            }
                                        }
                                        if ($is_allowed_video):
                                        ?>
                                        <iframe src="<?php echo esc_html($item['item_url']); ?>" title="<?php echo esc_html($item['title'] ?? 'Video Embed'); ?>" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe>
                                        <?php else: ?>
                                            <div class="p-3 text-center text-danger d-flex align-items-center justify-content-center" style="height: 200px; background-color:#f8d7da;">URL Video tidak valid/diizinkan.</div>
                                        <?php endif; ?>
                                    </div>
                                <?php else: ?>
                                     <div class="bg-secondary text-white d-flex align-items-center justify-content-center" style="height: 200px;">
                                        <i class="fas fa-photo-video fa-3x"></i>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($item['title']) || !empty($item['caption'])): ?>
                                <div class="card-body py-2 px-3">
                                    <?php if (!empty($item['title'])): ?>
                                        <h6 class="card-title small mb-1"><?php echo esc_html($item['title']); ?></h6>
                                    <?php endif; ?>
                                    <?php if (!empty($item['caption'])): ?>
                                        <p class="card-text very-small text-muted"><?php echo esc_html($item['caption']); ?></p>
                                    <?php endif; ?>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <p class="text-center text-muted mt-4">Tidak ada item (foto/video) di dalam album ini.</p>
                <?php endif; ?>
                
                <hr class="my-5">
                <div class="text-center">
                     <a href="<?php echo resolve_url('gallery.php'); ?>" class="btn btn-outline-secondary"><i class="fas fa-arrow-left me-1"></i> Kembali ke Daftar Album</a>
                </div>

            <?php else: ?>
                <div class="alert alert-warning text-center" role="alert">
                    <h4 class="alert-heading">Album Tidak Ditemukan</h4>
                    <p><?php echo $meta_description; ?></p>
                    <hr>
                    <p class="mb-0">Silakan kembali ke <a href="<?php echo resolve_url('gallery.php'); ?>" class="alert-link">daftar album galeri</a>.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</main>

<?php
// Lightbox JS (Bootstrap 5 Lightbox by Trvswgnr)
$public_page_scripts = [
    "https://cdn.jsdelivr.net/npm/bs5-lightbox@1.8.3/dist/index.bundle.min.js"
];
require_once $project_root . '/includes/footer_public.php';
?>
