<?php
// beranda/public/ajax_search_handler.php (Versi MeiliSearch)

header('Content-Type: application/json');
if (session_status() == PHP_SESSION_NONE) { session_start(); }

// Tentukan PROJECT_ROOT jika belum ada
if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__));
}

// Muat Composer Autoloader untuk library MeiliSearch
if (file_exists(PROJECT_ROOT . '/vendor/autoload.php')) {
    require_once PROJECT_ROOT . '/vendor/autoload.php';
} else {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Kesalahan Konfigurasi Server (Composer Autoload).']);
    exit;
}

// Muat file konfigurasi utama untuk BASE_URL, dll.
if (file_exists(PROJECT_ROOT . '/config.php')) {
    require_once PROJECT_ROOT . '/config.php';
} else {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Kesalahan Konfigurasi Server (Config).']);
    exit;
}

use MeiliSearch\Client;
use MeiliSearch\Exceptions\ApiException;

// --- Konfigurasi MeiliSearch ---
// (Sebaiknya pindahkan ini ke dalam file config.php Anda)
if (!defined('MEILISEARCH_URL')) {
    define('MEILISEARCH_URL', 'http://localhost:7700');
}
if (!defined('MEILISEARCH_MASTER_KEY')) {
    define('MEILISEARCH_MASTER_KEY', null); // Ganti dengan Master Key jika diset
}
if (!defined('MEILISEARCH_INDEX_NAME')) {
    define('MEILISEARCH_INDEX_NAME', 'ugkmb_content');
}

// --- Inisialisasi Klien MeiliSearch ---
try {
    $client = new Client(MEILISEARCH_URL, MEILISEARCH_MASTER_KEY);
    $index = $client->index(MEILISEARCH_INDEX_NAME);
} catch (\Exception $e) {
    http_response_code(503);
    error_log("MeiliSearch Connection Error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Layanan pencarian sedang tidak tersedia.']);
    exit;
}

$response = ['success' => false, 'message' => 'Parameter pencarian tidak valid.'];

// Ambil parameter dari request
$search_query = trim($_GET['q'] ?? '');
$limit = isset($_GET['limit']) ? max(1, (int)$_GET['limit']) : 20; // Default limit 20
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$offset = ($page - 1) * $limit;

if (empty($search_query)) {
    $response['message'] = 'Kata kunci pencarian tidak boleh kosong.';
    $response['results'] = [];
    $response['total_found'] = 0;
    echo json_encode($response);
    exit;
}

try {
    // Lakukan pencarian menggunakan MeiliSearch
    $searchResult = $index->search($search_query, [
        'limit' => $limit,
        'offset' => $offset,
        'attributesToHighlight' => ['title', 'excerpt'], // Field yang ingin di-highlight
        'showMatchesPosition' => true // Jika diperlukan untuk highlighting manual
    ]);

    $hits = $searchResult->getHits();
    $total_found = $searchResult->getEstimatedTotalHits();

    $results_formatted = [];
    foreach ($hits as $hit) {
        // Format hasil agar sesuai dengan yang diharapkan oleh JavaScript di halaman search.php
        $formatted_hit = [
            'type' => $hit['type'] ?? 'Unknown',
            'title' => $hit['_formatted']['title'] ?? esc_html($hit['title']),
            'url' => $hit['slug_url'] ?? '#',
            'excerpt' => $hit['_formatted']['excerpt'] ?? esc_html($hit['excerpt'] ?? ''),
            'image' => $hit['image_url'] ?? null,
            'date' => $hit['published_at_display'] ?? null,
            'category_name' => $hit['category_name'] ?? null,
            'category_slug' => $hit['category_slug'] ?? null,
            'relevance_score' => $hit['_rankingScore'] ?? 0 // MeiliSearch menyediakan skor relevansi
        ];
        $results_formatted[] = $formatted_hit;
    }

    $response['success'] = true;
    $response['message'] = $total_found > 0 ? 'Pencarian berhasil.' : 'Tidak ada hasil yang cocok ditemukan.';
    $response['results'] = $results_formatted;
    $response['total_found'] = $total_found;
    $response['current_page'] = $page;
    $response['limit'] = $limit;

} catch (ApiException $e) {
    http_response_code(500);
    error_log("MeiliSearch API Error for query '{$search_query}': " . $e->getMessage());
    $response['message'] = 'Terjadi kesalahan pada layanan pencarian.';
} catch (\Exception $e) {
    http_response_code(500);
    error_log("General Search Error for query '{$search_query}': " . $e->getMessage());
    $response['message'] = 'Terjadi kesalahan sistem saat memproses pencarian.';
}

echo json_encode($response);
exit;