<?php
// index_content_to_meilisearch.php

// Aktifkan error reporting untuk development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Set batas waktu eksekusi jika datanya banyak
set_time_limit(0); // 0 = tidak ada batas waktu
ini_set('memory_limit', '5012M'); // Naikkan jika perlu

// Tentukan PROJECT_ROOT jika script ini ada di root proyek atau di direktori khusus
if (!defined('PROJECT_ROOT')) {
    // Jika file ini di root proyek: define('PROJECT_ROOT', __DIR__);
    // Jika file ini misal di /scripts/ maka: define('PROJECT_ROOT', dirname(__DIR__));
    define('PROJECT_ROOT', __DIR__); // Asumsi di root proyek, sesuaikan jika perlu
}

// Load Composer Autoloader
if (file_exists(PROJECT_ROOT . '/vendor/autoload.php')) {
    require_once PROJECT_ROOT . '/vendor/autoload.php';
} else {
    die("Composer autoload.php tidak ditemukan. Pastikan Anda sudah menjalankan 'composer install'.\n");
}

// Load konfigurasi database dan fungsi dasar
if (file_exists(PROJECT_ROOT . '/config.php')) {
    require_once PROJECT_ROOT . '/config.php'; // config.php harus memuat db_connect.php
} else {
    die("File config.php tidak ditemukan.\n");
}

use MeiliSearch\Client;
use MeiliSearch\Exceptions\ApiException;

// --- Konfigurasi MeiliSearch ---
define('MEILISEARCH_URL', 'http://localhost:7700'); // Ganti dengan URL MeiliSearch Anda
define('MEILISEARCH_MASTER_KEY', null); // Ganti dengan Master API Key Anda jika diset, atau null jika tidak ada
define('MEILISEARCH_INDEX_NAME', 'ugkmb_content'); // Nama index yang akan digunakan di MeiliSearch
$documents_per_batch = 500; // Jumlah dokumen yang dikirim ke MeiliSearch per batch

// Inisialisasi MeiliSearch Client
try {
    $client = new Client(MEILISEARCH_URL, MEILISEARCH_MASTER_KEY);
} catch (\Exception $e) {
    die("Gagal terhubung ke MeiliSearch: " . $e->getMessage() . "\n");
}

// Inisialisasi PDO (diasumsikan $pdo sudah ada dari config.php -> db_connect.php)
if (!isset($pdo)) {
    die("Objek PDO tidak tersedia. Pastikan db_connect.php termuat dengan benar oleh config.php.\n");
}

echo "Memulai proses indexing ke MeiliSearch...\n";
echo "URL MeiliSearch: " . MEILISEARCH_URL . "\n";
echo "Index Target: " . MEILISEARCH_INDEX_NAME . "\n";

try {
    // 1. Dapatkan atau Buat Index
    $index = $client->index(MEILISEARCH_INDEX_NAME);
    try {
        $index->fetchInfo(); // Cek apakah index sudah ada
        echo "Index '" . MEILISEARCH_INDEX_NAME . "' sudah ada. Mengupdate...\n";
    } catch (ApiException $e) {
        if ($e->getCode() == 404) { // Index not found
            echo "Index '" . MEILISEARCH_INDEX_NAME . "' tidak ditemukan. Membuat index baru...\n";
            $client->createIndex(MEILISEARCH_INDEX_NAME, ['primaryKey' => 'uid']); // uid akan jadi primary key kita
            $index = $client->index(MEILISEARCH_INDEX_NAME); // Dapatkan objek index lagi
        } else {
            throw $e; // Lempar error lain
        }
    }

    // 2. Konfigurasi Index di MeiliSearch (Filterable, Sortable, Searchable Attributes)
    // Ini sebaiknya dilakukan sekali atau jika ada perubahan skema
    // Anda bisa menjalankan ini secara terpisah atau menyertakannya di sini
    echo "Mengkonfigurasi index settings...\n";
    $index->updateSettings([
        'searchableAttributes' => [ // Field mana saja yang bisa dicari teksnya
            'title',
            'content',
            'excerpt',
            'category_name',
            'tags_string', // Jika Anda menggabungkan tag
            'faculty_name',
            'program_name',
            'author_name',
            'meta_keywords',
            'file_name_original',
            'organizer',
            'location'
        ],
        'filterableAttributes' => [ // Field yang bisa digunakan untuk filtering
            'type',
            'status',
            'category_slug',
            'tag_slugs', // Jika Anda punya array tag slug
            'faculty_slug',
            'program_slug',
            'language_code', // Jika ada translasi
            'is_public',
            'is_published',
            'file_mime_type'
        ],
        'sortableAttributes' => [ // Field yang bisa digunakan untuk sorting
            'published_at',
            'updated_at',
            'created_at',
            'start_datetime', // Untuk events
            'title_sortable', // Buat field khusus jika ingin sort title case-insensitive
            'views_count',
            'relevance' // MeiliSearch akan menambahkan skor relevansi secara internal
        ],
        'displayedAttributes' => [ // Field yang akan dikembalikan dalam hasil pencarian
            'uid', 'type', 'title', 'slug_url', 'excerpt', 'content_snippet', 
            'image_url', 'published_at_display', 'category_name', 'category_slug',
            'faculty_name', 'program_name', 'author_name', 'tags_display',
            'location', 'organizer', 'start_datetime_display', 'file_mime_type'
        ],
        'rankingRules' => [ // Default ranking rules MeiliSearch biasanya sudah baik
            'words',
            'typo',
            'proximity',
            'attribute',
            'sort',
            'exactness',
            // Tambahkan kriteria custom jika perlu, misal:
            // 'published_at:desc', // Berita lebih baru lebih penting
            // 'views_count:desc'
        ],
        'stopWords' => [ // Tambahkan stop words Bahasa Indonesia jika perlu
            'yang', 'di', 'ke', 'dari', 'dan', 'atau', 'ini', 'itu', 'adalah', 'untuk', 'dengan', 'sebagai' 
            // ... (lebih banyak lagi) ...
        ],
        // 'synonyms' => [ // Jika Anda punya daftar sinonim
        //     'ugm' => ['universitas gajah mada', 'gadjah mada'],
        // ],
        // Untuk Bahasa Indonesia, aktifkan tokenizer yang sesuai jika MeiliSearch Anda mendukungnya
        // 'facetedAttributes' => ['type', 'category_name', 'faculty_name'] // Sama dengan filterableAttributes jika ingin facet
    ]);
    echo "Konfigurasi index berhasil diupdate/diset.\n";


    // 3. Hapus semua dokumen lama dari index (opsional, jika ingin full re-index)
    // $index->deleteAllDocuments();
    // echo "Semua dokumen lama telah dihapus dari index.\n";

    $documents_batch = [];
    $total_indexed_count = 0;

    // --- Helper Function untuk membersihkan HTML dan membuat excerpt ---
    function create_excerpt($html_content, $length = 200) {
        if (empty($html_content)) return '';
        $text = strip_tags($html_content);
        $text = preg_replace('/\s+/', ' ', $text); // Ganti spasi ganda dengan tunggal
        $text = trim($text);
        if (mb_strlen($text) > $length) {
            $text = mb_substr($text, 0, $length) . '...';
        }
        return $text;
    }
    function get_content_snippet($html_content, $length = 150) { // Untuk MeiliSearch highlight
        if (empty($html_content)) return '';
        $text = strip_tags($html_content);
        $text = preg_replace('/\s+/', ' ', $text);
        return trim($text); // MeiliSearch akan menangani snippet berdasarkan query
    }


    // --- Indexing Articles ---
    echo "Mengindeks Artikel...\n";
    $stmt_articles = $pdo->query("SELECT 
                                    a.id, a.title, a.slug, a.content_html, a.excerpt, a.featured_image_url, 
                                    a.published_at, a.updated_at, a.created_at, a.views_count,
                                    u.full_name as author_name, 
                                    c.name as category_name, c.slug as category_slug
                                  FROM portal_articles a
                                  LEFT JOIN portal_users u ON a.author_id = u.id
                                  LEFT JOIN portal_categories c ON a.category_id = c.id
                                  WHERE a.status = 'published'");
    while ($row = $stmt_articles->fetch(PDO::FETCH_ASSOC)) {
        $document = [
            'uid' => 'article_' . $row['id'], // ID unik global
            'type' => 'Artikel',
            'title' => $row['title'],
            'title_sortable' => strtolower($row['title']), // Untuk sort case-insensitive
            'slug_url' => resolve_url('public/article.php?slug=' . $row['slug']),
            'content' => get_content_snippet($row['content_html']),
            'excerpt' => create_excerpt($row['excerpt'] ?: $row['content_html'], 250),
            'image_url' => !empty($row['featured_image_url']) ? resolve_url($row['featured_image_url']) : null,
            'published_at' => $row['published_at'] ? strtotime($row['published_at']) : null, // Timestamp UNIX
            'published_at_display' => $row['published_at'] ? date('d M Y', strtotime($row['published_at'])) : null,
            'updated_at' => strtotime($row['updated_at']),
            'created_at' => strtotime($row['created_at']),
            'views_count' => (int)$row['views_count'],
            'author_name' => $row['author_name'],
            'category_name' => $row['category_name'],
            'category_slug' => $row['category_slug'],
            'status' => 'published', // Asumsi kita hanya indeks yang published
            // Anda bisa tambahkan tags jika ada
            // 'tags_string' => 'tag1, tag2', // Gabungkan tag jadi string
            // 'tag_slugs' => ['tag-1-slug', 'tag-2-slug'], // Array slug untuk filter
        ];
        $documents_batch[] = $document;
        if (count($documents_batch) >= $documents_per_batch) {
            $task = $index->addDocuments($documents_batch, 'uid');
            echo "Batch artikel dikirim. Task ID: " . ($task['taskUid'] ?? $task['uid']) . "\n"; // Versi SDK lama 'uid', baru 'taskUid'
            $total_indexed_count += count($documents_batch);
            $documents_batch = []; // Reset batch
            // $client->waitForTask($task['taskUid'] ?? $task['uid']); // Tunggu task selesai (opsional, bisa memperlambat)
        }
    }
    if (!empty($documents_batch)) { // Kirim sisa batch terakhir
        $task = $index->addDocuments($documents_batch, 'uid');
        echo "Batch artikel terakhir dikirim. Task ID: " . ($task['taskUid'] ?? $task['uid']) . "\n";
        $total_indexed_count += count($documents_batch);
        $documents_batch = [];
        // $client->waitForTask($task['taskUid'] ?? $task['uid']);
    }
    echo "Selesai mengindeks Artikel.\n";


    // --- Indexing Pages ---
    echo "Mengindeks Halaman Statis...\n";
    $stmt_pages = $pdo->query("SELECT 
                                p.id, p.title, p.slug, p.content_html, p.published_at, p.updated_at, p.created_at, p.views_count
                              FROM portal_pages p
                              WHERE p.status = 'published'");
    while ($row = $stmt_pages->fetch(PDO::FETCH_ASSOC)) {
         $document = [
            'uid' => 'page_' . $row['id'],
            'type' => 'Halaman',
            'title' => $row['title'],
            'title_sortable' => strtolower($row['title']),
            'slug_url' => resolve_url('public/page.php?slug=' . $row['slug']),
            'content' => get_content_snippet($row['content_html']),
            'excerpt' => create_excerpt($row['content_html'], 250),
            'image_url' => null, // Halaman biasanya tidak punya featured image utama
            'published_at' => $row['published_at'] ? strtotime($row['published_at']) : strtotime($row['updated_at']), // Gunakan updated_at jika published_at null
            'published_at_display' => $row['published_at'] ? date('d M Y', strtotime($row['published_at'])) : date('d M Y', strtotime($row['updated_at'])),
            'updated_at' => strtotime($row['updated_at']),
            'created_at' => strtotime($row['created_at']),
            'views_count' => (int)$row['views_count'],
            'status' => 'published',
        ];
        $documents_batch[] = $document;
        if (count($documents_batch) >= $documents_per_batch) {
            $task = $index->addDocuments($documents_batch, 'uid');
            echo "Batch halaman dikirim. Task ID: " . ($task['taskUid'] ?? $task['uid']) . "\n";
            $total_indexed_count += count($documents_batch);
            $documents_batch = [];
        }
    }
    if (!empty($documents_batch)) {
        $task = $index->addDocuments($documents_batch, 'uid');
        echo "Batch halaman terakhir dikirim. Task ID: " . ($task['taskUid'] ?? $task['uid']) . "\n";
        $total_indexed_count += count($documents_batch);
        $documents_batch = [];
    }
    echo "Selesai mengindeks Halaman Statis.\n";


    // --- Indexing Documents ---
    echo "Mengindeks Dokumen...\n";
    $stmt_docs = $pdo->query("SELECT 
                                d.id, d.title, d.slug, d.description, d.file_url, d.file_name_original, d.file_mime_type, 
                                d.publish_date, d.updated_at, d.created_at, d.download_count,
                                dt.name as document_type_name,
                                c.name as category_name 
                              FROM portal_documents d
                              LEFT JOIN portal_document_types dt ON d.document_type_id = dt.id
                              LEFT JOIN portal_categories c ON d.category_id = c.id
                              WHERE d.is_public = 1");
    while ($row = $stmt_docs->fetch(PDO::FETCH_ASSOC)) {
         $document = [
            'uid' => 'document_' . $row['id'],
            'type' => 'Dokumen',
            'title' => $row['title'],
            'title_sortable' => strtolower($row['title']),
            'slug_url' => resolve_url($row['file_url']), // Link langsung ke file
            'content' => get_content_snippet($row['description'] . " " . $row['file_name_original']), // Gabungkan deskripsi dan nama file untuk dicari
            'excerpt' => create_excerpt($row['description'], 250),
            'image_url' => null, // Dokumen biasanya tidak punya preview image
            'published_at' => $row['publish_date'] ? strtotime($row['publish_date']) : strtotime($row['created_at']),
            'published_at_display' => $row['publish_date'] ? date('d M Y', strtotime($row['publish_date'])) : date('d M Y', strtotime($row['created_at'])),
            'updated_at' => strtotime($row['updated_at']),
            'created_at' => strtotime($row['created_at']),
            'download_count' => (int)$row['download_count'],
            'file_name_original' => $row['file_name_original'],
            'file_mime_type' => $row['file_mime_type'],
            'document_type_name' => $row['document_type_name'],
            'category_name' => $row['category_name'], // Kategori umum jika ada
            'status' => 'public',
        ];
        $documents_batch[] = $document;
        if (count($documents_batch) >= $documents_per_batch) {
            $task = $index->addDocuments($documents_batch, 'uid');
            echo "Batch dokumen dikirim. Task ID: " . ($task['taskUid'] ?? $task['uid']) . "\n";
            $total_indexed_count += count($documents_batch);
            $documents_batch = [];
        }
    }
    if (!empty($documents_batch)) {
        $task = $index->addDocuments($documents_batch, 'uid');
        echo "Batch dokumen terakhir dikirim. Task ID: " . ($task['taskUid'] ?? $task['uid']) . "\n";
        $total_indexed_count += count($documents_batch);
        $documents_batch = [];
    }
    echo "Selesai mengindeks Dokumen.\n";
    
    // --- Indexing Events ---
    echo "Mengindeks Agenda/Events...\n";
    $stmt_events = $pdo->query("SELECT 
                                e.id, e.title, e.slug, e.description_html, e.start_datetime, e.end_datetime, 
                                e.location, e.organizer, e.featured_image_url, e.updated_at, e.created_at
                              FROM portal_events e
                              WHERE e.is_published = 1");
    while ($row = $stmt_events->fetch(PDO::FETCH_ASSOC)) {
         $document = [
            'uid' => 'event_' . $row['id'],
            'type' => 'Agenda',
            'title' => $row['title'],
            'title_sortable' => strtolower($row['title']),
            'slug_url' => resolve_url('public/event.php?slug=' . $row['slug']),
            'content' => get_content_snippet($row['description_html']),
            'excerpt' => create_excerpt($row['description_html'], 250),
            'image_url' => !empty($row['featured_image_url']) ? resolve_url($row['featured_image_url']) : null,
            'published_at' => strtotime($row['created_at']), // Atau start_datetime jika lebih relevan untuk "baru"
            'published_at_display' => date('d M Y', strtotime($row['created_at'])),
            'start_datetime' => strtotime($row['start_datetime']),
            'start_datetime_display' => date('d M Y, H:i', strtotime($row['start_datetime'])),
            'end_datetime' => $row['end_datetime'] ? strtotime($row['end_datetime']) : null,
            'updated_at' => strtotime($row['updated_at']),
            'created_at' => strtotime($row['created_at']),
            'location' => $row['location'],
            'organizer' => $row['organizer'],
            'status' => 'published',
        ];
        $documents_batch[] = $document;
        if (count($documents_batch) >= $documents_per_batch) {
            $task = $index->addDocuments($documents_batch, 'uid');
            echo "Batch agenda dikirim. Task ID: " . ($task['taskUid'] ?? $task['uid']) . "\n";
            $total_indexed_count += count($documents_batch);
            $documents_batch = [];
        }
    }
    if (!empty($documents_batch)) {
        $task = $index->addDocuments($documents_batch, 'uid');
        echo "Batch agenda terakhir dikirim. Task ID: " . ($task['taskUid'] ?? $task['uid']) . "\n";
        $total_indexed_count += count($documents_batch);
        $documents_batch = [];
    }
    echo "Selesai mengindeks Agenda/Events.\n";

    // --- Indexing Faculties & Study Programs (Jika ingin bisa dicari terpisah) ---
    // Anda bisa memilih untuk mengindeks fakultas dan prodi sebagai tipe dokumen sendiri,
    // atau informasi mereka sudah terwakili dalam artikel/halaman terkait.
    // Jika diindeks sendiri, pastikan field 'content' atau 'excerpt' diisi dengan deskripsi yang relevan.

    echo "\nProses indexing selesai. Total {$total_indexed_count} dokumen coba diindeks.\n";
    echo "Anda bisa memantau status task di dashboard MeiliSearch Anda.\n";

} catch (ApiException $e) {
    echo "MeiliSearch API Error: " . $e->getMessage() . "\n";
    echo "HTTP Status: " . $e->getHttpStatus() . "\n";
    echo "Error Code: " . $e->getErrorCode() . "\n";
    echo "Error Type: " . $e->getErrorType() . "\n";
    echo "Link: " . $e->getLink() . "\n";
} catch (\PDOException $e) {
    echo "Database Error: " . $e->getMessage() . "\n";
} catch (\Exception $e) {
    echo "General Error: " . $e->getMessage() . "\n";
}

?>