document.addEventListener('DOMContentLoaded', function () {

    /**
     * --------------------------------------------------------------------------
     * Animate On Scroll (AOS-like functionality with Intersection Observer)
     * --------------------------------------------------------------------------
     *
     * Cara kerja:
     * 1. Cari semua elemen yang memiliki kelas `.animate-on-scroll`.
     * 2. Untuk setiap elemen tersebut, amati kapan elemen masuk ke viewport.
     * 3. Ketika elemen masuk viewport:
     *    a. Tambahkan kelas animasi yang didefinisikan di atribut `data-animation-class`.
     *       Contoh: data-animation-class="animate__fadeInUp"
     *    b. (Opsional) Tambahkan kelas delay yang didefinisikan di `data-animation-delay`.
     *       Contoh: data-animation-delay="animate__delay-1s"
     *    c. Hentikan pengamatan pada elemen tersebut agar animasi tidak berulang (kecuali jika diinginkan).
     */

    const animatedElements = document.querySelectorAll('.animate-on-scroll');

    if (animatedElements.length > 0) {
        const observerOptions = {
            root: null, // relatif terhadap viewport
            rootMargin: '0px',
            threshold: 0.1 // minimal 10% elemen terlihat sebelum trigger
        };

        const animationObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const element = entry.target;
                    const animationClass = element.dataset.animationClass;
                    const animationDelay = element.dataset.animationDelay;
                    const animationDuration = element.dataset.animationDuration;
                    const animationIteration = element.dataset.animationIteration;

                    if (animationClass) {
                        element.classList.add('animate__animated', animationClass);
                    }
                    if (animationDelay) {
                        element.classList.add(animationDelay);
                    }
                    if (animationDuration) {
                        element.classList.add(animationDuration); // e.g., animate__fast, animate__slower
                    }
                    if (animationIteration) {
                        element.classList.add(animationIteration); // e.g., animate__infinite
                    }


                    // Hapus kelas 'animate-on-scroll' agar tidak diobservasi lagi
                    // dan juga agar opacity/visibility kembali normal setelah animasi
                    element.classList.remove('animate-on-scroll');

                    // Tambahkan kelas 'is-visible' sebagai penanda bahwa elemen sudah dianimasikan
                    element.classList.add('is-visible');

                    // Berhenti mengamati elemen ini agar animasi tidak terulang saat scroll bolak-balik
                    // Kecuali jika Anda ingin animasi berulang, maka baris ini bisa di-komen.
                    // Untuk performa, lebih baik hanya sekali.
                    observer.unobserve(element);
                }
            });
        }, observerOptions);

        animatedElements.forEach(el => {
            // Sembunyikan elemen terlebih dahulu sampai animasi dijalankan
            // Anda bisa juga mengatur ini via CSS: .animate-on-scroll { opacity: 0; }
            // el.style.opacity = '0'; // Ini bisa menyebabkan 'flash' jika JS lambat dimuat

            animationObserver.observe(el);
        });
    }

    /**
     * --------------------------------------------------------------------------
     * Smooth Scroll untuk link dengan hash (#)
     * --------------------------------------------------------------------------
     */
    const smoothScrollLinks = document.querySelectorAll('a[href^="#"]');
    smoothScrollLinks.forEach(link => {
        link.addEventListener('click', function (e) {
            const hash = this.getAttribute('href');
            if (hash !== '#' && hash !== '#!') { // Abaikan link # atau #! yang biasa untuk JavaScript
                const targetElement = document.querySelector(hash);
                if (targetElement) {
                    e.preventDefault();
                    targetElement.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });

                    // Optional: update URL hash without page jump
                    // history.pushState(null, null, hash);
                }
            }
        });
    });


    /**
     * --------------------------------------------------------------------------
     * Inisialisasi komponen Bootstrap lain jika ada, contoh: Tooltip atau Popover
     * --------------------------------------------------------------------------
     */
    // Contoh Tooltip (jika Anda menggunakannya)
    // const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
    // tooltipTriggerList.map(function (tooltipTriggerEl) {
    //   return new bootstrap.Tooltip(tooltipTriggerEl)
    // })

    // Contoh Popover (jika Anda menggunakannya)
    // const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'))
    // popoverTriggerList.map(function (popoverTriggerEl) {
    //   return new bootstrap.Popover(popoverTriggerEl)
    // })


    console.log("Homepage animations script loaded and initialized.");
});