<?php
// api/v1/events.php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__, 2));
}
require_once PROJECT_ROOT . '/includes/db_connect.php';

// Basic API Key Check (Example)
// $api_key_header = $_SERVER['HTTP_X_API_KEY'] ?? null;
// $valid_api_key = "YOUR_SECRET_API_KEY_FOR_EVENTS"; 
// if ($api_key_header !== $valid_api_key) {
//     http_response_code(401); 
//     echo json_encode(["error" => "Unauthorized: Invalid API Key"]);
//     exit;
// }

if (!isset($pdo)) {
    http_response_code(503);
    echo json_encode(["error" => "Database service unavailable."]);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        if (isset($_GET['slug'])) {
            get_event_by_slug($pdo, $_GET['slug']);
        } elseif (isset($_GET['id'])) {
            get_event_by_id($pdo, (int)$_GET['id']);
        } else {
            get_all_events($pdo);
        }
        break;
    default:
        http_response_code(405);
        echo json_encode(["error" => "Method not allowed."]);
        break;
}

function get_all_events($pdo_conn) {
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 10;
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $offset = ($page - 1) * $limit;
    $filter = $_GET['filter'] ?? 'upcoming'; // upcoming, past, all

    $sql_where_parts = ["e.is_published = 1"];
    $sql_order = "ORDER BY e.start_datetime ";

    if ($filter === 'upcoming') {
        $sql_where_parts[] = "(e.end_datetime IS NULL OR e.end_datetime >= NOW()) AND e.start_datetime >= CURDATE()";
        $sql_order .= "ASC";
    } elseif ($filter === 'past') {
        $sql_where_parts[] = "(e.end_datetime < NOW() OR (e.end_datetime IS NULL AND e.start_datetime < CURDATE()))";
        $sql_order .= "DESC";
    } else { // 'all'
        $sql_order .= "DESC";
    }
    
    $sql_where = "WHERE " . implode(" AND ", $sql_where_parts);

    $sql_events = "SELECT e.id, e.title, e.slug, e.description_html, e.start_datetime, e.end_datetime, e.location, e.featured_image_url, e.organizer, e.event_url
                   FROM portal_events e
                   {$sql_where} {$sql_order}
                   LIMIT :limit OFFSET :offset";
    
    $sql_count = "SELECT COUNT(*) FROM portal_events e {$sql_where}";

    try {
        $stmt = $pdo_conn->prepare($sql_events);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $events = $stmt->fetchAll();

        $stmt_total = $pdo_conn->query($sql_count);
        $total_count = $stmt_total->fetchColumn();

        if ($events) {
            http_response_code(200);
            echo json_encode([
                "message" => "Events fetched successfully.",
                "data" => $events,
                "pagination" => [
                    "currentPage" => $page,
                    "perPage" => $limit,
                    "totalItems" => (int)$total_count,
                    "totalPages" => ceil($total_count / $limit)
                ]
            ]);
        } else {
            http_response_code(404);
            echo json_encode(["message" => "No events found matching criteria."]);
        }
    } catch (\PDOException $e) {
        http_response_code(500);
        echo json_encode(["error" => "Database error: " . $e->getMessage()]);
    }
}

function get_event_by_slug($pdo_conn, $slug) {
    $sql = "SELECT * FROM portal_events WHERE slug = :slug AND is_published = 1 LIMIT 1";
    try {
        $stmt = $pdo_conn->prepare($sql);
        $stmt->bindParam(':slug', $slug);
        $stmt->execute();
        $event = $stmt->fetch();
        if ($event) {
            http_response_code(200);
            echo json_encode(["message" => "Event fetched successfully.", "data" => $event]);
        } else {
            http_response_code(404);
            echo json_encode(["message" => "Event not found or not published."]);
        }
    } catch (\PDOException $e) {
        http_response_code(500);
        echo json_encode(["error" => "Database error: " . $e->getMessage()]);
    }
}

function get_event_by_id($pdo_conn, $id) {
    $sql = "SELECT * FROM portal_events WHERE id = :id AND is_published = 1 LIMIT 1";
     try {
        $stmt = $pdo_conn->prepare($sql);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $event = $stmt->fetch();
        if ($event) {
            http_response_code(200);
            echo json_encode(["message" => "Event fetched successfully.", "data" => $event]);
        } else {
            http_response_code(404);
            echo json_encode(["message" => "Event not found or not published."]);
        }
    } catch (\PDOException $e) {
        http_response_code(500);
        echo json_encode(["error" => "Database error: " . $e->getMessage()]);
    }
}

?>