<?php
// api/v1/articles.php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *"); // Adjust for production if needed
header("Access-Control-Allow-Methods: GET, POST, OPTIONS"); // OPTIONS for preflight
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle OPTIONS request (preflight)
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Define PROJECT_ROOT if not already defined
if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__, 2)); // api/v1/ -> project root
}

require_once PROJECT_ROOT . '/includes/db_connect.php'; // For $pdo

// Basic API Key Check (Example - replace with a more robust mechanism)
$api_key_header = $_SERVER['HTTP_X_API_KEY'] ?? null;
$valid_api_key = "YOUR_SECRET_API_KEY"; // Store this securely, not hardcoded

// if ($api_key_header !== $valid_api_key) {
//     http_response_code(401); // Unauthorized
//     echo json_encode(["error" => "Unauthorized: Invalid API Key"]);
//     exit;
// }


if (!isset($pdo)) {
    http_response_code(503); // Service Unavailable
    echo json_encode(["error" => "Database service unavailable."]);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        // Get all articles or a specific article
        if (isset($_GET['slug'])) {
            get_article_by_slug($pdo, $_GET['slug']);
        } elseif (isset($_GET['id'])) {
            get_article_by_id($pdo, (int)$_GET['id']);
        } else {
            get_all_articles($pdo);
        }
        break;
    // POST, PUT, DELETE can be added for more complete API
    // For now, keeping it simple with GET
    default:
        http_response_code(405); // Method Not Allowed
        echo json_encode(["error" => "Method not allowed."]);
        break;
}

function get_all_articles($pdo_conn) {
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 10;
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $offset = ($page - 1) * $limit;
    $category_slug = $_GET['category_slug'] ?? null;
    $tag_slug = $_GET['tag_slug'] ?? null;

    $sql = "SELECT a.id, a.title, a.slug, a.excerpt, a.featured_image_url, a.published_at, c.name as category_name, c.slug as category_slug
            FROM portal_articles a
            LEFT JOIN portal_categories c ON a.category_id = c.id ";
    $where_clauses = ["a.status = 'published'", "a.published_at <= NOW()"];
    $params = [];

    if ($category_slug) {
        $sql .= " AND c.slug = :category_slug ";
        $params[':category_slug'] = $category_slug;
    }
    if ($tag_slug) {
        $sql .= " JOIN portal_article_tag pat ON a.id = pat.article_id
                  JOIN portal_tags t ON pat.tag_id = t.id AND t.slug = :tag_slug ";
        $params[':tag_slug'] = $tag_slug;
    }
    
    $sql_where = "";
    if (count($where_clauses) > 0) {
        $sql_where = " WHERE " . implode(" AND ", $where_clauses);
    }
    $sql .= $sql_where;
    $sql .= " ORDER BY a.published_at DESC LIMIT :limit OFFSET :offset";
    
    $params[':limit'] = $limit;
    $params[':offset'] = $offset;

    try {
        $stmt = $pdo_conn->prepare($sql);
        $stmt->execute($params);
        $articles = $stmt->fetchAll();

        // Count total for pagination (simplified, ideally adjust query for full count)
        $stmt_total = $pdo_conn->prepare("SELECT COUNT(*) FROM portal_articles a LEFT JOIN portal_categories c ON a.category_id = c.id " . ($tag_slug ? "JOIN portal_article_tag pat ON a.id = pat.article_id JOIN portal_tags t ON pat.tag_id = t.id AND t.slug = :tag_slug" : "") . $sql_where);
        $count_params = [];
        if($category_slug) $count_params[':category_slug'] = $category_slug;
        if($tag_slug && !isset($count_params[':tag_slug'])) $count_params[':tag_slug'] = $tag_slug; // Add if not already added by category filter
        $stmt_total->execute($count_params);
        $total_count = $stmt_total->fetchColumn();


        if ($articles) {
            http_response_code(200);
            echo json_encode([
                "message" => "Articles fetched successfully.",
                "data" => $articles,
                "pagination" => [
                    "currentPage" => $page,
                    "perPage" => $limit,
                    "totalItems" => (int)$total_count,
                    "totalPages" => ceil($total_count / $limit)
                ]
            ]);
        } else {
            http_response_code(404);
            echo json_encode(["message" => "No articles found."]);
        }
    } catch (\PDOException $e) {
        http_response_code(500);
        echo json_encode(["error" => "Database error: " . $e->getMessage()]);
    }
}

function get_article_by_slug($pdo_conn, $slug) {
    $sql = "SELECT a.*, c.name as category_name, u.full_name as author_name
            FROM portal_articles a
            LEFT JOIN portal_categories c ON a.category_id = c.id
            LEFT JOIN portal_users u ON a.author_id = u.id
            WHERE a.slug = :slug AND a.status = 'published' AND a.published_at <= NOW() LIMIT 1";
    try {
        $stmt = $pdo_conn->prepare($sql);
        $stmt->bindParam(':slug', $slug);
        $stmt->execute();
        $article = $stmt->fetch();
        if ($article) {
            // Optionally, fetch tags
            $stmt_tags = $pdo_conn->prepare("SELECT t.name, t.slug FROM portal_tags t JOIN portal_article_tag pat ON t.id = pat.tag_id WHERE pat.article_id = :article_id");
            $stmt_tags->execute([':article_id' => $article['id']]);
            $article['tags'] = $stmt_tags->fetchAll();

            http_response_code(200);
            echo json_encode(["message" => "Article fetched successfully.", "data" => $article]);
        } else {
            http_response_code(404);
            echo json_encode(["message" => "Article not found or not published."]);
        }
    } catch (\PDOException $e) {
        http_response_code(500);
        echo json_encode(["error" => "Database error: " . $e->getMessage()]);
    }
}

function get_article_by_id($pdo_conn, $id) {
    // Similar to get_article_by_slug but using ID
    $sql = "SELECT a.*, c.name as category_name, u.full_name as author_name
            FROM portal_articles a
            LEFT JOIN portal_categories c ON a.category_id = c.id
            LEFT JOIN portal_users u ON a.author_id = u.id
            WHERE a.id = :id AND a.status = 'published' AND a.published_at <= NOW() LIMIT 1";
    try {
        $stmt = $pdo_conn->prepare($sql);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $article = $stmt->fetch();
         if ($article) {
            $stmt_tags = $pdo_conn->prepare("SELECT t.name, t.slug FROM portal_tags t JOIN portal_article_tag pat ON t.id = pat.tag_id WHERE pat.article_id = :article_id");
            $stmt_tags->execute([':article_id' => $article['id']]);
            $article['tags'] = $stmt_tags->fetchAll();
            http_response_code(200);
            echo json_encode(["message" => "Article fetched successfully.", "data" => $article]);
        } else {
            http_response_code(404);
            echo json_encode(["message" => "Article not found or not published."]);
        }
    } catch (\PDOException $e) {
        http_response_code(500);
        echo json_encode(["error" => "Database error: " . $e->getMessage()]);
    }
}

?>