

<?php
// admin/modules/menus/process_menu_group.php

// Pastikan config.php dimuat paling awal dan hanya sekali
if (file_exists(dirname(__DIR__, 3) . '/config.php')) {
    require_once dirname(__DIR__, 3) . '/config.php';
} elseif (file_exists(dirname(__FILE__, 4) . '/config.php')) { 
    require_once dirname(__FILE__, 4) . '/config.php';
} else {
    error_log("CRITICAL FAILURE: config.php not found from process_menu_group.php. Path checked: " . dirname(__DIR__, 3) . '/config.php' . " AND " . dirname(__FILE__, 4) . '/config.php');
    // Jangan output JSON jika config saja tidak ada, karena header belum tentu diset
    die("Konfigurasi sistem tidak ditemukan. Harap hubungi administrator.");
}

// Fungsi untuk deteksi AJAX (jika belum ada di functions.php global)
if (!function_exists('is_ajax_request')) {
    function is_ajax_request() {
        return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
    }
}

$is_ajax = is_ajax_request();

// Set header JSON jika AJAX, tapi hanya jika belum ada output lain
if ($is_ajax && !headers_sent()) {
    header('Content-Type: application/json');
}

// Respons default, selalu sertakan token CSRF baru
$response = ['success' => false, 'message' => 'Aksi tidak dikenal atau gagal diproses.', 'new_csrf_token' => generate_csrf_token()]; 

// Pengecekan Sesi dan Role Admin
if (!isset($_SESSION['user_id']) || !isset($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    $response['message'] = 'Akses ditolak. Anda tidak memiliki izin untuk melakukan aksi ini.';
    if ($is_ajax) { 
        http_response_code(403); // Forbidden
        echo json_encode($response); 
        exit; 
    }
    set_flash_message('danger', $response['message']);
    header('Location: ' . admin_url('login.php'));
    exit;
}

// Pengecekan Koneksi Database
if (!isset($pdo)) {
    $response['message'] = 'Koneksi database gagal. Operasi tidak dapat dilanjutkan.';
    error_log("CRITICAL: PDO object not available in process_menu_group.php.");
    if ($is_ajax) { 
        http_response_code(503); // Service Unavailable
        echo json_encode($response); 
        exit; 
    }
    set_flash_message('danger', $response['message']);
    header('Location: ' . admin_url('modules/menus/index.php')); // Redirect ke halaman aman
    exit;
}

// Validasi CSRF Token
$csrf_is_valid = false;
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['csrf_token']) && validate_csrf_token($_POST['csrf_token'])) {
        $csrf_is_valid = true;
    }
} elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'delete_menu_group') {
    // Hanya validasi CSRF untuk GET jika actionnya adalah delete
    if (isset($_GET['csrf_token']) && validate_csrf_token($_GET['csrf_token'])) {
        $csrf_is_valid = true;
    }
} else {
    // Untuk GET non-delete atau request method lain yang tidak dikenali,
    // kita tidak validasi CSRF di sini, tapi action switch akan menolak jika tidak ada action yang cocok.
    // Jika Anda ingin semua GET di-proteksi, tambahkan validasi di sini.
}

if (($_SERVER['REQUEST_METHOD'] === 'POST' || (isset($_GET['action']) && $_GET['action'] === 'delete_menu_group')) && !$csrf_is_valid) {
    $response['message'] = 'Kesalahan validasi keamanan (CSRF). Sesi Anda mungkin telah berakhir atau permintaan tidak sah. Silakan muat ulang halaman dan coba lagi.';
    if ($is_ajax) { 
        http_response_code(403); // Forbidden
        echo json_encode($response); 
        exit; 
    }
    set_flash_message('danger', $response['message']);
    $menu_id_for_csrf_redirect = isset($_REQUEST['menu_group_id']) ? (int)$_REQUEST['menu_group_id'] : (isset($_REQUEST['menu_id']) ? (int)$_REQUEST['menu_id'] : null);
    $redirect_url_after_csrf_fail = $menu_id_for_csrf_redirect ? admin_url('modules/menus/create_edit_menu.php?id=' . $menu_id_for_csrf_redirect) : admin_url('modules/menus/index.php');
    header('Location: ' . $redirect_url_after_csrf_fail);
    exit;
}


$action = $_REQUEST['action'] ?? null; // Ambil action dari POST atau GET

try {
    switch ($action) {
        case 'add_menu_group':
        case 'edit_menu_group':
            $group_id = ($action === 'edit_menu_group') ? (int)($_POST['menu_group_id'] ?? 0) : null;
            $name = trim($_POST['name'] ?? '');
            $location = trim($_POST['location'] ?? ''); 
            $description = trim($_POST['description'] ?? null); // Bisa null
            $errors_group_form = [];

            if (empty($name)) {
                $errors_group_form['name'] = 'Nama grup menu wajib diisi.';
            } elseif (strlen($name) > 100) {
                $errors_group_form['name'] = 'Nama grup menu maksimal 100 karakter.';
            }

            if (empty($location)) {
                $errors_group_form['location'] = 'Lokasi (key unik) menu wajib diisi.';
            } elseif (strlen($location) > 50) {
                $errors_group_form['location'] = 'Lokasi menu maksimal 50 karakter.';
            } elseif (!preg_match('/^[a-z0-9_]+$/', $location)) {
                $errors_group_form['location'] = 'Lokasi menu hanya boleh berisi huruf kecil, angka, dan underscore (_). Contoh: primary_nav.';
            } else {
                // Cek duplikasi lokasi (kecuali saat edit item yang sama)
                $sql_check_location = "SELECT id FROM portal_menus WHERE location = :location";
                $params_check_location = [':location' => $location];
                if ($action === 'edit_menu_group' && $group_id > 0) {
                    $sql_check_location .= " AND id != :id";
                    $params_check_location[':id'] = $group_id;
                }
                $stmt_check_location = $pdo->prepare($sql_check_location);
                $stmt_check_location->execute($params_check_location);
                if ($stmt_check_location->fetch()) {
                    $errors_group_form['location'] = 'Lokasi menu "' . esc_html($location) . '" sudah digunakan. Harap pilih key unik lainnya.';
                }
            }
            
            if ($description !== null && strlen($description) > 255) { // Hanya validasi jika tidak null
                $errors_group_form['description'] = 'Deskripsi maksimal 255 karakter.';
            }

            if (!empty($errors_group_form)) {
                $response['message'] = 'Validasi gagal. Harap periksa kembali input Anda.';
                $response['errors'] = $errors_group_form;
                if (!$is_ajax) {
                    $_SESSION['form_errors_menu_group'] = $errors_group_form;
                    $_SESSION['form_data_menu_group'] = $_POST; 
                }
                break; 
            }

            if ($action === 'add_menu_group') {
                $sql_op_group = "INSERT INTO portal_menus (name, location, description, created_at, updated_at) 
                                 VALUES (:name, :location, :description, NOW(), NOW())";
                $stmt_op_group = $pdo->prepare($sql_op_group);
            } else { // edit_menu_group
                if (!$group_id || $group_id <= 0) { 
                    $response['message'] = 'ID Grup Menu tidak valid untuk proses edit.'; 
                    if ($is_ajax) http_response_code(400); // Bad request
                    break; 
                }
                $sql_op_group = "UPDATE portal_menus SET name = :name, location = :location, description = :description, updated_at = NOW() 
                                 WHERE id = :id";
                $stmt_op_group = $pdo->prepare($sql_op_group);
                $stmt_op_group->bindParam(':id', $group_id, PDO::PARAM_INT);
            }
            $stmt_op_group->bindParam(':name', $name);
            $stmt_op_group->bindParam(':location', $location);
            $stmt_op_group->bindParam(':description', $description, $description === null ? PDO::PARAM_NULL : PDO::PARAM_STR);

            if ($stmt_op_group->execute()) {
                $response['success'] = true;
                $response['message'] = 'Grup menu berhasil ' . ($action === 'add_menu_group' ? 'ditambahkan.' : 'diperbarui.');
                $affected_group_id = ($action === 'add_menu_group') ? $pdo->lastInsertId() : $group_id;
                // Untuk AJAX, bisa kirim ID, untuk non-AJAX, redirect_url akan digunakan.
                $response['group_id'] = $affected_group_id; 
                // URL redirect jika sukses dan bukan AJAX
                $response['redirect_url'] = admin_url('modules/menus/manage_menu_items.php?menu_id=' . $affected_group_id); 
            } else {
                $errorInfoOperation = $stmt_op_group->errorInfo();
                $response['message'] = 'Gagal menyimpan grup menu ke database. Error: ' . ($errorInfoOperation[2] ?? 'Tidak diketahui');
                error_log("DB Error (portal_menus table, action: {$action}): " . print_r($errorInfoOperation, true));
                if ($is_ajax) http_response_code(500);
            }
            break;

        case 'delete_menu_group':
            // Untuk delete, bisa dari GET (jika link biasa) atau POST (jika form/AJAX)
            $group_id_to_delete = (int)($_REQUEST['menu_group_id'] ?? ($_REQUEST['id'] ?? 0)); 

            if ($group_id_to_delete <= 0) {
                $response['message'] = 'ID Grup Menu tidak valid atau tidak disediakan untuk dihapus.';
                if ($is_ajax) http_response_code(400);
                break;
            }

            // PENTING: Hapus semua item menu terkait SEBELUM menghapus grup menu
            // Atau pastikan ada ON DELETE CASCADE constraint dari portal_menu_items.menu_id ke portal_menus.id
            // Jika tidak ada ON DELETE CASCADE, lakukan penghapusan manual item dulu:
            $pdo->beginTransaction();
            try {
                $stmt_delete_related_items = $pdo->prepare("DELETE FROM portal_menu_items WHERE menu_id = :menu_id");
                $stmt_delete_related_items->execute([':menu_id' => $group_id_to_delete]);
                // Tidak masalah jika tidak ada item yang terhapus (rowCount bisa 0), lanjutkan hapus grup

                $stmt_delete_the_group = $pdo->prepare("DELETE FROM portal_menus WHERE id = :id");
                if ($stmt_delete_the_group->execute([':id' => $group_id_to_delete])) {
                    if ($stmt_delete_the_group->rowCount() > 0) {
                        $response['success'] = true;
                        $response['message'] = 'Grup menu dan semua item di dalamnya berhasil dihapus.';
                        $pdo->commit();
                    } else {
                        // Grupnya sendiri tidak ditemukan, mungkin sudah dihapus
                        $response['message'] = 'Grup menu tidak ditemukan atau sudah dihapus sebelumnya.';
                        $pdo->rollBack(); 
                    }
                } else {
                    $errorInfoDeleteGroup = $stmt_delete_the_group->errorInfo();
                    $response['message'] = 'Gagal menghapus grup menu. Error: ' . ($errorInfoDeleteGroup[2] ?? 'Tidak diketahui');
                    error_log("DB Error deleting menu group (portal_menus): " . print_r($errorInfoDeleteGroup, true));
                    $pdo->rollBack();
                    if ($is_ajax) http_response_code(500);
                }
            } catch (PDOException $e_delete_op) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                $response['message'] = 'Gagal menghapus grup menu karena kesalahan database: ' . $e_delete_op->getMessage();
                error_log("PDOException during delete_menu_group operation: " . $e_delete_op->getMessage());
                if ($is_ajax) http_response_code(500);
            }
            break;
        
        default:
            $response['message'] = 'Aksi untuk grup menu tidak valid atau tidak ditentukan: ' . esc_html((string)$action);
            if ($is_ajax) http_response_code(400); // Bad Request
            break;
    }
} catch (\PDOException $e_main_process) {
    error_log("PDOException in process_menu_group.php (Action: $action): " . $e_main_process->getMessage() . " at " . $e_main_process->getFile() . ":" . $e_main_process->getLine());
    $response['message'] = 'Terjadi kesalahan internal pada database. Silakan coba lagi atau hubungi administrator.';
    if ($is_ajax && !headers_sent()) http_response_code(500);
} catch (\Exception $e_general_process) {
    error_log("General Exception in process_menu_group.php (Action: $action): " . $e_general_process->getMessage() . " at " . $e_general_process->getFile() . ":" . $e_general_process->getLine());
    $response['message'] = 'Terjadi kesalahan umum pada sistem: ' . $e_general_process->getMessage();
    if ($is_ajax && !headers_sent()) http_response_code(500);
}

// Kirim respons
if ($is_ajax) {
    if (!headers_sent()) { // Pastikan header belum terkirim
         // Jika response success tapi http_response_code belum diset, set ke 200
        if ($response['success'] && http_response_code() === 200 && !isset(http_response_code(null)['code'])) {
             // Tidak perlu set eksplisit jika default sudah 200 OK
        } elseif (!$response['success'] && http_response_code() === 200 && !isset(http_response_code(null)['code'])) {
            // Jika gagal tapi belum ada kode error spesifik, set ke 400 atau 500
             http_response_code(isset($response['errors']) ? 422 : 500); // 422 jika error validasi
        }
    }
    echo json_encode($response);
} else {
    // Logika redirect untuk non-AJAX
    $redirect_url_non_ajax = admin_url('modules/menus/index.php'); // Default redirect

    if ($response['success']) {
        set_flash_message('success', $response['message']);
        if (!empty($response['redirect_url'])) { 
            $redirect_url_non_ajax = $response['redirect_url'];
        }
    } else { 
        set_flash_message('danger', $response['message']);
        // Jika validasi gagal, kembali ke form sebelumnya dengan data dan error
        if (($action === 'add_menu_group' || $action === 'edit_menu_group') && !empty($response['errors'])) {
            // $_SESSION['form_errors_menu_group'] dan $_SESSION['form_data_menu_group'] sudah di-set
            $group_id_for_redirect_fail = $_POST['menu_group_id'] ?? ($action === 'edit_menu_group' ? ($group_id ?? null) : null);
            if ($group_id_for_redirect_fail) {
                $redirect_url_non_ajax = admin_url('modules/menus/create_edit_menu.php?id=' . $group_id_for_redirect_fail);
            } else { // Gagal tambah, kembali ke form tambah kosong
                $redirect_url_non_ajax = admin_url('modules/menus/create_edit_menu.php');
            }
        }
    }
    if (!headers_sent()){
        header('Location: ' . $redirect_url_non_ajax);
    } else {
        error_log("Headers already sent in process_menu_group.php, cannot redirect to: " . $redirect_url_non_ajax);
        // Tampilkan link manual jika redirect gagal
        echo "<p>Operasi selesai. <a href='" . esc_html($redirect_url_non_ajax) . "'>Kembali ke Manajemen Menu</a></p>";
    }
}
exit;
?>
