<?php
// admin/modules/gallery_albums/process_album.php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__, 3));
}
// Pastikan config.php (yang memuat db_connect.php dan functions.php) di-require sekali saja
require_once PROJECT_ROOT . '/config.php'; 
// Jadi tidak perlu require_once db_connect.php dan admin/includes/functions.php lagi di sini
// jika config.php sudah melakukannya.

function is_ajax_request_gallery_album() { // Fungsi lokal untuk file ini
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
}
$is_ajax = is_ajax_request_gallery_album();
if ($is_ajax) {
    header('Content-Type: application/json');
}

$module_redirect_url_base_admin = admin_url('modules/gallery_albums/index.php'); // Gunakan admin_url
$manage_items_redirect_base_admin = admin_url('modules/gallery_albums/manage_items.php?album_id='); // Gunakan admin_url

if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['admin', 'editor'])) {
    if ($is_ajax) { http_response_code(403); echo json_encode(['success' => false, 'message' => 'Akses ditolak.']); exit; }
    set_flash_message('danger', 'Anda tidak memiliki izin untuk aksi ini.'); 
    header('Location: ' . admin_url('login.php')); // Redirect ke login admin
    exit;
}
if (!isset($pdo)) {
    if ($is_ajax) { http_response_code(503); echo json_encode(['success' => false, 'message' => 'Koneksi database gagal.']); exit; }
    set_flash_message('danger', 'Koneksi database gagal.'); 
    header('Location: ' . $module_redirect_url_base_admin); 
    exit;
}

// Fungsi validasi input untuk album
function validate_album_input_process($data, $is_edit = false, $album_id_to_edit = null, $pdo_conn = null) {
    $errors = [];
    if (empty($data['title'])) { $errors['title'] = 'Judul album wajib diisi.'; }
    elseif (strlen($data['title']) > 150) { $errors['title'] = 'Judul album maksimal 150 karakter.';}
    
    // Validasi slug (jika diisi, atau jika edit dan dikosongkan)
    if (!empty($data['slug'])) {
        if (!preg_match('/^[a-z0-9]+(?:-[a-z0-9]+)*$/', $data['slug'])) { 
            $errors['slug'] = 'Slug hanya boleh berisi huruf kecil, angka, dan tanda hubung (-).';
        } elseif (strlen($data['slug']) > 170) { 
            $errors['slug'] = 'Slug maksimal 170 karakter.';
        } else {
            // Cek duplikasi slug
            $sql_check_slug = "SELECT id FROM portal_gallery_albums WHERE slug = :slug";
            $params_check_slug = [':slug' => $data['slug']];
            if ($is_edit && $album_id_to_edit) { 
                $sql_check_slug .= " AND id != :id"; 
                $params_check_slug[':id'] = $album_id_to_edit;
            }
            $stmt_check_slug = $pdo_conn->prepare($sql_check_slug); 
            $stmt_check_slug->execute($params_check_slug);
            if ($stmt_check_slug->fetch()) { 
                $errors['slug'] = 'Slug "' . esc_html($data['slug']) . '" sudah digunakan. Harap gunakan slug lain.';
            }
        }
    } elseif ($is_edit && isset($data['slug_explicitly_empty']) && $data['slug_explicitly_empty']) { 
        // Jika saat edit, user sengaja mengosongkan slug (bukan tidak disubmit)
        $errors['slug'] = 'Slug wajib diisi saat mengedit album.';
    }
    // Untuk 'create', jika slug kosong, akan di-generate otomatis, jadi tidak ada error di sini.

    if (!empty($data['album_date'])) { 
        try { new DateTime($data['album_date']);} 
        catch (Exception $e) { $errors['album_date'] = 'Format tanggal album tidak valid.';}
    }
    if (!isset($data['is_published']) || !in_array($data['is_published'], [0, 1, '0', '1'], true)) { 
        $errors['is_published'] = 'Status publikasi tidak valid.';
    }
    return $errors;
}

// Fungsi untuk menangani upload gambar sampul
function handle_album_cover_image_process($file_key_name, $old_file_path, &$errors_ref) {
    // global $project_root; // $project_root sudah didefinisikan di scope global via config.php
    $uploaded_file_db_path = $old_file_path; // Default ke path lama jika tidak ada upload baru
    if (isset($_FILES[$file_key_name]) && $_FILES[$file_key_name]['error'] == UPLOAD_ERR_OK) {
        $upload_dir_relative = 'uploads/gallery_covers/'; // Path relatif dari PROJECT_ROOT
        $upload_dir_absolute = rtrim(PROJECT_ROOT, '/') . '/' . $upload_dir_relative;
        
        // Pastikan fungsi upload_file tersedia (dari admin/includes/functions.php atau config.php)
        $upload_result = upload_file($_FILES[$file_key_name], $upload_dir_absolute, ['image/jpeg', 'image/png', 'image/gif', 'image/webp'], 1 * 1024 * 1024); // Maks 1MB

        if ($upload_result['success']) {
            // Hapus gambar lama jika ada dan upload baru berhasil
            if (!empty($old_file_path) && file_exists(rtrim(PROJECT_ROOT, '/') . '/' . ltrim($old_file_path, '/'))) { 
                @unlink(rtrim(PROJECT_ROOT, '/') . '/' . ltrim($old_file_path, '/'));
            }
            $uploaded_file_db_path = $upload_result['filepath_relative']; // Simpan path relatif ke DB
        } else { 
            $errors_ref['cover_image_file'] = $upload_result['message']; 
        }
    } elseif (isset($_FILES[$file_key_name]) && $_FILES[$file_key_name]['error'] != UPLOAD_ERR_NO_FILE) {
        // Jika ada file tapi error selain "no file"
        $errors_ref['cover_image_file'] = "Error saat upload gambar sampul: Kode error PHP " . $_FILES[$file_key_name]['error'];
    }
    return $uploaded_file_db_path;
}


if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        if ($is_ajax) { http_response_code(403); echo json_encode(['success' => false, 'message' => 'Kesalahan CSRF token. Harap muat ulang halaman.']); exit; }
        set_flash_message('danger', 'Kesalahan CSRF token.');
        $action_redirect = $_POST['action'] ?? 'create'; $id_redirect = $_POST['id'] ?? null;
        header('Location: ' . admin_url('modules/gallery_albums/' . ($action_redirect === 'edit' && $id_redirect ? 'edit.php?id=' . $id_redirect : 'create.php'))); 
        exit;
    }

    $action = $_POST['action'] ?? null;
    $album_id = isset($_POST['id']) ? (int)$_POST['id'] : null;
    $current_user_id = $_SESSION['user_id']; // Ambil ID user yang sedang login

    $redirect_on_error_url = admin_url('modules/gallery_albums/' . ($action === 'edit' && $album_id ? 'edit.php?id=' . $album_id : 'create.php'));

    $form_data_input = [
        'title' => trim($_POST['title'] ?? ''),
        'slug' => trim($_POST['slug'] ?? ''),
        'description' => trim($_POST['description'] ?? ''),
        'album_date' => !empty($_POST['album_date']) ? $_POST['album_date'] : null,
        'is_published' => isset($_POST['is_published']) ? (int)$_POST['is_published'] : 0,
        'old_cover_image_url' => $_POST['old_cover_image_url'] ?? null,
        // Untuk membedakan slug kosong karena tidak diisi (generate) vs sengaja dikosongkan saat edit
        'slug_explicitly_empty' => (isset($_POST['slug']) && $_POST['slug'] === '') 
    ];
    $form_data_input['cover_image_url'] = $form_data_input['old_cover_image_url']; // Default ke gambar lama

    // Generate slug jika kosong saat create dan judul ada
    if ($action === 'create' && empty($form_data_input['slug']) && !empty($form_data_input['title'])) {
        $form_data_input['slug'] = generate_slug($form_data_input['title']);
        if(empty($form_data_input['slug'])) { // Fallback jika judul hanya simbol
            $form_data_input['slug'] = 'album-' . time(); 
        }
    }

    $errors = validate_album_input_process($form_data_input, ($action === 'edit'), $album_id, $pdo);
    // Handle upload gambar setelah validasi dasar, karena error upload bisa ditambahkan ke $errors
    $form_data_input['cover_image_url'] = handle_album_cover_image_process('cover_image_file', $form_data_input['old_cover_image_url'], $errors);

    if (!empty($errors)) {
        if ($is_ajax) { http_response_code(422); echo json_encode(['success' => false, 'message' => 'Validasi gagal. Harap periksa input Anda.', 'errors' => $errors]); }
        else {
            $_SESSION['form_errors'] = $errors; 
            $_SESSION['form_data'] = $_POST; // Kembalikan semua data POST agar form terisi kembali
            set_flash_message('danger', 'Gagal menyimpan album. Periksa error di bawah.'); 
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    }

    try {
        $sql_params = [
            ':title' => $form_data_input['title'], 
            ':slug' => $form_data_input['slug'],
            ':description' => $form_data_input['description'], 
            ':cover_image_url' => $form_data_input['cover_image_url'], // Ini sudah hasil dari handle_album_cover_image
            ':album_date' => $form_data_input['album_date'], 
            ':is_published' => $form_data_input['is_published'],
        ];

        if ($action === 'create') {
            $sql = "INSERT INTO portal_gallery_albums 
                        (title, slug, description, cover_image_url, album_date, is_published, created_by_user_id, created_at, updated_at)
                    VALUES 
                        (:title, :slug, :description, :cover_image_url, :album_date, :is_published, :created_by_user_id, NOW(), NOW())";
            $sql_params[':created_by_user_id'] = $current_user_id;
            $stmt = $pdo->prepare($sql);
        } elseif ($action === 'edit' && $album_id) {
            $sql = "UPDATE portal_gallery_albums SET 
                        title = :title, slug = :slug, description = :description, 
                        cover_image_url = :cover_image_url, album_date = :album_date, 
                        is_published = :is_published, updated_by_user_id = :updated_by_user_id, updated_at = NOW()
                    WHERE id = :id";
            $sql_params[':updated_by_user_id'] = $current_user_id;
            $sql_params[':id'] = $album_id;
            $stmt = $pdo->prepare($sql);
        } else { 
            throw new Exception("Aksi tidak valid atau ID Album tidak ada."); 
        }
        
        $stmt->execute($sql_params);
        $current_album_id = ($action === 'create') ? $pdo->lastInsertId() : $album_id;

        if ($is_ajax) {
            echo json_encode([
                'success' => true, 
                'message' => 'Album berhasil ' . ($action === 'create' ? 'dibuat' : 'diperbarui') . '.', 
                'redirect_url' => $manage_items_redirect_base_admin . $current_album_id // Redirect ke halaman kelola item
            ]);
        } else {
            set_flash_message('success', 'Album berhasil ' . ($action === 'create' ? 'dibuat' : 'diperbarui') . '. Anda akan diarahkan untuk mengelola item.');
            header('Location: ' . $manage_items_redirect_base_admin . $current_album_id);
        }
        exit;

    } catch (PDOException $e) { // Tangkap PDOException secara spesifik
        error_log("PDO Error processing album ($action): " . $e->getMessage() . " SQL Params: " . json_encode($sql_params));
        $error_message_user = 'Terjadi kesalahan database saat menyimpan album.';
        if(strpos($e->getMessage(), 'Duplicate entry') !== false && strpos($e->getMessage(), 'for key \'slug\'') !== false) {
            $error_message_user = 'Slug yang Anda masukkan sudah digunakan. Harap gunakan slug lain.';
            // Jika bisa, tambahkan ini ke $errors['slug'] untuk AJAX response
            $errors['slug'] = $error_message_user; 
             if ($is_ajax) { http_response_code(422); echo json_encode(['success' => false, 'message' => 'Validasi gagal.', 'errors' => $errors]); exit;}
        }

        if ($is_ajax) { http_response_code(500); echo json_encode(['success' => false, 'message' => $error_message_user]); }
        else {
            set_flash_message('danger', $error_message_user);
            $_SESSION['form_errors'] = $errors; // Kirim ulang errors jika ada
            $_SESSION['form_data'] = $_POST;
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    } catch (Exception $e) { // Tangkap Exception umum lainnya
        error_log("General Error processing album ($action): " . $e->getMessage());
        if ($is_ajax) { http_response_code(500); echo json_encode(['success' => false, 'message' => 'Terjadi kesalahan umum: ' . $e->getMessage()]); }
        else {
            set_flash_message('danger', 'Terjadi kesalahan umum: ' . $e->getMessage());
            $_SESSION['form_data'] = $_POST;
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    }

// Bagian DELETE dan GET lainnya tetap sama seperti kode Anda sebelumnya
} elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'delete') {
    // ... (Logika DELETE album Anda sudah baik, pastikan $project_root dari config.php benar) ...
    // Pastikan PROJECT_ROOT digunakan dengan benar untuk path unlink
    if (!isset($_GET['csrf_token']) || !validate_csrf_token($_GET['csrf_token'])) {
        if ($is_ajax) { http_response_code(403); echo json_encode(['success' => false, 'message' => 'Kesalahan CSRF.']); exit; }
        set_flash_message('danger', 'Kesalahan CSRF token.'); header('Location: ' . $module_redirect_url_base_admin); exit;
    }

    $album_id_to_delete = isset($_GET['id']) ? (int)$_GET['id'] : 0;
    if ($album_id_to_delete <= 0) {
        if ($is_ajax) { http_response_code(400); echo json_encode(['success' => false, 'message' => 'ID Album tidak valid.']); exit; }
        set_flash_message('danger', 'ID Album tidak valid.'); header('Location: ' . $module_redirect_url_base_admin); exit;
    }

    try {
        $pdo->beginTransaction();
        // Ambil info file sampul album
        $stmt_album_files = $pdo->prepare("SELECT cover_image_url FROM portal_gallery_albums WHERE id = :id");
        $stmt_album_files->execute([':id' => $album_id_to_delete]);
        $album_cover = $stmt_album_files->fetchColumn();

        // Ambil info file item-item dalam album
        $stmt_item_files = $pdo->prepare("SELECT item_url, item_type FROM portal_gallery_items WHERE album_id = :album_id");
        $stmt_item_files->execute([':album_id' => $album_id_to_delete]);
        $items_to_delete_files = $stmt_item_files->fetchAll(PDO::FETCH_ASSOC);

        // Hapus item dari DB
        $stmt_delete_items_db = $pdo->prepare("DELETE FROM portal_gallery_items WHERE album_id = :album_id");
        $stmt_delete_items_db->execute([':album_id' => $album_id_to_delete]);

        // Hapus album dari DB
        $stmt_delete_album_db = $pdo->prepare("DELETE FROM portal_gallery_albums WHERE id = :id");
        $stmt_delete_album_db->execute([':id' => $album_id_to_delete]);

        if ($stmt_delete_album_db->rowCount() > 0) {
            // Hapus file sampul dari server
            if (!empty($album_cover) && file_exists(rtrim(PROJECT_ROOT, '/') . '/' . ltrim($album_cover, '/'))) { 
                @unlink(rtrim(PROJECT_ROOT, '/') . '/' . ltrim($album_cover, '/')); 
            }
            // Hapus file item gambar dari server
            foreach ($items_to_delete_files as $item_file) {
                if ($item_file['item_type'] === 'image' && !empty($item_file['item_url']) && file_exists(rtrim(PROJECT_ROOT, '/') . '/' . ltrim($item_file['item_url'], '/'))) {
                    @unlink(rtrim(PROJECT_ROOT, '/') . '/' . ltrim($item_file['item_url'], '/'));
                }
            }
            $pdo->commit();
            if ($is_ajax) { echo json_encode(['success' => true, 'message' => 'Album dan semua item di dalamnya berhasil dihapus.']); }
            else { set_flash_message('success', 'Album dan semua item di dalamnya berhasil dihapus.'); header('Location: ' . $module_redirect_url_base_admin); }
        } else {
            $pdo->rollBack();
            if ($is_ajax) { echo json_encode(['success' => false, 'message' => 'Album tidak ditemukan atau sudah dihapus.']); }
            else { set_flash_message('warning', 'Album tidak ditemukan atau sudah dihapus.'); header('Location: ' . $module_redirect_url_base_admin); }
        }
    } catch (PDOException $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        error_log("Error deleting album (ID: {$album_id_to_delete}): " . $e->getMessage());
        if ($is_ajax) { http_response_code(500); echo json_encode(['success' => false, 'message' => 'Gagal menghapus album: Kesalahan database.']); }
        else { set_flash_message('danger', 'Gagal menghapus album karena kesalahan database.'); header('Location: ' . $module_redirect_url_base_admin); }
    }
    exit;
} else {
    // Jika bukan POST atau GET dengan action yang dikenal
    if ($is_ajax) { http_response_code(400); echo json_encode(['success' => false, 'message' => 'Permintaan tidak valid.']); }
    else { set_flash_message('danger', 'Permintaan tidak valid.'); header('Location: ' . $module_redirect_url_base_admin); }
    exit;
}
?>