<?php
/**
 * process_event.php
 * Menangani pembuatan (create), pembaruan (edit), dan penghapusan (delete) data acara.
 * Didesain untuk bekerja dengan permintaan AJAX dan form biasa.
 */

// 1. Muat Konfigurasi Inti (Wajib paling atas)
// Ini akan memuat session, koneksi DB ($pdo), dan fungsi-fungsi global.
require_once dirname(__DIR__, 3) . '/config.php';

// 2. Tentukan Jenis Permintaan (AJAX atau Biasa)
$is_ajax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
if ($is_ajax) {
    header('Content-Type: application/json');
}

// 3. Otentikasi & Otorisasi
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['admin', 'editor'])) {
    $errorMessage = 'Akses ditolak. Anda tidak memiliki izin untuk melakukan aksi ini.';
    if ($is_ajax) {
        http_response_code(403); // Forbidden
        echo json_encode(['success' => false, 'message' => $errorMessage]);
    } else {
        set_flash_message('danger', $errorMessage);
        header('Location: ' . admin_url('login.php'));
    }
    exit;
}

// 4. Proses Berdasarkan Metode HTTP
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    process_post_request($pdo, $is_ajax);
} else {
    // Di masa depan, Anda bisa menambahkan logika untuk GET (misal: delete) di sini.
    $errorMessage = 'Metode permintaan tidak valid.';
    if ($is_ajax) {
        http_response_code(405); // Method Not Allowed
        echo json_encode(['success' => false, 'message' => $errorMessage]);
    } else {
        set_flash_message('danger', $errorMessage);
        header('Location: index.php');
    }
    exit;
}

/**
 * Fungsi untuk memproses permintaan POST (Create & Edit)
 */
function process_post_request($pdo, $is_ajax) {
    // Validasi Token CSRF
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        $errorMessage = 'Sesi tidak valid atau telah kedaluwarsa. Silakan muat ulang halaman dan coba lagi.';
        if ($is_ajax) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => $errorMessage]);
        } else {
            set_flash_message('danger', $errorMessage);
            header('Location: ' . ($_POST['id'] ? 'edit.php?id=' . (int)$_POST['id'] : 'create.php'));
        }
        exit;
    }

    // Ambil dan sanitasi semua input dari form
    $input = [
        'id'               => isset($_POST['id']) ? (int)$_POST['id'] : null,
        'action'           => $_POST['action'] ?? 'create',
        'title'            => trim($_POST['title'] ?? ''),
        'slug'             => trim($_POST['slug'] ?? ''),
        'description_html' => $_POST['description_html'] ?? null,
        'start_datetime'   => trim($_POST['start_datetime'] ?? ''),
        'end_datetime'     => !empty($_POST['end_datetime']) ? trim($_POST['end_datetime']) : null,
        'location'         => trim($_POST['location'] ?? ''),
        'organizer'        => trim($_POST['organizer'] ?? ''),
        'contact_person'   => trim($_POST['contact_person'] ?? ''),
        'contact_email'    => trim($_POST['contact_email'] ?? ''),
        'contact_phone'    => trim($_POST['contact_phone'] ?? ''),
        'event_url'        => trim($_POST['event_url'] ?? ''),
        'is_published'     => isset($_POST['is_published']) ? (int)$_POST['is_published'] : 0,
        'old_image'        => $_POST['old_featured_image_url'] ?? null,
    ];

    // **LOGIKA KUNCI**: Jika slug kosong, buat otomatis dari judul.
    // Ini berlaku untuk 'create' dan 'edit', menyelesaikan masalah utama.
    if (empty($input['slug'])) {
        $input['slug'] = generate_slug($input['title']);
        if (empty($input['slug'])) { // Fallback jika judul juga kosong
            $input['slug'] = 'acara-' . time();
        }
    }

    $errors = [];

    // Validasi data
    validate_event_data($input, $errors, $pdo);

    // Proses upload gambar
    $input['featured_image_url'] = handle_image_upload($input['old_image'], $errors);

    // Jika ada error setelah validasi dan upload
    if (!empty($errors)) {
        if ($is_ajax) {
            http_response_code(422); // Unprocessable Content
            echo json_encode(['success' => false, 'message' => 'Terdapat kesalahan pada input Anda.', 'errors' => $errors]);
        } else {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $input;
            set_flash_message('danger', 'Gagal menyimpan. Periksa kembali semua kolom yang bertanda merah.');
            header('Location: ' . ($input['id'] ? 'edit.php?id=' . $input['id'] : 'create.php'));
        }
        exit;
    }

    // Jika tidak ada error, lanjutkan ke database
    save_event_to_db($input, $pdo, $is_ajax);
}

/**
 * Fungsi untuk validasi data acara
 */
function validate_event_data(&$input, &$errors, $pdo) {
    if (empty($input['title'])) {
        $errors['title'] = 'Judul acara wajib diisi.';
    }
    if (empty($input['start_datetime'])) {
        $errors['start_datetime'] = 'Tanggal & waktu mulai wajib diisi.';
    }

    // Validasi keunikan slug
    try {
        $sql = "SELECT id FROM portal_events WHERE slug = :slug";
        $params = [':slug' => $input['slug']];
        if ($input['action'] === 'edit' && $input['id']) {
            $sql .= " AND id != :id";
            $params[':id'] = $input['id'];
        }
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        if ($stmt->fetch()) {
            $errors['slug'] = 'Slug ini sudah digunakan. Harap gunakan slug lain.';
        }
    } catch (PDOException $e) {
        $errors['database'] = 'Gagal memvalidasi slug: ' . $e->getMessage();
    }
}

/**
 * Fungsi untuk menangani upload gambar
 */
function handle_image_upload($old_image_path, &$errors) {
    if (isset($_FILES['featured_image_file']) && $_FILES['featured_image_file']['error'] == UPLOAD_ERR_OK) {
        // Asumsikan Anda punya fungsi upload_file() yang andal di functions.php
        $upload_result = upload_file(
            $_FILES['featured_image_file'],
            PROJECT_ROOT . '/uploads/events_featured/',
            ['image/jpeg', 'image/png', 'image/gif', 'image/webp'],
            2 * 1024 * 1024 // 2MB max
        );

        if ($upload_result['success']) {
            // Hapus gambar lama jika ada dan upload baru berhasil
            if (!empty($old_image_path) && file_exists(PROJECT_ROOT . '/' . $old_image_path)) {
                @unlink(PROJECT_ROOT . '/' . $old_image_path);
            }
            return $upload_result['filepath_relative'];
        } else {
            $errors['featured_image_file'] = $upload_result['message'];
            return $old_image_path; // Gagal upload, kembalikan path gambar lama
        }
    }
    // Tidak ada file baru yang diupload, gunakan gambar lama
    return $old_image_path;
}

/**
 * Fungsi untuk menyimpan data ke database
 */
function save_event_to_db($input, $pdo, $is_ajax) {
    if ($input['action'] === 'create') {
        $sql = "INSERT INTO portal_events (title, slug, description_html, start_datetime, end_datetime, location, organizer, contact_person, contact_email, contact_phone, event_url, is_published, featured_image_url, created_at, updated_at) 
                VALUES (:title, :slug, :description_html, :start_datetime, :end_datetime, :location, :organizer, :contact_person, :contact_email, :contact_phone, :event_url, :is_published, :featured_image_url, NOW(), NOW())";
    } else {
        $sql = "UPDATE portal_events SET 
                    title = :title, slug = :slug, description_html = :description_html, 
                    start_datetime = :start_datetime, end_datetime = :end_datetime, location = :location, 
                    organizer = :organizer, contact_person = :contact_person, contact_email = :contact_email, 
                    contact_phone = :contact_phone, event_url = :event_url, is_published = :is_published, 
                    featured_image_url = :featured_image_url, updated_at = NOW() 
                WHERE id = :id";
    }

    // Hapus 'action' dan 'old_image' sebelum binding parameter
    $params_to_bind = $input;
    unset($params_to_bind['action'], $params_to_bind['old_image']);

    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params_to_bind);
        
        $message = 'Acara berhasil ' . ($input['action'] === 'create' ? 'dibuat' : 'diperbarui') . '.';
        if ($is_ajax) {
            echo json_encode(['success' => true, 'message' => $message, 'redirect_url' => 'index.php']);
        } else {
            set_flash_message('success', $message);
            header('Location: index.php');
        }
        exit;

    } catch (PDOException $e) {
        error_log("DATABASE ERROR on event save: " . $e->getMessage());
        $errorMessage = 'Terjadi kesalahan pada database saat menyimpan data.';
        if ($is_ajax) {
            http_response_code(500); // Internal Server Error
            echo json_encode(['success' => false, 'message' => $errorMessage]);
        } else {
            set_flash_message('danger', $errorMessage);
            header('Location: ' . ($input['id'] ? 'edit.php?id=' . $input['id'] : 'create.php'));
        }
        exit;
    }
}
?>