<?php
// admin/modules/articles/process_article.php
if (session_status() == PHP_SESSION_NONE) {
    // Session management will be handled by header.php or config.php
    // session_start(); // Start session if not already started by an included file
}

if (!defined('PROJECT_ROOT')) {
    define('PROJECT_ROOT', dirname(__DIR__, 3));
}
require_once PROJECT_ROOT . '/includes/db_connect.php'; // Should also load config.php if setup that way
require_once PROJECT_ROOT . '/admin/includes/functions.php'; // Contains set_flash_message, esc_html, validate_csrf_token, generate_slug, upload_file

// Fungsi untuk mendeteksi AJAX request
function is_ajax_request_article() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
}

$is_ajax = is_ajax_request_article();
if ($is_ajax) {
    header('Content-Type: application/json');
}

// Base URL for redirection within this module
$module_redirect_url_base = 'index.php'; // Or potentially resolve_url('admin/modules/articles/index.php') if using a URL helper

// Check user permission (admin or editor)
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['admin', 'editor'])) {
    if ($is_ajax) {
        http_response_code(403); // Forbidden
        echo json_encode(['success' => false, 'message' => 'Akses ditolak. Anda tidak memiliki izin.']);
        exit;
    }
    set_flash_message('danger', 'Anda tidak memiliki izin untuk melakukan aksi ini.');
    header('Location: ' . $module_redirect_url_base); // Adjust redirect if needed, e.g., to login
    exit;
}

// Check database connection
if (!isset($pdo)) {
    if ($is_ajax) {
        http_response_code(503); // Service Unavailable
        echo json_encode(['success' => false, 'message' => 'Koneksi database gagal.']);
        exit;
    }
    set_flash_message('danger', 'Koneksi database gagal.');
    header('Location: ' . $module_redirect_url_base);
    exit;
}

// --- Validation Function (Specific for Articles) ---
function validate_article_input($data, $is_edit = false, $article_id = null, $pdo_conn = null) {
    $errors = [];
    if (empty($data['title'])) {
        $errors['title'] = 'Judul artikel wajib diisi.';
    } elseif (strlen($data['title']) > 255) {
        $errors['title'] = 'Judul artikel maksimal 255 karakter.';
    }

    if (empty($data['content_html'])) {
        $errors['content_html'] = 'Konten artikel wajib diisi.';
    }
    
    if (empty($data['category_id']) || !is_numeric($data['category_id'])) {
        $errors['category_id'] = 'Kategori wajib dipilih.';
    } else { 
        // Validate category_id exists
        if ($pdo_conn) {
            $stmt_cat = $pdo_conn->prepare("SELECT id FROM portal_categories WHERE id = :cat_id");
            $stmt_cat->execute([':cat_id' => $data['category_id']]);
            if (!$stmt_cat->fetch()) {
                $errors['category_id'] = 'Kategori yang dipilih tidak valid.';
            }
        } else {
            // Cannot validate category if pdo_conn is not available
            // This scenario should ideally not happen if $pdo is checked globally
        }
    }

    // Slug validation
    if (!empty($data['slug'])) {
        if (!preg_match('/^[a-z0-9]+(?:-[a-z0-9]+)*$/', $data['slug'])) {
            $errors['slug'] = 'Slug hanya boleh berisi huruf kecil, angka, dan tanda hubung (-).';
        } elseif (strlen($data['slug']) > 270) { // Max length in DB schema for slug
            $errors['slug'] = 'Slug maksimal 270 karakter.';
        } else { 
            // Check for slug uniqueness
            if ($pdo_conn) {
                $sql_slug = "SELECT id FROM portal_articles WHERE slug = :slug" . ($is_edit && $article_id ? " AND id != :id" : "");
                $params_slug = [':slug' => $data['slug']];
                if ($is_edit && $article_id) {
                    $params_slug[':id'] = $article_id;
                }
                $stmt_slug = $pdo_conn->prepare($sql_slug);
                $stmt_slug->execute($params_slug);
                if ($stmt_slug->fetch()) {
                    $errors['slug'] = 'Slug "' . esc_html($data['slug']) . '" sudah digunakan. Harap gunakan slug lain.';
                }
            }
        }
    } elseif ($is_edit && empty($data['slug']) && !empty($data['title'])) { 
        // If slug is emptied on edit, it might be auto-generated later, or it could be an error if a title exists
        // For now, we allow it to be auto-generated. If it must be non-empty for edit, add error:
        // $errors['slug'] = 'Slug wajib diisi saat edit atau akan digenerate otomatis dari judul.';
    }


    // Status validation - MODIFIED to include 'pending_review'
    $allowed_statuses = ['draft', 'published', 'pending_review']; // Matched with DB enum and edit.php
    // Add other statuses from your ENUM ('archived', 'private') if they can be set through this form
    // For now, sticking to what was in edit.php options.
    if (empty($data['status']) || !in_array($data['status'], $allowed_statuses)) { //
        $errors['status'] = 'Status tidak valid. Pilih dari: ' . implode(', ', $allowed_statuses) . '.';
    }
    
    // Published_at validation (if provided)
    if (!empty($data['published_at'])) {
        try {
            new DateTime($data['published_at']);
        } catch (Exception $e) {
            $errors['published_at'] = 'Format tanggal publikasi tidak valid. Gunakan format YYYY-MM-DDTHH:MM.';
        }
    }
    return $errors;
}


// --- Main Processing Logic ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF Token Validation
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        if ($is_ajax) {
            http_response_code(403); // Forbidden
            echo json_encode(['success' => false, 'message' => 'Kesalahan CSRF token. Mohon muat ulang halaman dan coba lagi.']);
            exit;
        }
        set_flash_message('danger', 'Kesalahan CSRF token. Mohon muat ulang halaman dan coba lagi.');
        header('Location: ' . $module_redirect_url_base); // Or redirect to form page
        exit;
    }

    $action = $_POST['action'] ?? null;
    $article_id = isset($_POST['id']) ? filter_var($_POST['id'], FILTER_VALIDATE_INT) : null;
    
    // Determine redirect URL for errors based on action
    $redirect_on_error_url = ($action === 'edit' && $article_id ? 'edit.php?id=' . $article_id : 'create.php');

    // Gather form data
    $form_data_input = [
        'title' => trim($_POST['title'] ?? ''),
        'slug' => trim($_POST['slug'] ?? ''),
        'category_id' => isset($_POST['category_id']) ? filter_var($_POST['category_id'], FILTER_VALIDATE_INT) : null,
        'content_html' => $_POST['content_html'] ?? '', // Assuming TinyMCE content, potential for sanitization if needed
        'excerpt' => trim($_POST['excerpt'] ?? ''),
        'status' => $_POST['status'] ?? 'draft', // Default to 'draft'
        'published_at' => !empty($_POST['published_at']) ? $_POST['published_at'] : null,
        'meta_keywords' => trim($_POST['meta_keywords'] ?? ''),
        'meta_description' => trim($_POST['meta_description'] ?? ''),
        'allow_comments' => isset($_POST['allow_comments']) ? 1 : 0,
        'tags' => isset($_POST['tags']) && is_array($_POST['tags']) ? array_map('intval', $_POST['tags']) : [], // Ensure tags are integers
        'author_id' => $_SESSION['user_id'], // Set author from session
        'old_featured_image_url' => $_POST['old_featured_image_url'] ?? null,
    ];
    $form_data_input['featured_image_url'] = $form_data_input['old_featured_image_url']; // Initialize with old, might be overwritten by new upload

    // Auto-generate slug if empty and title is present
    if (($action === 'create' || $action === 'edit') && empty($form_data_input['slug']) && !empty($form_data_input['title'])) {
        $form_data_input['slug'] = generate_slug($form_data_input['title']); // Ensure generate_slug function exists and works well
    }
    // Fallback slug if still empty (e.g. title was also empty, though title is required by validation)
    if (empty($form_data_input['slug']) && !empty($form_data_input['title'])) { // This condition might be redundant if generate_slug always works
        $form_data_input['slug'] = 'artikel-' . time(); // Fallback unique slug
    }


    // Validate submitted data
    $errors = validate_article_input($form_data_input, ($action === 'edit'), $article_id, $pdo);

    // Featured Image Upload Handling
    if (isset($_FILES['featured_image_file']) && $_FILES['featured_image_file']['error'] == UPLOAD_ERR_OK) {
        $upload_dir_relative = 'uploads/articles_featured/'; // Relative to PROJECT_ROOT
        $upload_dir_absolute = PROJECT_ROOT . '/' . $upload_dir_relative;
        
        // Ensure upload directory exists and is writable
        if (!is_dir($upload_dir_absolute)) {
            if (!mkdir($upload_dir_absolute, 0755, true)) { // Create recursive
                $errors['featured_image_file'] = "Gagal membuat direktori upload.";
            }
        } elseif (!is_writable($upload_dir_absolute)) {
            $errors['featured_image_file'] = "Direktori upload tidak dapat ditulis.";
        }

        if (empty($errors['featured_image_file'])) { // Proceed if no directory errors
            $upload_result = upload_file(
                $_FILES['featured_image_file'],
                $upload_dir_absolute,
                ['image/jpeg', 'image/png', 'image/gif', 'image/webp'], // Allowed MIME types
                2 * 1024 * 1024 // Max file size: 2MB
            );

            if ($upload_result['success']) {
                // Delete old image if a new one is successfully uploaded and an old one exists
                if (!empty($form_data_input['old_featured_image_url']) && file_exists(PROJECT_ROOT . '/' . $form_data_input['old_featured_image_url'])) {
                    @unlink(PROJECT_ROOT . '/' . $form_data_input['old_featured_image_url']);
                }
                $form_data_input['featured_image_url'] = $upload_result['filepath_relative']; // Store relative path
            } else {
                $errors['featured_image_file'] = $upload_result['message'];
            }
        }
    } elseif (isset($_FILES['featured_image_file']) && $_FILES['featured_image_file']['error'] != UPLOAD_ERR_NO_FILE) {
        // Handle other upload errors
        $errors['featured_image_file'] = "Error saat upload gambar unggulan: Kode error " . $_FILES['featured_image_file']['error'];
    }


    // If validation errors exist
    if (!empty($errors)) {
        if ($is_ajax) {
            http_response_code(422); // Unprocessable Entity
            echo json_encode(['success' => false, 'message' => 'Validasi gagal. Harap periksa input Anda.', 'errors' => $errors]);
        } else {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $form_data_input; // Preserve submitted data
            set_flash_message('danger', 'Gagal menyimpan artikel. Harap periksa input Anda dan coba lagi.');
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    }

    // Prepare published_at for database (convert to Y-m-d H:i:s or set to NOW() if published)
    $published_at_db = null;
    if ($form_data_input['status'] === 'published') {
        $published_at_db = !empty($form_data_input['published_at']) ? (new DateTime($form_data_input['published_at']))->format('Y-m-d H:i:s') : date('Y-m-d H:i:s');
    } elseif (!empty($form_data_input['published_at'])) {
         // If status is draft/pending but a date is set, store it
         $published_at_db = (new DateTime($form_data_input['published_at']))->format('Y-m-d H:i:s');
    }


    // Database Transaction
    try {
        $pdo->beginTransaction();

        $sql_params_article = [
            ':title' => $form_data_input['title'],
            ':slug' => $form_data_input['slug'],
            ':category_id' => $form_data_input['category_id'],
            ':content_html' => $form_data_input['content_html'],
            ':excerpt' => $form_data_input['excerpt'],
            ':status' => $form_data_input['status'],
            ':published_at' => $published_at_db, // This can be null
            ':featured_image_url' => $form_data_input['featured_image_url'], // This can be null
            ':meta_keywords' => $form_data_input['meta_keywords'],
            ':meta_description' => $form_data_input['meta_description'],
            ':allow_comments' => $form_data_input['allow_comments'],
            ':author_id' => $form_data_input['author_id'],
            // ':updated_by_user_id' => $_SESSION['user_id'], // Assuming you have this column from SQL dump
        ];

        $current_article_id = null;

        if ($action === 'create') {
            // Add created_at, updated_at, and updated_by_user_id if your table uses it
            $sql_article = "INSERT INTO portal_articles (title, slug, category_id, content_html, excerpt, status, published_at, featured_image_url, meta_keywords, meta_description, allow_comments, author_id, created_at, updated_at)
                            VALUES (:title, :slug, :category_id, :content_html, :excerpt, :status, :published_at, :featured_image_url, :meta_keywords, :meta_description, :allow_comments, :author_id, NOW(), NOW())";
            // If you have 'updated_by_user_id', add it:
            // $sql_params_article[':updated_by_user_id'] = $_SESSION['user_id'];
            
            $stmt_article = $pdo->prepare($sql_article);
            $stmt_article->execute($sql_params_article);
            $current_article_id = $pdo->lastInsertId();
            $success_message_verb = 'dibuat';

        } elseif ($action === 'edit' && $article_id) {
            $sql_article = "UPDATE portal_articles SET 
                                title = :title, slug = :slug, category_id = :category_id, 
                                content_html = :content_html, excerpt = :excerpt, status = :status, 
                                published_at = :published_at, featured_image_url = :featured_image_url, 
                                meta_keywords = :meta_keywords, meta_description = :meta_description, 
                                allow_comments = :allow_comments, author_id = :author_id, 
                                updated_at = NOW() 
                                -- updated_by_user_id = :updated_by_user_id -- If applicable
                            WHERE id = :id";
            $sql_params_article[':id'] = $article_id;
            // If you have 'updated_by_user_id', ensure it's in $sql_params_article

            $stmt_article = $pdo->prepare($sql_article);
            $stmt_article->execute($sql_params_article);
            $current_article_id = $article_id;
            $success_message_verb = 'diperbarui';

            // Clear existing tags for this article before adding new ones
            $stmt_clear_tags = $pdo->prepare("DELETE FROM portal_article_tag WHERE article_id = :article_id");
            $stmt_clear_tags->execute([':article_id' => $current_article_id]);

        } else {
            throw new Exception("Aksi tidak valid atau ID Artikel tidak ada untuk operasi yang diminta.");
        }

        // Handle Tags
        if (!empty($form_data_input['tags']) && is_array($form_data_input['tags']) && $current_article_id) {
            $stmt_add_tag = $pdo->prepare("INSERT INTO portal_article_tag (article_id, tag_id) VALUES (:article_id, :tag_id)");
            foreach ($form_data_input['tags'] as $tag_id_val) {
                if (is_numeric($tag_id_val) && $tag_id_val > 0) { // Basic validation for tag ID
                    try {
                        $stmt_add_tag->execute([':article_id' => $current_article_id, ':tag_id' => $tag_id_val]);
                    } catch (PDOException $e) {
                        // Handle potential constraint violations (e.g., tag_id doesn't exist in portal_tags)
                        // Log error, or add to a non-critical error list
                        error_log("Error adding tag ID {$tag_id_val} for article ID {$current_article_id}: " . $e->getMessage());
                    }
                }
            }
        }

        $pdo->commit();

        // Prepare success response
        $response_data = [
            'success' => true,
            'message' => 'Artikel berhasil ' . $success_message_verb . '.',
            'redirect_url' => $module_redirect_url_base // Redirect to index after successful operation
        ];
        if ($action === 'edit' && $is_ajax) { // For edit AJAX, you might want to return the updated image URL
            $response_data['updated_data'] = ['featured_image_url' => $form_data_input['featured_image_url']];
        }


        if ($is_ajax) {
            echo json_encode($response_data);
        } else {
            set_flash_message('success', $response_data['message']);
            header('Location: ' . $response_data['redirect_url']);
        }
        exit;

    } catch (PDOException $e) { // Database specific errors
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("PDOException processing article ($action, ID: $article_id): " . $e->getMessage());
        if ($is_ajax) {
            http_response_code(500); // Internal Server Error
            echo json_encode(['success' => false, 'message' => 'Terjadi kesalahan database: ' . $e->getMessage()]);
        } else {
            set_flash_message('danger', 'Terjadi kesalahan database saat menyimpan artikel. Detail: ' . $e->getMessage());
            $_SESSION['form_data'] = $form_data_input; // Preserve data on error
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    } catch (Exception $e) { // General errors
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Exception processing article ($action, ID: $article_id): " . $e->getMessage());
        if ($is_ajax) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Terjadi kesalahan umum: ' . $e->getMessage()]);
        } else {
            set_flash_message('danger', 'Terjadi kesalahan umum: ' . $e->getMessage());
            $_SESSION['form_data'] = $form_data_input;
            header('Location: ' . $redirect_on_error_url);
        }
        exit;
    }

} elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'delete') {
    // --- Delete Action (via GET, ensure CSRF protection) ---
    if (!isset($_GET['csrf_token']) || !validate_csrf_token($_GET['csrf_token'])) {
        if ($is_ajax) {
            http_response_code(403); echo json_encode(['success' => false, 'message' => 'Kesalahan CSRF token untuk aksi delete.']); exit;
        }
        set_flash_message('danger', 'Kesalahan CSRF token.'); header('Location: ' . $module_redirect_url_base); exit;
    }

    $article_id_to_delete = isset($_GET['id']) ? filter_var($_GET['id'], FILTER_VALIDATE_INT) : 0;

    if ($article_id_to_delete <= 0) {
        if ($is_ajax) {
            http_response_code(400); // Bad Request
            echo json_encode(['success' => false, 'message' => 'ID Artikel tidak valid untuk dihapus.']); exit;
        }
        set_flash_message('danger', 'ID Artikel tidak valid untuk dihapus.'); header('Location: ' . $module_redirect_url_base); exit;
    }

    try {
        $pdo->beginTransaction();

        // Get featured image URL to delete the file
        $stmt_get_image = $pdo->prepare("SELECT featured_image_url FROM portal_articles WHERE id = :id");
        $stmt_get_image->execute([':id' => $article_id_to_delete]);
        $image_to_delete_path = $stmt_get_image->fetchColumn();

        // Delete associated tags first (due to potential foreign key constraints)
        $stmt_delete_tags = $pdo->prepare("DELETE FROM portal_article_tag WHERE article_id = :article_id");
        $stmt_delete_tags->execute([':article_id' => $article_id_to_delete]);

        // Delete the article
        $stmt_delete_article = $pdo->prepare("DELETE FROM portal_articles WHERE id = :id");
        $stmt_delete_article->execute([':id' => $article_id_to_delete]);

        if ($stmt_delete_article->rowCount() > 0) {
            // If article deleted, try to delete its featured image file
            if (!empty($image_to_delete_path) && file_exists(PROJECT_ROOT . '/' . $image_to_delete_path)) {
                @unlink(PROJECT_ROOT . '/' . $image_to_delete_path);
            }
            $pdo->commit();
            if ($is_ajax) {
                echo json_encode(['success' => true, 'message' => 'Artikel berhasil dihapus.']);
            } else {
                set_flash_message('success', 'Artikel berhasil dihapus.');
                header('Location: ' . $module_redirect_url_base);
            }
        } else {
            $pdo->rollBack(); // Nothing was deleted
            if ($is_ajax) {
                // May also use 404 Not Found if appropriate
                echo json_encode(['success' => false, 'message' => 'Artikel tidak ditemukan atau sudah dihapus sebelumnya.']);
            } else {
                set_flash_message('warning', 'Artikel tidak ditemukan atau sudah dihapus sebelumnya.');
                header('Location: ' . $module_redirect_url_base);
            }
        }
    } catch (PDOException $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Error deleting article (ID: $article_id_to_delete): " . $e->getMessage());
        if ($is_ajax) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Gagal menghapus artikel: Kesalahan database.']);
        } else {
            set_flash_message('danger', 'Gagal menghapus artikel karena kesalahan database.');
            header('Location: ' . $module_redirect_url_base);
        }
    }
    exit;

} else {
    // Invalid request method or action
    if ($is_ajax) {
        http_response_code(400); // Bad Request
        echo json_encode(['success' => false, 'message' => 'Permintaan tidak valid. Aksi atau metode tidak didukung.']);
    } else {
        set_flash_message('danger', 'Permintaan tidak valid.');
        header('Location: ' . $module_redirect_url_base);
    }
    exit;
}
?>