<?php
// admin/index.php (Dashboard)
if (session_status() == PHP_SESSION_NONE) {
    // session_start(); // Diharapkan sudah di-handle oleh config.php -> db_connect.php atau header.php
}

// PROJECT_ROOT diharapkan sudah didefinisikan oleh config.php (via db_connect.php)
// Jika tidak, fallback ini bisa berisiko jika struktur direktori tidak standar.
if (!defined('PROJECT_ROOT')) {
    // Dari admin/index.php, naik 2 level ke root proyek
    define('PROJECT_ROOT', dirname(__DIR__, 1));
    // Ini sebaiknya tidak terjadi jika db_connect.php berhasil dimuat
    error_log("Peringatan: PROJECT_ROOT tidak terdefinisi sebelum admin/index.php, fallback digunakan.");
}

// Path ke db_connect.php di root proyek
require_once PROJECT_ROOT . '/includes/db_connect.php';
// Path ke functions.php di dalam admin/includes/
require_once PROJECT_ROOT . '/admin/includes/functions.php'; // Memuat fungsi global admin termasuk admin_url() dan esc_html()

// Jika $pdo tidak ada setelah include db_connect.php, ada masalah kritis
if (!isset($pdo)) {
    error_log("CRITICAL: PDO object not available in admin/index.php. Halting.");
    // Tampilkan pesan error yang lebih user-friendly dan hentikan eksekusi
    // Ini bisa terjadi jika db_connect.php gagal total atau tidak benar-benar menginisialisasi $pdo global
    die("Koneksi ke database gagal atau konfigurasi sistem bermasalah. Silakan hubungi administrator. (Error Code: ADM_IDX_DBF)");
}


$page_title = "Dashboard Admin";
// $module_base_path digunakan oleh admin_url() untuk membangun path relatif dari root admin
// Karena kita di admin/index.php, base path ke direktori admin itu sendiri adalah './'
$module_base_path = './';

// Header HTML, session check, dan navigasi atas
// header.php akan menggunakan PROJECT_ROOT dan BASE_URL yang sudah didefinisikan
require_once PROJECT_ROOT . '/admin/includes/header.php';
require_once PROJECT_ROOT . '/admin/includes/sidebar.php'; // Sidebar navigasi

// Data awal untuk statistik (akan di-fetch oleh PHP saat load awal, bisa di-refresh AJAX)
$stats = [
    'total_articles' => 0, 'published_articles' => 0, 'draft_articles' => 0,
    'total_pages' => 0, 'total_categories' => 0, 'total_tags' => 0,
    'total_users' => 0, 'active_users' => 0, 'unread_messages' => 0,
    'upcoming_events' => 0,
];
$latest_articles_list = [];
$recent_messages_list = [];

// Hanya coba fetch data jika $pdo benar-benar ada
if (isset($pdo)) {
    try {
        $stats['total_articles'] = $pdo->query("SELECT COUNT(*) FROM portal_articles")->fetchColumn();
        $stats['published_articles'] = $pdo->query("SELECT COUNT(*) FROM portal_articles WHERE status = 'published'")->fetchColumn();
        $stats['draft_articles'] = $pdo->query("SELECT COUNT(*) FROM portal_articles WHERE status = 'draft'")->fetchColumn();
        
        // Cek apakah tabel portal_pages ada sebelum query
        $stmt_check_pages = $pdo->query("SHOW TABLES LIKE 'portal_pages'");
        if ($stmt_check_pages && $stmt_check_pages->rowCount() > 0) {
            $stats['total_pages'] = $pdo->query("SELECT COUNT(*) FROM portal_pages")->fetchColumn();
        } else {
            $stats['total_pages'] = 0;
            error_log("Tabel portal_pages tidak ditemukan saat mengambil statistik dashboard (admin/index.php).");
        }
        
        $stats['total_categories'] = $pdo->query("SELECT COUNT(*) FROM portal_categories")->fetchColumn();
        $stats['total_tags'] = $pdo->query("SELECT COUNT(*) FROM portal_tags")->fetchColumn();
        $stats['total_users'] = $pdo->query("SELECT COUNT(*) FROM portal_users")->fetchColumn();
        $stats['active_users'] = $pdo->query("SELECT COUNT(*) FROM portal_users WHERE is_active = 1")->fetchColumn();
        $stats['unread_messages'] = $pdo->query("SELECT COUNT(*) FROM portal_contact_messages WHERE is_read = 0 AND is_archived = 0")->fetchColumn();
        $stats['upcoming_events'] = $pdo->query("SELECT COUNT(*) FROM portal_events WHERE start_datetime >= CURDATE() AND is_published = 1")->fetchColumn();

        $stmt_latest_articles = $pdo->query("SELECT id, title, slug, created_at FROM portal_articles ORDER BY created_at DESC LIMIT 5");
        $latest_articles_list = $stmt_latest_articles->fetchAll();
        
        $stmt_recent_messages = $pdo->query("SELECT id, sender_name, subject, created_at FROM portal_contact_messages WHERE is_read = 0 AND is_archived = 0 ORDER BY created_at DESC LIMIT 5");
        $recent_messages_list = $stmt_recent_messages->fetchAll();

    } catch (\PDOException $e) {
        error_log("Dashboard stats query error in admin/index.php: " . $e->getMessage());
        // Set nilai default jika query gagal agar halaman tidak rusak total
        foreach ($stats as $key => $value) { $stats[$key] = 'Error'; }
        // Tidak set flash message di sini karena AJAX akan handle update
    }
} else {
    // Ini seharusnya tidak terjadi jika die() di atas dieksekusi
    error_log("Dashboard: PDO object not available for stats query in admin/index.php.");
    set_flash_message('danger', 'Koneksi database tidak tersedia untuk memuat statistik.');
}
?>

<div class="container-fluid px-4">
    <div class="d-flex justify-content-between align-items-center pt-3 mb-2 border-bottom">
        <?php echo generate_breadcrumbs(['Dashboard']); ?>
        <div class="btn-toolbar mb-2 mb-md-0">
            <!--<button class="btn btn-sm btn-outline-secondary" id="refreshDashboardButton" title="Refresh Data Dashboard">-->
            <!--    <i class="fas fa-sync-alt me-1"></i> Refresh Data-->
            <!--</button>-->
        </div>
    </div>
    
    <h1 class="h2 mt-0 mb-3">Selamat Datang, <?php echo isset($_SESSION['full_name']) && !empty($_SESSION['full_name']) ? esc_html($_SESSION['full_name']) : esc_html($_SESSION['username'] ?? 'Pengguna'); ?>!</h1>
    
    <?php echo display_flash_message(); ?>

    <p class="lead">Ini adalah halaman dashboard CMS Universitas Graha Karya Muara Bulian. Gunakan menu di samping untuk mengelola konten website.</p>
    <hr class="mb-4">

    <!-- Baris Statistik -->
    <div class="row" id="dashboardStatsRow">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-start border-primary border-4 shadow-sm h-100">
                <div class="card-body py-3">
                    <div class="row align-items-center">
                        <div class="col">
                            <div class="text-xs fw-bold text-primary text-uppercase mb-1">Total Artikel</div>
                            <div class="h5 mb-0 fw-bold text-gray-800 stat-value" id="totalArticlesStat"><?php echo $stats['total_articles']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-newspaper fa-2x text-gray-400"></i>
                        </div>
                    </div>
                </div>
                <a href="<?php echo admin_url('modules/articles/index.php', $module_base_path); ?>" class="stretched-link" aria-label="Kelola Artikel"></a>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-start border-success border-4 shadow-sm h-100">
                <div class="card-body py-3">
                    <div class="row align-items-center">
                        <div class="col">
                            <div class="text-xs fw-bold text-success text-uppercase mb-1">Halaman Statis</div>
                            <div class="h5 mb-0 fw-bold text-gray-800 stat-value" id="totalPagesStat"><?php echo $stats['total_pages']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-file-alt fa-2x text-gray-400"></i>
                        </div>
                    </div>
                     <a href="<?php echo admin_url('modules/pages/index.php', $module_base_path); ?>" class="stretched-link" aria-label="Kelola Halaman Statis"></a>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-start border-info border-4 shadow-sm h-100">
                <div class="card-body py-3">
                    <div class="row align-items-center">
                        <div class="col">
                            <div class="text-xs fw-bold text-info text-uppercase mb-1">Pesan Belum Dibaca</div>
                            <div class="h5 mb-0 fw-bold text-gray-800 stat-value" id="unreadMessagesStat"><?php echo $stats['unread_messages']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-envelope-open-text fa-2x text-gray-400"></i>
                        </div>
                    </div>
                    <a href="<?php echo admin_url('modules/contact_messages/index.php', $module_base_path); ?>" class="stretched-link" aria-label="Kelola Pesan Kontak"></a>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-start border-warning border-4 shadow-sm h-100">
                <div class="card-body py-3">
                    <div class="row align-items-center">
                        <div class="col">
                            <div class="text-xs fw-bold text-warning text-uppercase mb-1">Pengguna Aktif</div>
                            <div class="h5 mb-0 fw-bold text-gray-800 stat-value" id="activeUsersStat"><?php echo $stats['active_users']; ?> / <?php echo $stats['total_users']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-users fa-2x text-gray-400"></i>
                        </div>
                    </div>
                     <a href="<?php echo admin_url('modules/users/index.php', $module_base_path); ?>" class="stretched-link" aria-label="Kelola Pengguna"></a>
                </div>
            </div>
        </div>
    </div>

    <!-- Baris Aksi Cepat dan Daftar Terbaru -->
    <div class="row">
        <div class="col-lg-7 mb-4">
            <div class="card shadow-sm mb-4">
                <div class="card-header">
                    <h6 class="m-0 fw-bold text-primary"><i class="fas fa-bolt me-1"></i>Aksi Cepat</h6>
                </div>
                <div class="card-body">
                    <div class="row row-cols-2 row-cols-sm-2 row-cols-md-3 g-3">
                        <div class="col">
                            <a href="<?php echo admin_url('modules/articles/create.php', $module_base_path); ?>" class="btn btn-outline-primary w-100 py-3 d-flex flex-column align-items-center justify-content-center h-100 quick-action-btn">
                                <i class="fas fa-plus-circle fa-2x mb-2"></i> <span class="text-center small">Tambah Artikel</span>
                            </a>
                        </div>
                        <div class="col">
                            <a href="<?php echo admin_url('modules/pages/create.php', $module_base_path); ?>" class="btn btn-outline-info w-100 py-3 d-flex flex-column align-items-center justify-content-center h-100 quick-action-btn">
                                <i class="fas fa-file-medical fa-2x mb-2"></i> <span class="text-center small">Tambah Halaman</span>
                            </a>
                        </div>
                        <div class="col">
                             <a href="<?php echo admin_url('modules/events/create.php', $module_base_path); ?>" class="btn btn-outline-success w-100 py-3 d-flex flex-column align-items-center justify-content-center h-100 quick-action-btn">
                                <i class="fas fa-calendar-plus fa-2x mb-2"></i> <span class="text-center small">Tambah Acara</span>
                            </a>
                        </div>
                        <div class="col">
                            <a href="<?php echo admin_url('modules/gallery_albums/create.php', $module_base_path); ?>" class="btn btn-outline-warning w-100 py-3 d-flex flex-column align-items-center justify-content-center h-100 quick-action-btn">
                                <i class="fas fa-images fa-2x mb-2"></i> <span class="text-center small">Tambah Album</span>
                            </a>
                        </div>
                        <?php if (isset($_SESSION['role']) && $_SESSION['role'] === 'admin'): ?>
                        <div class="col">
                             <a href="<?php echo admin_url('modules/users/index.php', $module_base_path); ?>" class="btn btn-outline-secondary w-100 py-3 d-flex flex-column align-items-center justify-content-center h-100 quick-action-btn">
                                <i class="fas fa-users-cog fa-2x mb-2"></i> <span class="text-center small">Kelola Pengguna</span>
                            </a>
                        </div>
                         <div class="col">
                             <a href="<?php echo admin_url('modules/settings/index.php', $module_base_path); ?>" class="btn btn-outline-dark w-100 py-3 d-flex flex-column align-items-center justify-content-center h-100 quick-action-btn">
                                <i class="fas fa-cogs fa-2x mb-2"></i> <span class="text-center small">Pengaturan</span>
                            </a>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-lg-5 mb-4">
            <div class="card shadow-sm mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h6 class="m-0 fw-bold text-primary"><i class="fas fa-history me-1"></i>Artikel Terbaru</h6>
                    <a href="<?php echo admin_url('modules/articles/index.php', $module_base_path); ?>" class="btn btn-sm btn-outline-secondary py-1 px-2">Lihat Semua</a>
                </div>
                <div class="list-group list-group-flush" id="latestArticlesList">
                    <?php if (!empty($latest_articles_list)): ?>
                        <?php foreach ($latest_articles_list as $article): ?>
                            <a href="<?php echo admin_url('modules/articles/edit.php?id=' . $article['id'], $module_base_path); ?>" class="list-group-item list-group-item-action py-2">
                                <div class="d-flex w-100 justify-content-between">
                                    <h6 class="mb-1 text-truncate" style="max-width: 80%;"><?php echo esc_html($article['title']); ?></h6>
                                    <small class="text-muted"><?php echo date('d M', strtotime($article['created_at'])); ?></small>
                                </div>
                                <small class="text-muted">Klik untuk edit.</small>
                            </a>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="list-group-item text-muted fst-italic">Belum ada artikel.</div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="card shadow-sm" id="recentMessagesCard" style="<?php echo empty($recent_messages_list) && $stats['unread_messages'] == 0 ? 'display:none;' : ''; ?>">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h6 class="m-0 fw-bold text-primary"><i class="fas fa-inbox me-1"></i>Pesan Kontak Terbaru (Belum Dibaca)</h6>
                     <a href="<?php echo admin_url('modules/contact_messages/index.php', $module_base_path); ?>" class="btn btn-sm btn-outline-secondary py-1 px-2">Lihat Semua</a>
                </div>
                 <div class="list-group list-group-flush" id="recentMessagesList">
                    <?php if (!empty($recent_messages_list)): ?>
                        <?php foreach ($recent_messages_list as $msg): ?>
                        <a href="<?php echo admin_url('modules/contact_messages/view.php?id=' . $msg['id'], $module_base_path); ?>" class="list-group-item list-group-item-action py-2">
                             <div class="d-flex w-100 justify-content-between">
                                <strong class="mb-1 text-truncate" style="max-width: 70%;"><?php echo esc_html($msg['subject']); ?></strong>
                                <small class="text-muted"><?php echo date('d M, H:i', strtotime($msg['created_at'])); ?></small>
                            </div>
                            <small class="text-muted">Dari: <?php echo esc_html($msg['sender_name']); ?></small>
                        </a>
                        <?php endforeach; ?>
                    <?php elseif($stats['unread_messages'] > 0): // Jika ada pesan tapi tidak di $recent_messages_list (misal lebih dari 5) ?>
                        <div class="list-group-item text-muted fst-italic">Lihat semua untuk pesan yang lebih lama.</div>
                    <?php else: ?>
                         <div class="list-group-item text-muted fst-italic">Tidak ada pesan baru yang belum dibaca.</div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
$page_scripts = []; // Pastikan array ini diinisialisasi
// Script JS yang sudah ada sebelumnya untuk escapeHtmlJs, formatDate*, loadDashboardData
// Sebaiknya fungsi-fungsi helper JS (escapeHtmlJs, formatDate*, showAdminNotification) ada di admin-global.js
$page_scripts[] = "<script>
// Pastikan showAdminNotification tersedia global (dari admin-global.js)
// function escapeHtmlJs(unsafe) { /* ... */ }
// function formatDateForDashboard(dateString) { /* ... */ }
// function formatDateTimeForDashboard(dateString) { /* ... */ }

function loadDashboardData() {
    const refreshButton = $('#refreshDashboardButton');
    const originalButtonHtml = refreshButton.html(); // Simpan HTML tombol asli
    refreshButton.prop('disabled', true).html('<span class=\"spinner-border spinner-border-sm me-1\" role=\"status\" aria-hidden=\"true\"></span>Memuat...');

    $('.stat-value').html('<span class=\"spinner-border spinner-border-xs text-muted\" style=\"width: 0.8rem; height: 0.8rem; border-width: .15em;\"></span>');

    $.ajax({
        url: '" . admin_url("api/dashboard_data.php", $module_base_path) . "',
        method: 'GET',
        dataType: 'json',
        data: { csrf_token: '" . generate_csrf_token() . "' },
        success: function(response) {
            if (response.success && response.data) {
                const stats = response.data.stats;
                $('#totalArticlesStat').text(stats.total_articles || 0);
                $('#totalPagesStat').text(stats.total_pages || 0);
                $('#unreadMessagesStat').text(stats.unread_messages || 0);
                $('#activeUsersStat').text((stats.active_users || 0) + ' / ' + (stats.total_users || 0));

                const latestArticlesContainer = $('#latestArticlesList');
                latestArticlesContainer.empty();
                if (response.data.latest_articles && response.data.latest_articles.length > 0) {
                    $.each(response.data.latest_articles, function(i, article) {
                        latestArticlesContainer.append(
                            `<a href=\"\${escapeHtmlJs(article.edit_url)}\" class=\"list-group-item list-group-item-action py-2\">
                                <div class=\"d-flex w-100 justify-content-between\">
                                    <h6 class=\"mb-1 text-truncate\" style=\"max-width: 80%;\">\${escapeHtmlJs(article.title)}</h6>
                                    <small class=\"text-muted\">\${formatDateForDashboard(article.created_at)}</small>
                                </div>
                                <small class=\"text-muted\">Klik untuk edit.</small>
                            </a>`
                        );
                    });
                } else {
                    latestArticlesContainer.html('<div class=\"list-group-item text-muted fst-italic\">Belum ada artikel.</div>');
                }

                const recentMessagesContainer = $('#recentMessagesList');
                const recentMessagesCard = $('#recentMessagesCard');
                recentMessagesContainer.empty();
                if (response.data.recent_messages && response.data.recent_messages.length > 0) {
                    recentMessagesCard.show();
                    $.each(response.data.recent_messages, function(i, msg) {
                        recentMessagesContainer.append(
                            `<a href=\"\${escapeHtmlJs(msg.view_url)}\" class=\"list-group-item list-group-item-action py-2\">
                                <div class=\"d-flex w-100 justify-content-between\">
                                    <strong class=\"mb-1 text-truncate\" style=\"max-width: 70%;\">\${escapeHtmlJs(msg.subject)}</strong>
                                    <small class=\"text-muted\">\${formatDateTimeForDashboard(msg.created_at)}</small>
                                </div>
                                <small class=\"text-muted\">Dari: \${escapeHtmlJs(msg.sender_name)}</small>
                            </a>`
                        );
                    });
                } else {
                    recentMessagesContainer.html('<div class=\"list-group-item text-muted fst-italic\">Tidak ada pesan baru yang belum dibaca.</div>');
                    if (stats.unread_messages == 0) { // Sembunyikan card jika memang tidak ada pesan belum dibaca sama sekali
                        recentMessagesCard.hide();
                    } else { // Jika ada pesan tapi > 5 (tidak muncul di list pendek), tetap tampilkan card
                        recentMessagesCard.show();
                    }
                }
                if (typeof showAdminNotification === 'function') {
                    showAdminNotification('success', 'Data dashboard berhasil diperbarui.');
                }
            } else {
                 if (typeof showAdminNotification === 'function') {
                    showAdminNotification('warning', response.message || 'Gagal memuat data dashboard dari server.');
                }
                $('#totalArticlesStat').text('N/A'); $('#totalPagesStat').text('N/A');
                $('#unreadMessagesStat').text('N/A'); $('#activeUsersStat').text('N/A / N/A');
            }
        },
        error: function(jqXHR) {
            let errorMsg = 'Kesalahan server saat memuat data dashboard.';
            if(jqXHR.responseJSON && jqXHR.responseJSON.message) errorMsg = jqXHR.responseJSON.message;
            if (typeof showAdminNotification === 'function') showAdminNotification('danger', errorMsg + ' Cek konsol browser.');
            else alert(errorMsg);
            console.error('AJAX Dashboard Error:', jqXHR.status, jqXHR.responseText);
            $('#totalArticlesStat').text('Err'); $('#totalPagesStat').text('Err');
            $('#unreadMessagesStat').text('Err'); $('#activeUsersStat').text('Err / Err');
        },
        complete: function() {
            refreshButton.prop('disabled', false).html(originalButtonHtml);
        }
    });
}

$(document).ready(function() {
    $('#refreshDashboardButton').on('click', loadDashboardData);
    // Anda bisa memanggil loadDashboardData() di sini jika ingin data dimuat oleh AJAX saat halaman pertama kali dibuka,
    // daripada oleh PHP. Namun, memuat oleh PHP untuk tampilan awal lebih cepat untuk user.
    // loadDashboardData(); 
});
</script>";
require_once PROJECT_ROOT . '/admin/includes/footer.php';
?>